/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.util.concurrency;

import com.google.common.base.Preconditions;
import java.io.Closeable;
import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Random;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import org.apache.james.util.concurrent.NamedThreadFactory;
import org.reactivestreams.Publisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Mono;

public class ConcurrentTestRunner
implements Closeable {
    public static final int DEFAULT_OPERATION_COUNT = 1;
    private static final Logger LOGGER = LoggerFactory.getLogger(ConcurrentTestRunner.class);
    private final int threadCount;
    private final int operationCount;
    private final boolean suppressLogger;
    private final CountDownLatch countDownLatch;
    private final ConcurrentOperation biConsumer;
    private final ExecutorService executorService;
    private final List<Future<?>> futures;

    public static RequireOperation builder() {
        return operation -> threadCount -> new Builder(threadCount, operation);
    }

    private ConcurrentTestRunner(int threadCount, int operationCount, boolean suppressLogger, ConcurrentOperation biConsumer) {
        this.threadCount = threadCount;
        this.operationCount = operationCount;
        this.countDownLatch = new CountDownLatch(threadCount);
        this.suppressLogger = suppressLogger;
        this.biConsumer = biConsumer;
        NamedThreadFactory threadFactory = NamedThreadFactory.withClassName(this.getClass());
        this.executorService = Executors.newFixedThreadPool(threadCount, threadFactory);
        this.futures = new ArrayList();
    }

    public ConcurrentTestRunner run() {
        for (int i = 0; i < this.threadCount; ++i) {
            this.futures.add(this.executorService.submit(new ConcurrentRunnableTask(i, this.biConsumer, this.suppressLogger)));
        }
        return this;
    }

    public ConcurrentTestRunner assertNoException() throws ExecutionException, InterruptedException {
        for (Future<?> future : this.futures) {
            future.get();
        }
        return this;
    }

    public ConcurrentTestRunner awaitTermination(Duration duration) throws InterruptedException {
        this.executorService.shutdown();
        boolean terminated = this.executorService.awaitTermination(duration.toMillis(), TimeUnit.MILLISECONDS);
        if (!terminated) {
            throw new NotTerminatedException();
        }
        return this;
    }

    public ConcurrentTestRunner runSuccessfullyWithin(Duration duration) throws InterruptedException, ExecutionException {
        return this.run().awaitTermination(duration).assertNoException();
    }

    public ConcurrentTestRunner runAcceptingErrorsWithin(Duration duration) throws InterruptedException, ExecutionException {
        return this.run().awaitTermination(duration);
    }

    @Override
    public void close() throws IOException {
        this.executorService.shutdownNow();
    }

    @FunctionalInterface
    public static interface RequireOperation {
        public RequireThreadCount operation(ConcurrentOperation var1);

        default public RequireThreadCount reactorOperation(ReactorOperation reactorOperation) {
            return this.operation(reactorOperation.blocking());
        }

        default public RequireThreadCount randomlyDistributedOperations(ConcurrentOperation firstOperation, ConcurrentOperation ... operations) {
            Random random = this.createReproductibleRandom();
            ConcurrentOperation aggregateOperation = (threadNumber, step) -> this.selectRandomOperation(random, firstOperation, operations).execute(threadNumber, step);
            return this.operation(aggregateOperation);
        }

        default public RequireThreadCount randomlyDistributedReactorOperations(ReactorOperation firstReactorOperation, ReactorOperation ... reactorOperations) {
            Random random = this.createReproductibleRandom();
            ReactorOperation aggregateOperation = (threadNumber, step) -> this.selectRandomOperation(random, firstReactorOperation, reactorOperations).execute(threadNumber, step);
            return this.reactorOperation(aggregateOperation);
        }

        default public Random createReproductibleRandom() {
            return new Random(2134L);
        }

        default public <OperationT> OperationT selectRandomOperation(Random random, OperationT firstReactorOperation, OperationT ... reactorOperations) {
            int whichAction = random.nextInt(reactorOperations.length + 1);
            if (whichAction == 0) {
                return firstReactorOperation;
            }
            return reactorOperations[whichAction - 1];
        }
    }

    @FunctionalInterface
    public static interface ConcurrentOperation {
        public void execute(int var1, int var2) throws Exception;
    }

    private class ConcurrentRunnableTask
    implements Runnable {
        private final int threadNumber;
        private final ConcurrentOperation concurrentOperation;
        private final boolean noErrorLogs;
        private Exception exception;

        public ConcurrentRunnableTask(int threadNumber, ConcurrentOperation concurrentOperation, boolean noErrorLogs) {
            this.threadNumber = threadNumber;
            this.concurrentOperation = concurrentOperation;
            this.noErrorLogs = noErrorLogs;
        }

        @Override
        public void run() {
            this.exception = null;
            ConcurrentTestRunner.this.countDownLatch.countDown();
            for (int i = 0; i < ConcurrentTestRunner.this.operationCount; ++i) {
                try {
                    this.concurrentOperation.execute(this.threadNumber, i);
                    continue;
                }
                catch (Exception e) {
                    if (!this.noErrorLogs) {
                        LOGGER.error("Error caught during concurrent testing (iteration {}, threadNumber {})", i, this.threadNumber, e);
                    }
                    this.exception = e;
                }
            }
            if (this.exception != null) {
                throw new RuntimeException(this.exception);
            }
        }
    }

    public static class NotTerminatedException
    extends RuntimeException {
    }

    @FunctionalInterface
    public static interface RequireThreadCount {
        public Builder threadCount(int var1);
    }

    public static class Builder {
        private final int threadCount;
        private final ConcurrentOperation operation;
        private Optional<Integer> operationCount;
        private Optional<Boolean> noErrorLogs;

        private Builder(int threadCount, ConcurrentOperation operation) {
            Preconditions.checkArgument(threadCount > 0, "Thread count should be strictly positive");
            Preconditions.checkNotNull(operation);
            this.threadCount = threadCount;
            this.operation = operation;
            this.operationCount = Optional.empty();
            this.noErrorLogs = Optional.empty();
        }

        public Builder operationCount(int operationCount) {
            Preconditions.checkArgument(operationCount > 0, "Operation count should be strictly positive");
            this.operationCount = Optional.of(operationCount);
            return this;
        }

        public Builder noErrorLogs() {
            this.noErrorLogs = Optional.of(true);
            return this;
        }

        private ConcurrentTestRunner build() {
            return new ConcurrentTestRunner(this.threadCount, this.operationCount.orElse(1), this.noErrorLogs.orElse(false), this.operation);
        }

        public ConcurrentTestRunner run() {
            ConcurrentTestRunner testRunner = this.build();
            testRunner.run();
            return testRunner;
        }

        public ConcurrentTestRunner runSuccessfullyWithin(Duration duration) throws InterruptedException, ExecutionException {
            return this.build().runSuccessfullyWithin(duration);
        }

        public ConcurrentTestRunner runAcceptingErrorsWithin(Duration duration) throws InterruptedException, ExecutionException {
            return this.build().runAcceptingErrorsWithin(duration);
        }
    }

    @FunctionalInterface
    public static interface ReactorOperation {
        public Publisher<Void> execute(int var1, int var2) throws Exception;

        default public ConcurrentOperation blocking() {
            return (threadNumber, step) -> Mono.from(this.execute(threadNumber, step)).then().block();
        }
    }
}

