/*******************************************************************************
 * Copyright (c) 2006, 2007 Ecliptical Software Inc. and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Ecliptical Software Inc. - initial API and implementation
 *******************************************************************************/
package org.eclipse.emf.mint.internal.ui.source;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.ListenerList;
import org.eclipse.core.runtime.SafeRunner;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.mint.CodeGenStatus;
import org.eclipse.emf.mint.IMemberAnnotationListener;
import org.eclipse.emf.mint.IMemberAnnotationManager;
import org.eclipse.emf.mint.MemberAnnotationChangedEvent;
import org.eclipse.emf.mint.MintCore;
import org.eclipse.emf.mint.internal.ui.MintUI;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IMember;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jface.resource.ColorRegistry;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.jface.util.SafeRunnable;
import org.eclipse.jface.viewers.IDecoration;
import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.jface.viewers.ILightweightLabelDecorator;
import org.eclipse.jface.viewers.LabelProviderChangedEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.ui.progress.WorkbenchJob;
import org.eclipse.ui.themes.IThemeManager;


public class GeneratedElementDecorator implements ILightweightLabelDecorator,
		IMemberAnnotationListener, IPropertyChangeListener {

	private final ListenerList listeners = new ListenerList();

	private IMemberAnnotationManager manager;

	private IThemeManager themeManager;

	private Color generatedColor;

	private Color generatedNotColor;

	private ColorJob colorJob;

	private final Object colorLock = new Object();

	public GeneratedElementDecorator() {
		manager = MintCore.getInstance().getMemberAnnotationManager();
		manager.addMemberAnnotationListener(this);
		themeManager = MintUI.getDefault().getWorkbench().getThemeManager();
		themeManager.addPropertyChangeListener(this);
	}

	public void decorate(Object element, IDecoration decoration) {
		try {
			if (element instanceof ICompilationUnit) {
				ICompilationUnit cu = (ICompilationUnit) element;
				IType[] types = cu.getTypes();
				CodeGenStatus status;
				for (int i = 0; i < types.length; ++i) {
					if ((status = manager.getCodeGenStatus(types[i])) != CodeGenStatus.NONE)
						decorate(status, decoration);
				}

				return;
			}

			if (!(element instanceof IMember))
				return;

			CodeGenStatus status = manager.getCodeGenStatus((IMember) element);
			decorate(status, decoration);
		} catch (JavaModelException e) {
			MintCore.getInstance().logError(
					Messages.GeneratedElementDecorator_ErrorDecorate, e);
		}
	}

	private void decorate(CodeGenStatus status, IDecoration decoration) {
		Color color = null;
		switch (status) {
		case GENERATED:
			color = getGeneratedColor();
			break;
		case GENERATED_NOT:
			color = getGeneratedNotColor();
			break;
		}

		if (color != null)
			decoration.setForegroundColor(color);
	}

	public void addListener(ILabelProviderListener listener) {
		listeners.add(listener);
	}

	public void dispose() {
		themeManager.removePropertyChangeListener(this);
		themeManager = null;
		manager.removeMemberAnnotationListener(this);
		manager = null;
		listeners.clear();
	}

	public boolean isLabelProperty(Object element, String property) {
		return false;
	}

	public void removeListener(ILabelProviderListener listener) {
		listeners.remove(listener);
	}

	protected void fireLabelProviderChangedEvent(
			final LabelProviderChangedEvent event) {
		Object[] l = listeners.getListeners();
		for (int i = 0; i < l.length; ++i) {
			final ILabelProviderListener listener = (ILabelProviderListener) l[i];
			SafeRunner.run(new SafeRunnable() {
				public void run() throws Exception {
					listener.labelProviderChanged(event);
				}
			});
		}
	}

	public void memberAnnotationChanged(MemberAnnotationChangedEvent event) {
		fireLabelProviderChangedEvent(new LabelProviderChangedEvent(this, event
				.getChanges().keySet().toArray()));
	}

	public void propertyChange(PropertyChangeEvent event) {
		boolean generatedColorChanged = MintUI.GENERATED_COLOR.equals(event
				.getProperty());
		boolean generatedNotColorChanged = MintUI.GENERATED_NOT_COLOR
				.equals(event.getProperty());
		if (generatedColorChanged || generatedNotColorChanged) {
			synchronized (colorLock) {
				if (colorJob == null)
					return;
			}

			colorJob.schedule();
		}
	}

	private Color getGeneratedColor() {
		synchronized (colorLock) {
			if (generatedColor == null) {
				if (colorJob == null)
					colorJob = new ColorJob();
			}
		}

		try {
			colorJob.join();
		} catch (InterruptedException e) {
			// ignore
		}

		return generatedColor;
	}

	private Color getGeneratedNotColor() {
		synchronized (colorLock) {
			if (generatedNotColor == null) {
				if (colorJob == null)
					colorJob = new ColorJob();
			}
		}

		try {
			colorJob.join();
		} catch (InterruptedException e) {
			// ignore
		}

		return generatedNotColor;
	}

	private class ColorJob extends WorkbenchJob {

		private boolean refresh;

		public ColorJob() {
			super(Messages.GeneratedElementDecorator_ThemeJob);
			setSystem(true);
			schedule();
		}

		public IStatus runInUIThread(IProgressMonitor monitor) {
			if (themeManager == null)
				return Status.CANCEL_STATUS;

			ColorRegistry registry = themeManager.getCurrentTheme()
					.getColorRegistry();
			synchronized (colorLock) {
				generatedColor = registry.get(MintUI.GENERATED_COLOR);
				generatedNotColor = registry.get(MintUI.GENERATED_NOT_COLOR);
			}

			if (refresh)
				fireLabelProviderChangedEvent(new LabelProviderChangedEvent(
						GeneratedElementDecorator.this));

			refresh = true;
			return Status.OK_STATUS;
		}
	}
}