package prefuse.util.display;

import java.util.Arrays;

import prefuse.util.ArrayLib;
import prefuse.visual.VisualItem;
import prefuse.visual.sort.ItemSorter;


/**
 * A helper class representing rendering and picking queues. This functionality
 * is listed separately to keep the Display implementation a bit cleaner.
 * Fields are public and used directly by a single Display instance.
 * 
 * @author <a href="http://jheer.org">jeffrey heer</a>
 */
public class RenderingQueue {

    private static final int DEFAULT_SIZE = 256;
    
    public ItemSorter   sort   = new ItemSorter();
    
    // rendering queue
    public VisualItem[] ritems  = new VisualItem[DEFAULT_SIZE];
    public int[]        rscores = new int[DEFAULT_SIZE];
    public int          rsize   = 0;
    
    // picking queue
    public VisualItem[] pitems  = new VisualItem[DEFAULT_SIZE];
    public int[]        pscores = new int[DEFAULT_SIZE];
    public int          psize   = 0;
    public boolean      psorted = false;
    
    // buffer queues for use in sorting, these prevent continual re-allocation
    transient static VisualItem[] items_buf;
    transient static int[]        scores_buf;
    
    /**
     * Clear both rendering and picking queues.
     */
    public void clear() {
        Arrays.fill(this.ritems, 0, this.rsize, null);
        Arrays.fill(this.pitems, 0, this.psize, null);
        this.rsize = 0;
        this.psize = 0;
    }
    
    /**
     * Clears the rendering queue and resizes internal arrays to a small size.
     * This should help reclaim used memory.
     */
    public void clean() {
    	clear();
    	this.ritems = new VisualItem[DEFAULT_SIZE];
    	this.rscores = new int[DEFAULT_SIZE];
    	this.pitems = new VisualItem[DEFAULT_SIZE];
    	this.pscores = new int[DEFAULT_SIZE];
    	items_buf = null;
    	scores_buf = null;
    }
    
    /**
     * Add an item to the rendering queue.
     * @param item the item to add
     */
    public void addToRenderQueue(VisualItem item) {
        if ( this.ritems.length == this.rsize ) {
            int capacity = (3*this.ritems.length)/2 + 1;
            VisualItem[] q = new VisualItem[capacity];
            int[] s = new int[capacity];
            System.arraycopy(this.ritems, 0, q, 0, this.rsize);
            System.arraycopy(this.rscores, 0, s, 0, this.rsize);
            this.ritems = q;
            this.rscores = s;
        }
        this.ritems[this.rsize] = item;
        this.rscores[this.rsize++] = (this.sort != null ? this.sort.score(item) : 0);
    }
    
    /**
     * Add an item to the picking queue.
     * @param item the item to add
     */
    public void addToPickingQueue(VisualItem item) {
        if ( this.pitems.length == this.psize ) {
            int capacity = (3*this.pitems.length)/2 + 1;
            VisualItem[] q = new VisualItem[capacity];
            int[] s = new int[capacity];
            System.arraycopy(this.pitems, 0, q, 0, this.psize);
            System.arraycopy(this.pscores, 0, s, 0, this.psize);
            this.pitems = q;
            this.pscores = s;
        }
        this.pitems[this.psize] = item;
        this.pscores[this.psize++] = (this.sort != null ? this.sort.score(item) : 0);
        this.psorted = false;
    }
    
    /**
     * Sort the rendering queue.
     */
    public void sortRenderQueue() {
        sort(this.ritems, this.rscores, this.rsize);
    }
    
    /**
     * Sort the picking queue. 
     */
    public void sortPickingQueue() {
        sort(this.pitems, this.pscores, this.psize);
        this.psorted = true;
    }
    
    /**
     * Sort a queue of items based upon an array of ordering scores. 
     */
    private void sort(VisualItem[] items, int[] scores, int size) {
        if ( this.sort == null ) return;
        // first check buffer queues
        if ( items_buf == null || items_buf.length < size ) {
            items_buf = new VisualItem[items.length];
            scores_buf = new int[scores.length];
        }
        // now sort
        ArrayLib.sort(scores, items, scores_buf, items_buf, 0, size);
    }
    
} // end of class RenderingQueue
