package prefuse.visual.tuple;

import java.awt.BasicStroke;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.geom.Rectangle2D;

import prefuse.Visualization;
import prefuse.data.Graph;
import prefuse.data.Table;
import prefuse.data.Tuple;
import prefuse.data.tuple.TableTuple;
import prefuse.data.tuple.TupleSet;
import prefuse.render.Renderer;
import prefuse.visual.VisualItem;
import prefuse.visual.VisualTable;


/**
 * VisualItem implementation that uses data values from a backing VisualTable.
 * 
 * @author <a href="http://jheer.org">jeffrey heer</a>
 */
public class TableVisualItem extends TableTuple implements VisualItem {

    /**
     * Initialize a new TableVisualItem for the given table and row. This
     * method is used by the appropriate TupleManager instance, and should
     * not be called directly by client code, unless by a client-supplied
     * custom TupleManager.
     * @param table the data Table
     * @param graph ignored by this class
     * @param row the table row index
     */
    protected void init(Table table, Graph graph, int row) {
        this.m_table = table;
        this.m_row = this.m_table.isValidRow(row) ? row : -1;
    }
    
    /**
     * @see prefuse.visual.VisualItem#getVisualization()
     */
    public Visualization getVisualization() {
        return ((VisualTable)this.m_table).getVisualization();
    }
    
    /**
     * @see prefuse.visual.VisualItem#getGroup()
     */
    public String getGroup() {
        return ((VisualTable)this.m_table).getGroup();
    }
    
    /**
     * @see prefuse.visual.VisualItem#isInGroup(java.lang.String)
     */
    public boolean isInGroup(String group) {
        return getVisualization().isInGroup(this, group);
    }
    
    /**
     * @see prefuse.visual.VisualItem#getSourceData()
     */
    public TupleSet getSourceData() {
        VisualTable vt = (VisualTable)this.m_table;
        return vt.getVisualization().getSourceData(vt.getGroup());
    }

    /**
     * @see prefuse.visual.VisualItem#getSourceTuple()
     */
    public Tuple getSourceTuple() {
        VisualTable vt = (VisualTable)this.m_table;
        return vt.getVisualization().getSourceTuple(this);
    }
    
    /**
     * @see java.lang.Object#toString()
     */
    public String toString() {
        StringBuffer sbuf = new StringBuffer();
        sbuf.append("VisualItem[").append(getGroup());
        sbuf.append(",").append(this.m_row).append(',');
        VisualTable vt = (VisualTable)this.m_table;
        int local = vt.getLocalColumnCount();
        int inherited = vt.getColumnCount()-local;
        for ( int i=0; i<inherited; ++i ) {
            if ( i > 0 ) sbuf.append(',');
            String name = vt.getColumnName(local+i);
            sbuf.append(name);
            sbuf.append('=');
            if ( vt.canGetString(name) )
                sbuf.append(vt.getString(this.m_row, name));
            else
                sbuf.append(vt.get(this.m_row, name).toString());
        }
        sbuf.append(']');
        
        return sbuf.toString();
    }
    
    // ------------------------------------------------------------------------
    // VisualItem Methods
    
    /**
     * @see prefuse.visual.VisualItem#render(java.awt.Graphics2D)
     */
    public void render(Graphics2D g) {
        getRenderer().render(g, this);
    }
    
    /**
     * @see prefuse.visual.VisualItem#getRenderer()
     */
    public Renderer getRenderer() {
        return getVisualization().getRenderer(this);
    }
    
    /**
     * @see prefuse.visual.VisualItem#validateBounds()
     */
    public Rectangle2D validateBounds() {
        if ( isValidated() ) return getBounds();
        
        Visualization v = getVisualization();
                
        // set the new bounds from the renderer and validate
        getRenderer().setBounds(this);
        setValidated(true);
        
        // report damage from the new bounds and return
        Rectangle2D bounds = getBounds();
        v.damageReport(this, bounds);
        return bounds;
    }
    
    // -- Boolean Flags -------------------------------------------------------
       
    /**
     * @see prefuse.visual.VisualItem#isValidated()
     */
    public boolean isValidated() {
        return ((VisualTable)this.m_table).isValidated(this.m_row);
    }
    
    /**
     * @see prefuse.visual.VisualItem#setValidated(boolean)
     */
    public void setValidated(boolean value) {
        ((VisualTable)this.m_table).setValidated(this.m_row, value);
    }
    
    /**
     * @see prefuse.visual.VisualItem#isVisible()
     */
    public boolean isVisible() {
        return ((VisualTable)this.m_table).isVisible(this.m_row);
    }
    
    /**
     * @see prefuse.visual.VisualItem#setVisible(boolean)
     */
    public void setVisible(boolean value) {
        ((VisualTable)this.m_table).setVisible(this.m_row, value);
    }
    
    /**
     * @see prefuse.visual.VisualItem#isStartVisible()
     */
    public boolean isStartVisible() {
        return ((VisualTable)this.m_table).isStartVisible(this.m_row);
    }
    
    /**
     * @see prefuse.visual.VisualItem#setStartVisible(boolean)
     */
    public void setStartVisible(boolean value) {
        ((VisualTable)this.m_table).setStartVisible(this.m_row, value);
    }
    
    /**
     * @see prefuse.visual.VisualItem#isEndVisible()
     */
    public boolean isEndVisible() {
        return ((VisualTable)this.m_table).isEndVisible(this.m_row);
    }
    
    /**
     * @see prefuse.visual.VisualItem#setEndVisible(boolean)
     */
    public void setEndVisible(boolean value) {
        ((VisualTable)this.m_table).setEndVisible(this.m_row, value);
    }
    
    /**
     * @see prefuse.visual.VisualItem#isInteractive()
     */
    public boolean isInteractive() {
        return ((VisualTable)this.m_table).isInteractive(this.m_row);
    }

    /**
     * @see prefuse.visual.VisualItem#setInteractive(boolean)
     */
    public void setInteractive(boolean value) {
        ((VisualTable)this.m_table).setInteractive(this.m_row, value);
    }
    
    /**
     * @see prefuse.visual.VisualItem#isExpanded()
     */
    public boolean isExpanded() {
        return ((VisualTable)this.m_table).isExpanded(this.m_row);
    }

    /**
     * @see prefuse.visual.VisualItem#setExpanded(boolean)
     */
    public void setExpanded(boolean value) {
        ((VisualTable)this.m_table).setExpanded(this.m_row, value);
    }
    
    /**
     * @see prefuse.visual.VisualItem#isFixed()
     */
    public boolean isFixed() {
        return ((VisualTable)this.m_table).isFixed(this.m_row);
    }

    /**
     * @see prefuse.visual.VisualItem#setFixed(boolean)
     */
    public void setFixed(boolean value) {
        ((VisualTable)this.m_table).setFixed(this.m_row, value);
    }
    
    /**
     * @see prefuse.visual.VisualItem#isHighlighted()
     */
    public boolean isHighlighted() {
        return ((VisualTable)this.m_table).isHighlighted(this.m_row);
    }
    
    /**
     * @see prefuse.visual.VisualItem#setHighlighted(boolean)
     */
    public void setHighlighted(boolean value) {
        ((VisualTable)this.m_table).setHighlighted(this.m_row, value);
    }

    /**
     * @see prefuse.visual.VisualItem#isHover()
     */
    public boolean isHover() {
        return ((VisualTable)this.m_table).isHover(this.m_row);
    }
    
    /**
     * @see prefuse.visual.VisualItem#setHover(boolean)
     */
    public void setHover(boolean value) {
        ((VisualTable)this.m_table).setHover(this.m_row, value);
    }
    
    // ------------------------------------------------------------------------
    
    /**
     * @see prefuse.visual.VisualItem#getX()
     */
    public double getX() {
        return ((VisualTable)this.m_table).getX(this.m_row);
    }
    
    /**
     * @see prefuse.visual.VisualItem#setX(double)
     */
    public void setX(double x) {
        ((VisualTable)this.m_table).setX(this.m_row, x);
    }
    
    /**
     * @see prefuse.visual.VisualItem#getY()
     */
    public double getY() {
        return ((VisualTable)this.m_table).getY(this.m_row);
    }
    
    /**
     * @see prefuse.visual.VisualItem#setY(double)
     */
    public void setY(double y) {
        ((VisualTable)this.m_table).setY(this.m_row, y);
    }
    
    /**
     * @see prefuse.visual.VisualItem#getStartX()
     */
    public double getStartX() {
        return ((VisualTable)this.m_table).getStartX(this.m_row);
    }
    
    /**
     * @see prefuse.visual.VisualItem#setStartX(double)
     */
    public void setStartX(double x) {
        ((VisualTable)this.m_table).setStartX(this.m_row, x);
    }
    
    /**
     * @see prefuse.visual.VisualItem#getStartY()
     */
    public double getStartY() {
        return ((VisualTable)this.m_table).getStartY(this.m_row);
    }
    
    /**
     * @see prefuse.visual.VisualItem#setStartY(double)
     */
    public void setStartY(double y) {
        ((VisualTable)this.m_table).setStartY(this.m_row, y);
    }
    
    /**
     * @see prefuse.visual.VisualItem#getEndX()
     */
    public double getEndX() {
        return ((VisualTable)this.m_table).getEndX(this.m_row);
    }
    
    /**
     * @see prefuse.visual.VisualItem#setEndX(double)
     */
    public void setEndX(double x) {
        ((VisualTable)this.m_table).setEndX(this.m_row, x);
    }
    
    /**
     * @see prefuse.visual.VisualItem#getEndY()
     */
    public double getEndY() {
        return ((VisualTable)this.m_table).getEndY(this.m_row);
    }
    
    /**
     * @see prefuse.visual.VisualItem#setEndY(double)
     */
    public void setEndY(double y) {
        ((VisualTable)this.m_table).setEndY(this.m_row, y);
    }

    /**
     * @see prefuse.visual.VisualItem#getBounds()
     */
    public Rectangle2D getBounds() {
        if ( !isValidated() ) {
            return validateBounds();
        }
        return ((VisualTable)this.m_table).getBounds(this.m_row);
    }
    
    /**
     * @see prefuse.visual.VisualItem#setBounds(double, double, double, double)
     */
    public void setBounds(double x, double y, double w, double h) {
        ((VisualTable)this.m_table).setBounds(this.m_row, x, y, w, h);
    }
    
    // ------------------------------------------------------------------------
    
    /**
     * @see prefuse.visual.VisualItem#getStrokeColor()
     */
    public int getStrokeColor() {
        return ((VisualTable)this.m_table).getStrokeColor(this.m_row);
    }
    
    /**
     * @see prefuse.visual.VisualItem#setStrokeColor(int)
     */
    public void setStrokeColor(int color) {
        ((VisualTable)this.m_table).setStrokeColor(this.m_row, color);
    }
    
    /**
     * @see prefuse.visual.VisualItem#getStartStrokeColor()
     */
    public int getStartStrokeColor() {
        return ((VisualTable)this.m_table).getStartStrokeColor(this.m_row);
    }
    
    /**
     * @see prefuse.visual.VisualItem#setStartStrokeColor(int)
     */
    public void setStartStrokeColor(int color) {
        ((VisualTable)this.m_table).setStartStrokeColor(this.m_row, color);
    }
    
    /**
     * @see prefuse.visual.VisualItem#getEndStrokeColor()
     */
    public int getEndStrokeColor() {
        return ((VisualTable)this.m_table).getEndStrokeColor(this.m_row);
    }
    
    /**
     * @see prefuse.visual.VisualItem#setEndStrokeColor(int)
     */
    public void setEndStrokeColor(int color) {
        ((VisualTable)this.m_table).setEndStrokeColor(this.m_row, color);
    }
    
    /**
     * @see prefuse.visual.VisualItem#getFillColor()
     */
    public int getFillColor() {
        return ((VisualTable)this.m_table).getFillColor(this.m_row);
    }
    
    /**
     * @see prefuse.visual.VisualItem#setFillColor(int)
     */
    public void setFillColor(int color) {
        ((VisualTable)this.m_table).setFillColor(this.m_row, color);
    }
    
    /**
     * @see prefuse.visual.VisualItem#getStartFillColor()
     */
    public int getStartFillColor() {
        return ((VisualTable)this.m_table).getStartFillColor(this.m_row);
    }

    /**
     * @see prefuse.visual.VisualItem#setStartFillColor(int)
     */
    public void setStartFillColor(int color) {
        ((VisualTable)this.m_table).setStartFillColor(this.m_row, color);
    }
    
    /**
     * @see prefuse.visual.VisualItem#getEndFillColor()
     */
    public int getEndFillColor() {
        return ((VisualTable)this.m_table).getEndFillColor(this.m_row);
    }
    
    /**
     * @see prefuse.visual.VisualItem#setEndFillColor(int)
     */
    public void setEndFillColor(int color) {
        ((VisualTable)this.m_table).setEndFillColor(this.m_row, color);
    }
    
    /**
     * @see prefuse.visual.VisualItem#getTextColor()
     */
    public int getTextColor() {
        return ((VisualTable)this.m_table).getTextColor(this.m_row);
    }
    
    /**
     * @see prefuse.visual.VisualItem#setTextColor(int)
     */
    public void setTextColor(int color) {
        ((VisualTable)this.m_table).setTextColor(this.m_row, color);
    }
    
    /**
     * @see prefuse.visual.VisualItem#getStartTextColor()
     */
    public int getStartTextColor() {
        return ((VisualTable)this.m_table).getStartTextColor(this.m_row);
    }
    
    /**
     * @see prefuse.visual.VisualItem#setStartTextColor(int)
     */
    public void setStartTextColor(int color) {
        ((VisualTable)this.m_table).setStartTextColor(this.m_row, color);
    }
    
    /**
     * @see prefuse.visual.VisualItem#getEndTextColor()
     */
    public int getEndTextColor() {
        return ((VisualTable)this.m_table).getEndTextColor(this.m_row);
    }
    
    /**
     * @see prefuse.visual.VisualItem#setEndTextColor(int)
     */
    public void setEndTextColor(int color) {
        ((VisualTable)this.m_table).setEndTextColor(this.m_row, color);
    }
    
    // ------------------------------------------------------------------------
    
    /**
     * @see prefuse.visual.VisualItem#getSize()
     */
    public double getSize() {
        return ((VisualTable)this.m_table).getSize(this.m_row);
    }
    
    /**
     * @see prefuse.visual.VisualItem#setSize(double)
     */
    public void setSize(double size) {
        ((VisualTable)this.m_table).setSize(this.m_row, size);
    }
    
    /**
     * @see prefuse.visual.VisualItem#getStartSize()
     */
    public double getStartSize() {
        return ((VisualTable)this.m_table).getStartSize(this.m_row);
    }
    
    /**
     * @see prefuse.visual.VisualItem#setStartSize(double)
     */
    public void setStartSize(double size) {
        ((VisualTable)this.m_table).setStartSize(this.m_row, size);
    }
    
    /**
     * @see prefuse.visual.VisualItem#getEndSize()
     */
    public double getEndSize() {
        return ((VisualTable)this.m_table).getEndSize(this.m_row);
    }
    
    /**
     * @see prefuse.visual.VisualItem#setEndSize(double)
     */
    public void setEndSize(double size) {
        ((VisualTable)this.m_table).setEndSize(this.m_row, size);
    }
    
    // ------------------------------------------------------------------------
    
    /**
     * @see prefuse.visual.VisualItem#getShape()
     */
    public int getShape() {
        return ((VisualTable)this.m_table).getShape(this.m_row);
    }
    
    /**
     * @see prefuse.visual.VisualItem#setShape(int)
     */
    public void setShape(int shape) {
        ((VisualTable)this.m_table).setShape(this.m_row, shape);
    }
    
    // ------------------------------------------------------------------------
    
    /**
     * @see prefuse.visual.VisualItem#getStroke()
     */
    public BasicStroke getStroke() {
        return ((VisualTable)this.m_table).getStroke(this.m_row);
    }
    
    /**
     * @see prefuse.visual.VisualItem#setStroke(java.awt.BasicStroke)
     */
    public void setStroke(BasicStroke stroke) {
        ((VisualTable)this.m_table).setStroke(this.m_row, stroke);
    }
    
    // ------------------------------------------------------------------------    
    
    /**
     * @see prefuse.visual.VisualItem#getFont()
     */
    public Font getFont() {
        return ((VisualTable)this.m_table).getFont(this.m_row);
    }

    /**
     * @see prefuse.visual.VisualItem#setFont(java.awt.Font)
     */
    public void setFont(Font font) {
        ((VisualTable)this.m_table).setFont(this.m_row, font);
    }
    
    /**
     * @see prefuse.visual.VisualItem#getStartFont()
     */
    public Font getStartFont() {
        return ((VisualTable)this.m_table).getStartFont(this.m_row);
    }

    /**
     * @see prefuse.visual.VisualItem#setStartFont(java.awt.Font)
     */
    public void setStartFont(Font font) {
        ((VisualTable)this.m_table).setStartFont(this.m_row, font);
    }
    
    /**
     * @see prefuse.visual.VisualItem#getEndFont()
     */
    public Font getEndFont() {
        return ((VisualTable)this.m_table).getEndFont(this.m_row);
    }
    
    /**
     * @see prefuse.visual.VisualItem#setEndFont(java.awt.Font)
     */
    public void setEndFont(Font font) {
        ((VisualTable)this.m_table).setEndFont(this.m_row, font);
    }
    
    // ------------------------------------------------------------------------
    
    /**
     * @see prefuse.visual.VisualItem#getDOI()
     */
    public double getDOI() {
        return ((VisualTable)this.m_table).getDOI(this.m_row);
    }
    
    /**
     * @see prefuse.visual.VisualItem#setDOI(double)
     */
    public void setDOI(double doi) {
        ((VisualTable)this.m_table).setDOI(this.m_row, doi);
    }
    
} // end of class TableVisualItem
