/**
 * Copyright (c) 2010, 2013 Darmstadt University of Technology.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Marcel Bruch - initial API and implementation.
 */
package org.eclipse.recommenders.livedoc.cli.aether;

import static org.eclipse.recommenders.utils.Checks.ensureIsDirectory;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import org.apache.maven.repository.internal.MavenRepositorySystemUtils;
import org.eclipse.aether.DefaultRepositorySystemSession;
import org.eclipse.aether.RepositorySystem;
import org.eclipse.aether.RepositorySystemSession;
import org.eclipse.aether.artifact.Artifact;
import org.eclipse.aether.collection.CollectRequest;
import org.eclipse.aether.collection.CollectResult;
import org.eclipse.aether.collection.DependencyCollectionException;
import org.eclipse.aether.connector.basic.BasicRepositoryConnectorFactory;
import org.eclipse.aether.deployment.DeployRequest;
import org.eclipse.aether.deployment.DeploymentException;
import org.eclipse.aether.graph.Dependency;
import org.eclipse.aether.impl.DefaultServiceLocator;
import org.eclipse.aether.repository.LocalRepository;
import org.eclipse.aether.repository.RemoteRepository;
import org.eclipse.aether.resolution.ArtifactDescriptorException;
import org.eclipse.aether.resolution.ArtifactDescriptorRequest;
import org.eclipse.aether.resolution.ArtifactDescriptorResult;
import org.eclipse.aether.resolution.ArtifactRequest;
import org.eclipse.aether.resolution.ArtifactResolutionException;
import org.eclipse.aether.resolution.ArtifactResult;
import org.eclipse.aether.spi.connector.RepositoryConnectorFactory;
import org.eclipse.aether.spi.connector.transport.TransporterFactory;
import org.eclipse.aether.transport.file.FileTransporterFactory;
import org.eclipse.aether.transport.http.HttpTransporterFactory;
import org.eclipse.aether.util.artifact.JavaScopes;
import org.eclipse.aether.util.graph.visitor.PreorderNodeListGenerator;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RepositoryClient {

    private static final String REPOSITORY_LAYOUT = "default";

    private final File cacheDir;
    private final org.eclipse.aether.RepositorySystem repositorySystem;
    private static final Logger LOG = LoggerFactory.getLogger(RepositoryClient.class);

    public RepositoryClient(File cacheDir) throws IOException {
        this.cacheDir = ensureIsDirectory(cacheDir);
        repositorySystem = createRepositorySystem();
    }

    private RepositorySystem createRepositorySystem() {
        DefaultServiceLocator locator = MavenRepositorySystemUtils.newServiceLocator();

        locator.addService(RepositoryConnectorFactory.class, BasicRepositoryConnectorFactory.class);
        locator.addService(TransporterFactory.class, FileTransporterFactory.class);
        locator.addService(TransporterFactory.class, HttpTransporterFactory.class);

        return locator.getService(RepositorySystem.class);
    }

    public Artifact download(Artifact coordinate, URL repo) throws IOException {
        try {
            RepositorySystemSession session = newRepositorySystemSession();
            List<RemoteRepository> repos = Collections.singletonList(toRemoteRepository(repo));
            ArtifactRequest request = new ArtifactRequest(coordinate, repos, null);
            ArtifactResult result = repositorySystem.resolveArtifact(session, request);
            return result.getArtifact();
        } catch (ArtifactResolutionException e) {
            throw new IOException(e.getMessage(), e);
        }
    }

    public void upload(Artifact artifact, URL repo) throws IOException {
        try {
            RepositorySystemSession session = newRepositorySystemSession();
            DeployRequest deployRequest = new DeployRequest();
            deployRequest.addArtifact(artifact);
            deployRequest.setRepository(toRemoteRepository(repo));
            repositorySystem.deploy(session, deployRequest);
        } catch (DeploymentException e) {
            throw new IOException(e);
        }
    }

    private DefaultRepositorySystemSession newRepositorySystemSession() {
        DefaultRepositorySystemSession session = MavenRepositorySystemUtils.newSession();
        LocalRepository localRepo = new LocalRepository(cacheDir);
        session.setLocalRepositoryManager(repositorySystem.newLocalRepositoryManager(session, localRepo));

        // NullProgressMonitor: saving this one for a potential later use
        session.setTransferListener(new TransferListener(new NullProgressMonitor()));
        return session;
    }

    private RemoteRepository toRemoteRepository(URL repo) {
        return new RemoteRepository.Builder("remote", REPOSITORY_LAYOUT, repo.toExternalForm()).build();
    }

    public List<Artifact> resolveDirectDependencies(Artifact sourceArtifact, URL repo) throws IOException {

        RemoteRepository remoteRepo = toRemoteRepository(repo);

        List<Dependency> dependencies = getDirectDependencies(sourceArtifact, remoteRepo);
        return downloadDependencies(remoteRepo, dependencies);
    }

    private List<Dependency> getDirectDependencies(Artifact sourceArtifact, RemoteRepository remoteRepo)
            throws IOException {

        ArtifactDescriptorRequest descriptorRequest = new ArtifactDescriptorRequest();
        descriptorRequest.setArtifact(sourceArtifact);
        descriptorRequest.addRepository(remoteRepo);

        RepositorySystemSession session = newRepositorySystemSession();

        ArtifactDescriptorResult descriptorResult = null;
        try {
            descriptorResult = repositorySystem.readArtifactDescriptor(session, descriptorRequest);
        } catch (ArtifactDescriptorException e) {
            throw new IOException(e);
        }
        return descriptorResult.getDependencies();
    }

    private List<Artifact> downloadDependencies(RemoteRepository remoteRepo,
            List<Dependency> dependencies) {
        List<RemoteRepository> repos = Collections.singletonList(remoteRepo);

        List<Artifact> artifacts = new LinkedList<Artifact>();
        RepositorySystemSession session = newRepositorySystemSession();

        for (Dependency dependency : dependencies) {
            ArtifactRequest request = new ArtifactRequest(dependency.getArtifact(), repos, null);
            try {
                artifacts.add(repositorySystem.resolveArtifact(session, request).getArtifact());
            } catch (ArtifactResolutionException e) {
                LOG.warn("Failed to resolve dependency \"{}\"\n\t{}\n", dependency.getArtifact(), e);
            }
        }
        return artifacts;
    }

    public List<Artifact> resolveTransitiveDependencies(Artifact sourceArtifact, URL repo) throws IOException {

        RemoteRepository remoteRepo = toRemoteRepository(repo);

        List<Dependency> dependencies = getTransitiveDependencies(sourceArtifact, remoteRepo);
        return downloadDependencies(remoteRepo, dependencies);
    }

    private List<Dependency> getTransitiveDependencies(Artifact sourceArtifact, RemoteRepository remoteRepo)
            throws IOException {
        CollectRequest collectRequest = new CollectRequest();
        collectRequest.setRoot(new Dependency(sourceArtifact, JavaScopes.COMPILE));
        collectRequest.addRepository(remoteRepo);

        RepositorySystemSession session = newRepositorySystemSession();

        List<Dependency> dependencies = null;
        try {
            CollectResult result = repositorySystem.collectDependencies(session, collectRequest);
            PreorderNodeListGenerator nlg = new PreorderNodeListGenerator();
            result.getRoot().accept(nlg);
            dependencies = nlg.getDependencies(true);
            if (dependencies.size() != 0) {
                dependencies.remove(0);
            }
        } catch (DependencyCollectionException e) {
            throw new IOException(e);
        }
        return dependencies;
    }
}
