/**
 * Copyright (c) 2014 Patrick Gottschaemmer.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.recommenders.internal.livedoc.javadoc;

import java.net.URL;
import java.util.List;
import java.util.Map;

import org.eclipse.recommenders.internal.livedoc.IProviderManager;
import org.eclipse.recommenders.internal.livedoc.PluginProviderManager;
import org.eclipse.recommenders.livedoc.providers.ILivedocProvider;
import org.eclipse.recommenders.models.ProjectCoordinate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sun.javadoc.RootDoc;
import com.sun.tools.doclets.formats.html.HtmlDoclet;
import com.sun.tools.doclets.internal.toolkit.Configuration;
import com.sun.tools.doclets.standard.Standard;

public class RecommendersDoclet extends Standard {

    private static HtmlDoclet doclet;
    private static RecommendersDoclet instance;
    private IProviderManager manager;
    private List<ILivedocProvider<?>> livedocProviders;
    private static Logger LOG = LoggerFactory.getLogger(RecommendersDoclet.class);
    private List<URL> modelRepositories;
    private boolean highlight;
    private ProjectCoordinate projectCoordinate;

    private RecommendersDoclet() {
        manager = new PluginProviderManager();
    }

    public static RecommendersDoclet instance() {

        if (instance == null) {
            instance = new RecommendersDoclet();
        }
        return instance;
    }

    /**
     * Javadoc calls this Method
     * 
     * We want to use the standard HtmlDoclet, but call setOptions() first to make sure that
     * doclet.configuration.tagletManager != null for adding custom taglets on this TagletManager.
     */
    public static boolean start(RootDoc root) {
        try {
            doclet = new HtmlDoclet();

            // root shouldn't be null for setOptions()
            Configuration conf = doclet.configuration();
            conf.root = root;
            conf.setOptions();

            // load custom taglets via extension point
            instance().configureRecommendersTaglet();
            return doclet.start(doclet, root);
        } catch (Exception e) {
            LOG.error("Execution of RecommendersDoclet failed:", e);
            return false;
        }
    }

    private void configureRecommendersTaglet() {

        livedocProviders = manager.getProviders();
        RecommendersTaglet taglet = new RecommendersTaglet(projectCoordinate, modelRepositories, livedocProviders,
                highlight);

        try {
            taglet.setUp();
        } catch (RecommendersTagletException e) {
            LOG.error("RecommendersTaglet couldn't be successfully initiated. Generating plain Javadoc", e);
            return;
        }
        doclet.configuration.tagletManager.addCustomTag(taglet);
    }

    public void setProjectCoordinate(ProjectCoordinate projectCoordinate) {
        this.projectCoordinate = projectCoordinate;
    }

    public void setHighlight(boolean highlight) {
        this.highlight = highlight;
    }

    public void setModelRepositories(List<URL> modelRepositories) {
        this.modelRepositories = modelRepositories;
    }

    public List<String> getProviderIds() {
        return manager.getProviderIds();
    }

    public void loadProviders(Map<String, String[]> providerArguments) {
        manager.loadProviders(providerArguments);
    }
}
