/**
 * Copyright (c) 2014 Patrick Gottschaemmer.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.recommenders.livedoc.providers;

import java.io.File;

import org.apache.commons.lang3.ArrayUtils;
import org.eclipse.recommenders.models.IModelIndex;
import org.eclipse.recommenders.models.IModelRepository;
import org.eclipse.recommenders.models.ModelCoordinate;
import org.eclipse.recommenders.models.ProjectCoordinate;
import org.eclipse.recommenders.utils.Nullable;
import org.kohsuke.args4j.CmdLineException;
import org.kohsuke.args4j.CmdLineParser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Optional;
import com.sun.javadoc.AnnotationTypeDoc;
import com.sun.javadoc.AnnotationTypeElementDoc;
import com.sun.javadoc.ClassDoc;
import com.sun.javadoc.ConstructorDoc;
import com.sun.javadoc.FieldDoc;
import com.sun.javadoc.MethodDoc;
import com.sun.javadoc.PackageDoc;
import com.sun.javadoc.RootDoc;

public abstract class AbstractLiveDocProvider<C extends ProviderConfiguration> implements ILivedocProvider<C> {

    private static final Logger LOG = LoggerFactory.getLogger(AbstractLiveDocProvider.class);

    private ProjectCoordinate pc;
    private IModelRepository modelRepo;
    private IModelIndex modelIndex;
    private C configuration;
    @Nullable
    private String[] providerArguments;

    public AbstractLiveDocProvider() {
        configuration = getConfiguration();
    }

    @Override
    public void setUp(ProjectCoordinate pc, IModelRepository repo, IModelIndex index) throws LiveDocProviderException {
        this.pc = pc;
        this.modelRepo = repo;
        this.modelIndex = index;
        parseProviderArguments(ArrayUtils.nullToEmpty(providerArguments));
    }

    private void parseProviderArguments(String[] providerArguments) throws LiveDocProviderException {
        CmdLineParser parser = new CmdLineParser(getConfiguration());
        try {
            parser.parseArgument(providerArguments);
        } catch (CmdLineException e) {
            throw new LiveDocProviderException(e);
        }
    }

    @Override
    public C getConfiguration() {
        if (configuration == null) {
            configuration = newProviderConfiguration();
        }
        return configuration;
    }

    @Override
    public ProviderOutput documentOverview(RootDoc holder) {
        // No-op Implementation
        return null;
    }

    @Override
    public void beginPackage(PackageDoc newPackageDoc) {
        // No-op Implementation
    }

    @Override
    public ProviderOutput documentPackage(PackageDoc holder) {
        // No-op Implementation
        return null;
    }

    @Override
    public void beginClass(ClassDoc newClassDoc) {
        // No-op Implementation
    }

    @Override
    public ProviderOutput documentClass(ClassDoc holder) {
        // No-op Implementation
        return null;
    }

    @Override
    public ProviderOutput documentField(FieldDoc holder) {
        // No-op Implementation
        return null;
    }

    @Override
    public ProviderOutput documentConstructor(ConstructorDoc holder) {
        // No-op Implementation
        return null;
    }

    @Override
    public ProviderOutput documentMethod(MethodDoc holder) {
        // No-op Implementation
        return null;
    }

    @Override
    public void endClass(ClassDoc oldClassDoc) {
        // No-op Implementation
    }

    @Override
    public void beginAnnotationType(AnnotationTypeDoc newAnnotationType) {
        // No-op Implementation
    }

    @Override
    public ProviderOutput documentAnnotationType(AnnotationTypeDoc holder) {
        // No-op Implementation
        return null;
    }

    @Override
    public ProviderOutput documentAnnotationTypeElement(AnnotationTypeElementDoc holder) {
        // No-op Implementation
        return null;
    }

    @Override
    public void endAnnotationType(AnnotationTypeDoc oldAnnotationTypeDoc) {
        // No-op Implementation
    }

    @Override
    public void endPackage(PackageDoc oldPackageDoc) {
        // No-op Implementation
    }

    @Override
    public void tearDown() throws LiveDocProviderException {
        // No-op Implementation
    }

    protected Optional<File> fetchModelArchive(String classifier) {
        Optional<File> modelArchive = Optional.absent();

        Optional<ModelCoordinate> coordinate = modelIndex.suggest(pc, classifier);
        if (!coordinate.isPresent()) {
            return Optional.absent();
        }
        modelArchive = modelRepo.resolve(coordinate.get(), false);
        if (!modelArchive.isPresent()) {
            LOG.error("Couldn't find model for indexed coordinate. The index of model repository \"{}\" is outdated.",
                    modelRepo);
        }
        return modelArchive;
    }

    /**
     * Highlight HTML output if flag is set
     */
    protected StringBuilder highlight(StringBuilder sb) {
        if (configuration.isHighlight()) {
            sb.insert(0, "<div style=\"background-color:#FFFFD5;\">");
            sb.append("</div>");
        }
        return sb;
    }

    /**
     * Highlight HTML output if flag is set
     */
    protected String highlight(String string) {
        if (configuration.isHighlight()) {
            return new StringBuilder("<div style=\"background-color:#FFFFD5;\">").append(string).append("</div>")
                    .toString();
        } else {
            return string;
        }
    }

    @Override
    public int getRanking() {
        return Integer.MAX_VALUE;
    }

    public String[] getArguments() {
        return providerArguments;
    }

    @Override
    public void setArguments(String[] args) {
        this.providerArguments = args;
    }

    public ProjectCoordinate getProjectCoordinate() {
        return pc;
    }

    public IModelRepository getRepo() {
        return modelRepo;
    }

    public IModelIndex getIndex() {
        return modelIndex;
    }

    @Override
    public int compareTo(ILivedocProvider<?> o) {
        return Integer.valueOf(getRanking()).compareTo(o.getRanking());
    }
}
