/**
 * Copyright (c) 2010, 2013 Darmstadt University of Technology.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Olav Lenz - initial API and implementation.
 */
package org.eclipse.recommenders.livedoc.utils;

import static org.eclipse.recommenders.livedoc.utils.LiveDocUtils.*;

import java.util.Iterator;
import java.util.List;

import com.google.common.collect.Lists;

public class MethodDocumentationBuilder {

    public enum ListType {
        SEQUENTIAL,
        BULLETS
    }

    private String description;
    private String title;
    private List<String> content = Lists.newArrayList();

    private List<String> listItems = Lists.newArrayList();
    private ListType listType;

    private MethodDocumentationBuilder() {
    }

    public static MethodDocumentationBuilder create() {
        return new MethodDocumentationBuilder();
    }

    public MethodDocumentationBuilder title(String title) {
        this.title = title;
        return this;
    }

    public MethodDocumentationBuilder description(String description) {
        this.description = description;
        return this;
    }

    public MethodDocumentationBuilder paragraph(String paragraph) {
        content.add(paragraph);
        return this;
    }

    public MethodDocumentationBuilder beginList(ListType listType) {
        listItems = Lists.newArrayList();
        this.listType = listType;
        return this;
    }

    public MethodDocumentationBuilder addItem(String item) {
        listItems.add(item);
        return this;
    }

    public MethodDocumentationBuilder endList() {
        switch (listType) {
        case SEQUENTIAL:
            content.add(createSequentialList());
            break;
        case BULLETS:
            content.add(createBulletList());
            break;
        }
        return this;
    }

    private String createSequentialList() {
        StringBuilder sb = new StringBuilder();
        for (Iterator<String> it = listItems.iterator(); it.hasNext();) {
            sb.append(it.next());
            if (it.hasNext()) {
                sb.append(", ");
            }
        }
        return sb.toString();
    }

    private String createBulletList() {
        StringBuilder sb = new StringBuilder();
        sb.append(UL);
        for (Iterator<String> it = listItems.iterator(); it.hasNext();) {
            sb.append(LI);
            sb.append(it.next());
            if (it.hasNext()) {
                sb.append(",");
            }
            sb.append(LI_END);
        }
        sb.append(UL_END);
        return sb.toString();
    }

    public String build() {
        StringBuilder sb = new StringBuilder();
        sb.append(surroundWith(DT, createTitle(), DT_END));
        sb.append(surroundWith(DD, createDescription() + createContent(), DD_END));
        return sb.toString();
    }

    private String createTitle() {
        if (isNullOrEmpty(title)) {
            return "";
        }
        return surroundWith("<span class=\"strong\">", title, "</span>");
    }

    private boolean isNullOrEmpty(String string) {
        return string == null || "".equals(string);
    }

    private String createDescription() {
        if (isNullOrEmpty(description)) {
            return "";
        }
        return description + BR + BR;
    }

    private String createContent() {
        StringBuilder sb = new StringBuilder();
        for (Iterator<String> it = content.iterator(); it.hasNext();) {
            sb.append(it.next());
            if (it.hasNext()) {
                sb.append(BR);
            }
        }
        return sb.toString();
    }
}
