/*******************************************************************************
 * Copyright (c) 2006, 2017 IBM Corporation and others.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 *
 *******************************************************************************/
package org.eclipse.dltk.ui.text.completion;

import java.util.Comparator;
import java.util.List;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.jface.text.contentassist.ICompletionProposal;

/**
 * Abstract base class for sorters contributed to the
 * <code>org.eclipse.dltk.ui.javaCompletionProposalSorters</code> extension
 * point.
 * <p>
 * Subclasses need to implement
 * {@link #compare(ICompletionProposal, ICompletionProposal)} and may override
 * {@link #beginSorting(ContentAssistInvocationContext) beginSorting} and
 * {@link #endSorting() endSorting}.
 * </p>
 * <p>
 * The orderings imposed by a subclass need not be consistent with equals.
 * </p>
 */
public abstract class AbstractProposalSorter implements
		Comparator<ICompletionProposal> {

	/**
	 * Creates a new sorter. Note that subclasses must provide a zero-argument
	 * constructor to be instantiatable via
	 * {@link IConfigurationElement#createExecutableExtension(String)}.
	 */
	protected AbstractProposalSorter() {
	}

	/**
	 * Called once before sorting.
	 * <p>
	 * Clients may override, the default implementation does nothing.
	 * </p>
	 *
	 * @param context
	 *            the context of the content assist invocation
	 */
	public void beginSorting(ContentAssistInvocationContext context) {
	}

	/**
	 * Called once before sorting.
	 * <p>
	 * Clients may override, the default implementation does nothing.
	 * </p>
	 *
	 * @param context
	 *            the context of the content assist invocation
	 * @param proposals
	 *            the list of proposals to be sorted
	 */
	public void beginSorting(ContentAssistInvocationContext context,
			List<ICompletionProposal> proposals) {
		beginSorting(context);
	}

	/**
	 * Implements the same contract as
	 * {@link Comparator#compare(Object, Object)} but with completion proposals
	 * as parameters. This method will implement the {@link Comparator}
	 * interface if this class is ever converted to extend
	 * <code>Comparator&lt;ICompletionProposal&gt;</code>.
	 * <p>
	 * The orderings imposed by an implementation need not be consistent with
	 * equals.
	 * </p>
	 *
	 * @param p1
	 *            the first proposal to be compared
	 * @param p2
	 *            the second proposal to be compared
	 * @return a negative integer, zero, or a positive integer as the first
	 *         argument is less than, equal to, or greater than the second.
	 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	 */
	@Override
	public abstract int compare(ICompletionProposal p1, ICompletionProposal p2);

	/**
	 * Called once after sorting.
	 * <p>
	 * Clients may override, the default implementation does nothing.
	 * </p>
	 */
	public void endSorting() {
	}
}
