/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.smarthome.binding.meteoblue.handler;

import com.google.gson.Gson;
import java.awt.image.BufferedImage;
import java.awt.image.RenderedImage;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Calendar;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import javax.imageio.ImageIO;
import javax.measure.Unit;
import org.apache.commons.lang.StringUtils;
import org.eclipse.smarthome.binding.meteoblue.handler.MeteoBlueBridgeHandler;
import org.eclipse.smarthome.binding.meteoblue.internal.Forecast;
import org.eclipse.smarthome.binding.meteoblue.internal.MeteoBlueConfiguration;
import org.eclipse.smarthome.binding.meteoblue.internal.json.JsonData;
import org.eclipse.smarthome.core.library.items.ImageItem;
import org.eclipse.smarthome.core.library.types.DateTimeType;
import org.eclipse.smarthome.core.library.types.DecimalType;
import org.eclipse.smarthome.core.library.types.QuantityType;
import org.eclipse.smarthome.core.library.types.RawType;
import org.eclipse.smarthome.core.library.types.StringType;
import org.eclipse.smarthome.core.library.unit.MetricPrefix;
import org.eclipse.smarthome.core.library.unit.SIUnits;
import org.eclipse.smarthome.core.thing.Bridge;
import org.eclipse.smarthome.core.thing.Channel;
import org.eclipse.smarthome.core.thing.ChannelUID;
import org.eclipse.smarthome.core.thing.Thing;
import org.eclipse.smarthome.core.thing.ThingStatus;
import org.eclipse.smarthome.core.thing.ThingStatusDetail;
import org.eclipse.smarthome.core.thing.binding.BaseThingHandler;
import org.eclipse.smarthome.core.types.Command;
import org.eclipse.smarthome.core.types.State;
import org.eclipse.smarthome.io.net.http.HttpUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MeteoBlueHandler
extends BaseThingHandler {
    private final Logger logger = LoggerFactory.getLogger(MeteoBlueHandler.class);
    private Bridge bridge;
    private Forecast[] forecasts;
    private Gson gson = new Gson();
    private JsonData weatherData;
    private ScheduledFuture<?> refreshJob;
    private boolean properlyConfigured;

    public MeteoBlueHandler(Thing thing) {
        super(thing);
        this.forecasts = new Forecast[7];
    }

    public void handleCommand(ChannelUID channelUID, Command command) {
        if (this.properlyConfigured) {
            this.logger.debug("Received command '{}' for channel '{}'", (Object)command, (Object)channelUID);
            this.updateChannel(channelUID.getId());
        }
    }

    public void initialize() {
        this.logger.debug("Initializing the meteoblue handler...");
        this.bridge = this.getBridge();
        if (this.bridge == null) {
            this.logger.warn("Unable to initialize meteoblue. No bridge was configured.");
            this.updateStatus(ThingStatus.OFFLINE, ThingStatusDetail.CONFIGURATION_ERROR, "Bridge not configured.");
            return;
        }
        MeteoBlueConfiguration config = (MeteoBlueConfiguration)this.getConfigAs(MeteoBlueConfiguration.class);
        if (StringUtils.isBlank((String)config.serviceType)) {
            config.serviceType = "NonCommercial";
            this.logger.debug("Using default service type ({}).", (Object)config.serviceType);
            return;
        }
        if (StringUtils.isBlank((String)config.location)) {
            this.flagBadConfig("The location was not configured.");
            return;
        }
        config.parseLocation();
        if (config.latitude == null) {
            this.flagBadConfig(String.format("Could not determine latitude from the defined location setting (%s).", config.location));
            return;
        }
        if (config.latitude > 90.0 || config.latitude < -90.0) {
            this.flagBadConfig(String.format("Specified latitude value (%d) is not valid.", config.latitude));
            return;
        }
        if (config.longitude == null) {
            this.flagBadConfig(String.format("Could not determine longitude from the defined location setting (%s).", config.location));
            return;
        }
        if (config.longitude > 180.0 || config.longitude < -180.0) {
            this.flagBadConfig(String.format("Specified longitude value (%d) is not valid.", config.longitude));
            return;
        }
        this.updateStatus(ThingStatus.UNKNOWN);
        this.startAutomaticRefresh(config);
        this.properlyConfigured = true;
    }

    private void flagBadConfig(String message) {
        this.properlyConfigured = false;
        this.updateStatus(ThingStatus.OFFLINE, ThingStatusDetail.CONFIGURATION_ERROR, message);
    }

    private void startAutomaticRefresh(MeteoBlueConfiguration config) {
        if (this.refreshJob != null && !this.refreshJob.isCancelled()) {
            this.logger.trace("Refresh job already exists.");
            return;
        }
        Runnable runnable = () -> {
            boolean updateSuccessful = false;
            try {
                updateSuccessful = this.updateWeatherData();
                if (updateSuccessful) {
                    int i = 0;
                    while (i < 7) {
                        this.forecasts[i] = new Forecast(i, this.weatherData.getMetadata(), this.weatherData.getUnits(), this.weatherData.getDataDay());
                        ++i;
                    }
                    for (Channel channel : this.getThing().getChannels()) {
                        this.updateChannel(channel.getUID().getId());
                    }
                }
            }
            catch (Exception e) {
                this.logger.warn("Exception occurred during weather update: {}", (Object)e.getMessage(), (Object)e);
            }
        };
        int period = config.refresh != null ? config.refresh : 240;
        this.refreshJob = this.scheduler.scheduleWithFixedDelay(runnable, 0L, period, TimeUnit.MINUTES);
    }

    public void dispose() {
        this.logger.debug("Disposing meteoblue handler.");
        if (this.refreshJob != null && !this.refreshJob.isCancelled()) {
            this.refreshJob.cancel(true);
            this.refreshJob = null;
        }
    }

    private void updateChannel(String channelId) {
        Channel channel = this.getThing().getChannel(channelId);
        if (channel == null || !this.isLinked(channelId)) {
            this.logger.trace("Channel '{}' was null or not linked! Not updated.", (Object)channelId);
            return;
        }
        String[] channelParts = channelId.split("#");
        String forecastDay = channelParts[0];
        String datapointName = channelParts[1];
        if (channelParts.length != 2) {
            this.logger.debug("Skipped invalid channelId '{}'", (Object)channelId);
            return;
        }
        this.logger.debug("Updating channel '{}'", (Object)channelId);
        Forecast forecast = this.getForecast(forecastDay);
        if (forecast == null) {
            this.logger.debug("No forecast found for '{}'. Not updating.", (Object)forecastDay);
            return;
        }
        Object datapoint = forecast.getDatapoint(datapointName);
        this.logger.debug("Value for datapoint '{}' is '{}'", (Object)datapointName, datapoint);
        if (datapoint == null) {
            this.logger.debug("Couldn't get datapoint '{}' for '{}'. Not updating.", (Object)datapointName, (Object)forecastDay);
            return;
        }
        DateTimeType state = null;
        if (datapoint instanceof Calendar) {
            state = new DateTimeType((Calendar)datapoint);
        } else if (datapoint instanceof Integer) {
            state = this.getStateForType(channel.getAcceptedItemType(), (Integer)datapoint);
        } else if (datapoint instanceof Number) {
            BigDecimal decimalValue = new BigDecimal(datapoint.toString()).setScale(2, RoundingMode.HALF_UP);
            state = this.getStateForType(channel.getAcceptedItemType(), decimalValue);
        } else if (datapoint instanceof String) {
            state = new StringType(datapoint.toString());
        } else if (datapoint instanceof BufferedImage) {
            ImageItem item = new ImageItem("rain area");
            state = new RawType(this.renderImage((BufferedImage)datapoint), "image/png");
            item.setState((State)state);
        } else {
            this.logger.debug("Unsupported value type {}", (Object)datapoint.getClass().getSimpleName());
        }
        if (state != null) {
            this.logger.trace("Updating channel with state value {}. (object type {})", (Object)state, (Object)datapoint.getClass().getSimpleName());
            this.updateState(channelId, (State)state);
        }
    }

    private State getStateForType(String type, Integer value) {
        return this.getStateForType(type, new BigDecimal(value));
    }

    private State getStateForType(String type, BigDecimal value) {
        DecimalType state = new DecimalType(value);
        if (type.equals("Number:Temperature")) {
            state = new QuantityType((Number)value, SIUnits.CELSIUS);
        } else if (type.equals("Number:Length")) {
            state = new QuantityType((Number)value, MetricPrefix.MILLI((Unit)SIUnits.METRE));
        } else if (type.equals("Number:Pressure")) {
            state = new QuantityType((Number)value, MetricPrefix.HECTO((Unit)SIUnits.PASCAL));
        } else if (type.equals("Number:Speed")) {
            state = new QuantityType((Number)value, SIUnits.KILOMETRE_PER_HOUR);
        }
        return state;
    }

    private boolean updateWeatherData() {
        if (this.bridge == null) {
            this.logger.debug("Unable to update weather data. Bridge missing.");
            return false;
        }
        MeteoBlueBridgeHandler handler = (MeteoBlueBridgeHandler)this.bridge.getHandler();
        if (handler == null) {
            this.logger.debug("Unable to update weather data. Handler missing.");
            return false;
        }
        String apiKey = handler.getApiKey();
        this.logger.debug("Updating weather data...");
        MeteoBlueConfiguration config = (MeteoBlueConfiguration)this.getConfigAs(MeteoBlueConfiguration.class);
        config.parseLocation();
        String serviceType = config.serviceType;
        if (serviceType.equals("Commercial")) {
            this.logger.debug("Fetching weather data using Commercial API.");
        } else {
            this.logger.debug("Fetching weather data using NonCommercial API.");
        }
        String url = MeteoBlueConfiguration.getURL(serviceType);
        url = url.replace("#API_KEY#", apiKey);
        url = url.replace("#LATITUDE#", String.valueOf(config.latitude)).replace("#LONGITUDE#", String.valueOf(config.longitude));
        StringBuilder builder = new StringBuilder();
        if (config.altitude != null) {
            builder.append("&asl=" + config.altitude);
        }
        if (StringUtils.isNotBlank((String)config.timeZone)) {
            builder.append("&tz=" + config.timeZone);
        }
        url = url.replace("#FORMAT_PARAMS#", builder.toString());
        this.logger.trace("Using URL '{}'", (Object)url);
        String httpResponse = this.getWeatherData(url);
        if (httpResponse == null) {
            return false;
        }
        JsonData jsonResult = this.translateJson(httpResponse, serviceType);
        this.logger.trace("json object: {}", (Object)jsonResult);
        if (jsonResult == null) {
            this.logger.warn("No data was received from the weather service");
            return false;
        }
        String errorMessage = jsonResult.getErrorMessage();
        if (errorMessage != null) {
            if (errorMessage.equals("MB_REQUEST::DISPATCH: Invalid api key")) {
                this.updateStatus(ThingStatus.OFFLINE, ThingStatusDetail.CONFIGURATION_ERROR, "Invalid API Key");
            } else if (errorMessage.equals("MB_REQUEST::DISPATCH: This datafeed is not authorized for your api key")) {
                this.updateStatus(ThingStatus.OFFLINE, ThingStatusDetail.CONFIGURATION_ERROR, "API Key not authorized for this datafeed");
            } else {
                this.logger.warn("Failed to retrieve weather data due to unexpected error. Error message: {}", (Object)errorMessage);
                this.updateStatus(ThingStatus.OFFLINE, ThingStatusDetail.COMMUNICATION_ERROR, errorMessage);
            }
            return false;
        }
        this.weatherData = jsonResult;
        this.updateStatus(ThingStatus.ONLINE);
        return true;
    }

    private String getWeatherData(String url) {
        try {
            String httpResponse = HttpUtil.executeUrl((String)"GET", (String)url, (int)30000);
            this.logger.trace("http response: {}", (Object)httpResponse);
            return httpResponse;
        }
        catch (IOException e) {
            this.logger.debug("I/O Exception occurred while retrieving weather data.", (Throwable)e);
            this.updateStatus(ThingStatus.OFFLINE, ThingStatusDetail.COMMUNICATION_ERROR, "I/O Exception occurred while retrieving weather data.");
            return null;
        }
    }

    private JsonData translateJson(String stringData, String serviceType) {
        return (JsonData)this.gson.fromJson(stringData, JsonData.class);
    }

    private Forecast getForecast(String which) {
        switch (which) {
            case "forecastToday": {
                return this.forecasts[0];
            }
            case "forecastTomorrow": {
                return this.forecasts[1];
            }
            case "forecastDay2": {
                return this.forecasts[2];
            }
            case "forecastDay3": {
                return this.forecasts[3];
            }
            case "forecastDay4": {
                return this.forecasts[4];
            }
            case "forecastDay5": {
                return this.forecasts[5];
            }
            case "forecastDay6": {
                return this.forecasts[6];
            }
        }
        return null;
    }

    private byte[] renderImage(BufferedImage image) {
        byte[] data = null;
        try {
            try {
                ByteArrayOutputStream out = new ByteArrayOutputStream();
                ImageIO.write((RenderedImage)image, "png", out);
                out.flush();
                data = out.toByteArray();
                out.close();
            }
            catch (IOException ioe) {
                this.logger.debug("I/O exception occurred converting image data", (Throwable)ioe);
            }
        }
        catch (Throwable throwable) {}
        return data;
    }
}

