/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.smarthome.core.thing.internal.firmware;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.eclipse.jdt.annotation.NonNullByDefault;
import org.eclipse.jdt.annotation.Nullable;
import org.eclipse.smarthome.config.core.validation.ConfigDescriptionValidator;
import org.eclipse.smarthome.config.core.validation.ConfigValidationException;
import org.eclipse.smarthome.core.common.SafeCaller;
import org.eclipse.smarthome.core.common.ThreadPoolManager;
import org.eclipse.smarthome.core.events.Event;
import org.eclipse.smarthome.core.events.EventFilter;
import org.eclipse.smarthome.core.events.EventPublisher;
import org.eclipse.smarthome.core.events.EventSubscriber;
import org.eclipse.smarthome.core.i18n.LocaleProvider;
import org.eclipse.smarthome.core.i18n.TranslationProvider;
import org.eclipse.smarthome.core.thing.Thing;
import org.eclipse.smarthome.core.thing.ThingStatus;
import org.eclipse.smarthome.core.thing.ThingUID;
import org.eclipse.smarthome.core.thing.binding.firmware.Firmware;
import org.eclipse.smarthome.core.thing.binding.firmware.FirmwareUpdateBackgroundTransferHandler;
import org.eclipse.smarthome.core.thing.binding.firmware.FirmwareUpdateHandler;
import org.eclipse.smarthome.core.thing.events.ThingStatusInfoChangedEvent;
import org.eclipse.smarthome.core.thing.firmware.FirmwareEventFactory;
import org.eclipse.smarthome.core.thing.firmware.FirmwareRegistry;
import org.eclipse.smarthome.core.thing.firmware.FirmwareStatus;
import org.eclipse.smarthome.core.thing.firmware.FirmwareStatusInfo;
import org.eclipse.smarthome.core.thing.firmware.FirmwareUpdateService;
import org.eclipse.smarthome.core.thing.internal.firmware.ParameterChecks;
import org.eclipse.smarthome.core.thing.internal.firmware.ProgressCallbackImpl;
import org.eclipse.smarthome.core.util.BundleResolver;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Modified;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.component.annotations.ReferencePolicy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Component(immediate=true, service={EventSubscriber.class, FirmwareUpdateService.class})
@NonNullByDefault
public final class FirmwareUpdateServiceImpl
implements FirmwareUpdateService,
EventSubscriber {
    private static final String THREAD_POOL_NAME = FirmwareUpdateServiceImpl.class.getSimpleName();
    private static final Set<String> SUPPORTED_TIME_UNITS = Collections.unmodifiableSet(Stream.of(TimeUnit.SECONDS.name(), TimeUnit.MINUTES.name(), TimeUnit.HOURS.name(), TimeUnit.DAYS.name()).collect(Collectors.toSet()));
    protected static final String PERIOD_CONFIG_KEY = "period";
    protected static final String DELAY_CONFIG_KEY = "delay";
    protected static final String TIME_UNIT_CONFIG_KEY = "timeUnit";
    private static final String CONFIG_DESC_URI_KEY = "system:firmware-status-info-job";
    private final Logger logger = LoggerFactory.getLogger(FirmwareUpdateServiceImpl.class);
    private int firmwareStatusInfoJobPeriod = 3600;
    private int firmwareStatusInfoJobDelay = 3600;
    private TimeUnit firmwareStatusInfoJobTimeUnit = TimeUnit.SECONDS;
    private @Nullable ScheduledFuture<?> firmwareStatusInfoJob;
    protected int timeout = 1800000;
    private final Set<String> subscribedEventTypes = Collections.singleton(ThingStatusInfoChangedEvent.TYPE);
    private final Map<ThingUID, FirmwareStatusInfo> firmwareStatusInfoMap = new ConcurrentHashMap<ThingUID, FirmwareStatusInfo>();
    private final Map<ThingUID, ProgressCallbackImpl> progressCallbackMap = new ConcurrentHashMap<ThingUID, ProgressCallbackImpl>();
    private final List<FirmwareUpdateHandler> firmwareUpdateHandlers = new CopyOnWriteArrayList<FirmwareUpdateHandler>();
    @NonNullByDefault(value={})
    private FirmwareRegistry firmwareRegistry;
    @NonNullByDefault(value={})
    private EventPublisher eventPublisher;
    @NonNullByDefault(value={})
    private TranslationProvider i18nProvider;
    @NonNullByDefault(value={})
    private LocaleProvider localeProvider;
    @NonNullByDefault(value={})
    private SafeCaller safeCaller;
    @NonNullByDefault(value={})
    private ConfigDescriptionValidator configDescriptionValidator;
    @NonNullByDefault(value={})
    private BundleResolver bundleResolver;
    private final Runnable firmwareStatusRunnable = new Runnable(){

        @Override
        public void run() {
            FirmwareUpdateServiceImpl.this.logger.debug("Running firmware status check.");
            for (FirmwareUpdateHandler firmwareUpdateHandler : FirmwareUpdateServiceImpl.this.firmwareUpdateHandlers) {
                try {
                    FirmwareUpdateServiceImpl.this.logger.debug("Executing firmware status check for thing with UID {}.", (Object)firmwareUpdateHandler.getThing().getUID());
                    Firmware latestFirmware = FirmwareUpdateServiceImpl.this.getLatestSuitableFirmware(firmwareUpdateHandler.getThing());
                    FirmwareStatusInfo newFirmwareStatusInfo = FirmwareUpdateServiceImpl.this.getFirmwareStatusInfo(firmwareUpdateHandler, latestFirmware);
                    FirmwareUpdateServiceImpl.this.processFirmwareStatusInfo(firmwareUpdateHandler, newFirmwareStatusInfo, latestFirmware);
                }
                catch (Exception e) {
                    FirmwareUpdateServiceImpl.this.logger.debug("Exception occurred during firmware status check.", (Throwable)e);
                }
            }
        }
    };

    @Activate
    protected void activate(Map<String, Object> config) {
        this.modified(config);
    }

    @Modified
    protected synchronized void modified(Map<String, Object> config) {
        this.logger.debug("Modifying the configuration of the firmware update service.");
        if (!this.isValid(config)) {
            return;
        }
        this.cancelFirmwareUpdateStatusInfoJob();
        this.firmwareStatusInfoJobPeriod = config.containsKey(PERIOD_CONFIG_KEY) ? (Integer)config.get(PERIOD_CONFIG_KEY) : this.firmwareStatusInfoJobPeriod;
        this.firmwareStatusInfoJobDelay = config.containsKey(DELAY_CONFIG_KEY) ? (Integer)config.get(DELAY_CONFIG_KEY) : this.firmwareStatusInfoJobDelay;
        TimeUnit timeUnit = this.firmwareStatusInfoJobTimeUnit = config.containsKey(TIME_UNIT_CONFIG_KEY) ? TimeUnit.valueOf((String)config.get(TIME_UNIT_CONFIG_KEY)) : this.firmwareStatusInfoJobTimeUnit;
        if (!this.firmwareUpdateHandlers.isEmpty()) {
            this.createFirmwareUpdateStatusInfoJob();
        }
    }

    @Deactivate
    protected void deactivate() {
        this.cancelFirmwareUpdateStatusInfoJob();
        this.firmwareStatusInfoMap.clear();
        this.progressCallbackMap.clear();
    }

    @Override
    public @Nullable FirmwareStatusInfo getFirmwareStatusInfo(ThingUID thingUID) {
        ParameterChecks.checkNotNull((Object)thingUID, "Thing UID");
        FirmwareUpdateHandler firmwareUpdateHandler = this.getFirmwareUpdateHandler(thingUID);
        if (firmwareUpdateHandler == null) {
            this.logger.trace("No firmware update handler available for thing with UID {}.", (Object)thingUID);
            return null;
        }
        Firmware latestFirmware = this.getLatestSuitableFirmware(firmwareUpdateHandler.getThing());
        FirmwareStatusInfo firmwareStatusInfo = this.getFirmwareStatusInfo(firmwareUpdateHandler, latestFirmware);
        this.processFirmwareStatusInfo(firmwareUpdateHandler, firmwareStatusInfo, latestFirmware);
        return firmwareStatusInfo;
    }

    @Override
    public void updateFirmware(ThingUID thingUID, String firmwareVersion, @Nullable Locale locale) {
        ParameterChecks.checkNotNull((Object)thingUID, "Thing UID");
        ParameterChecks.checkNotNullOrEmpty(firmwareVersion, "Firmware version");
        FirmwareUpdateHandler firmwareUpdateHandler = this.getFirmwareUpdateHandler(thingUID);
        if (firmwareUpdateHandler == null) {
            throw new IllegalArgumentException(String.format("There is no firmware update handler for thing with UID %s.", new Object[]{thingUID}));
        }
        Thing thing = firmwareUpdateHandler.getThing();
        Firmware firmware = this.getFirmware(thing, firmwareVersion);
        this.validateFirmwareUpdateConditions(firmwareUpdateHandler, firmware);
        Locale currentLocale = locale != null ? locale : this.localeProvider.getLocale();
        ProgressCallbackImpl progressCallback = new ProgressCallbackImpl(firmwareUpdateHandler, this.eventPublisher, this.i18nProvider, this.bundleResolver, thingUID, firmware, currentLocale);
        this.progressCallbackMap.put(thingUID, progressCallback);
        this.logger.debug("Starting firmware update for thing with UID {} and firmware {}", (Object)thingUID, (Object)firmware);
        ((FirmwareUpdateHandler)this.safeCaller.create((Object)firmwareUpdateHandler, FirmwareUpdateHandler.class).withTimeout((long)this.timeout).withAsync().onTimeout(() -> {
            this.logger.error("Timeout occurred for firmware update of thing with UID {} and firmware {}.", (Object)thingUID, (Object)firmware);
            progressCallback.failedInternal("timeout-error");
        }).onException(e -> {
            this.logger.error("Unexpected exception occurred for firmware update of thing with UID {} and firmware {}.", new Object[]{thingUID, firmware, e.getCause()});
            progressCallback.failedInternal("unexpected-handler-error");
        }).build()).updateFirmware(firmware, progressCallback);
    }

    @Override
    public void cancelFirmwareUpdate(ThingUID thingUID) {
        ParameterChecks.checkNotNull((Object)thingUID, "Thing UID");
        FirmwareUpdateHandler firmwareUpdateHandler = this.getFirmwareUpdateHandler(thingUID);
        if (firmwareUpdateHandler == null) {
            throw new IllegalArgumentException(String.format("There is no firmware update handler for thing with UID %s.", new Object[]{thingUID}));
        }
        ProgressCallbackImpl progressCallback = this.getProgressCallback(thingUID);
        this.logger.debug("Cancelling firmware update for thing with UID {}.", (Object)thingUID);
        ((FirmwareUpdateHandler)this.safeCaller.create((Object)firmwareUpdateHandler, FirmwareUpdateHandler.class).withTimeout((long)this.timeout).withAsync().onTimeout(() -> {
            this.logger.error("Timeout occurred while cancelling firmware update of thing with UID {}.", (Object)thingUID);
            progressCallback.failedInternal("timeout-error-during-cancel");
        }).onException(e -> {
            this.logger.error("Unexpected exception occurred while cancelling firmware update of thing with UID {}.", (Object)thingUID, (Object)e.getCause());
            progressCallback.failedInternal("unexpected-handler-error-during-cancel");
        }).withIdentifier(new Object()).build()).cancel();
    }

    public Set<String> getSubscribedEventTypes() {
        return this.subscribedEventTypes;
    }

    public @Nullable EventFilter getEventFilter() {
        return null;
    }

    public void receive(Event event) {
        if (event instanceof ThingStatusInfoChangedEvent) {
            ThingStatusInfoChangedEvent changedEvent = (ThingStatusInfoChangedEvent)event;
            if (changedEvent.getStatusInfo().getStatus() != ThingStatus.ONLINE) {
                return;
            }
            ThingUID thingUID = changedEvent.getThingUID();
            FirmwareUpdateHandler firmwareUpdateHandler = this.getFirmwareUpdateHandler(thingUID);
            if (firmwareUpdateHandler != null && !this.firmwareStatusInfoMap.containsKey((Object)thingUID)) {
                this.initializeFirmwareStatus(firmwareUpdateHandler);
            }
        }
    }

    protected ProgressCallbackImpl getProgressCallback(ThingUID thingUID) {
        if (!this.progressCallbackMap.containsKey((Object)thingUID)) {
            throw new IllegalStateException(String.format("No ProgressCallback available for thing with UID %s.", new Object[]{thingUID}));
        }
        return this.progressCallbackMap.get((Object)thingUID);
    }

    private @Nullable Firmware getLatestSuitableFirmware(Thing thing) {
        Optional<Firmware> first;
        Collection<Firmware> firmwares = this.firmwareRegistry.getFirmwares(thing);
        if (firmwares != null && (first = firmwares.stream().findFirst()).isPresent()) {
            return first.get();
        }
        return null;
    }

    private FirmwareStatusInfo getFirmwareStatusInfo(FirmwareUpdateHandler firmwareUpdateHandler, @Nullable Firmware latestFirmware) {
        String thingFirmwareVersion = this.getThingFirmwareVersion(firmwareUpdateHandler);
        ThingUID thingUID = firmwareUpdateHandler.getThing().getUID();
        if (latestFirmware == null || thingFirmwareVersion == null) {
            return FirmwareStatusInfo.createUnknownInfo(thingUID);
        }
        if (latestFirmware.isSuccessorVersion(thingFirmwareVersion)) {
            if (firmwareUpdateHandler.isUpdateExecutable()) {
                return FirmwareStatusInfo.createUpdateExecutableInfo(thingUID, latestFirmware.getVersion());
            }
            return FirmwareStatusInfo.createUpdateAvailableInfo(thingUID);
        }
        return FirmwareStatusInfo.createUpToDateInfo(thingUID);
    }

    private synchronized void processFirmwareStatusInfo(FirmwareUpdateHandler firmwareUpdateHandler, FirmwareStatusInfo newFirmwareStatusInfo, @Nullable Firmware latestFirmware) {
        FirmwareStatusInfo previousFirmwareStatusInfo = this.firmwareStatusInfoMap.put(newFirmwareStatusInfo.getThingUID(), newFirmwareStatusInfo);
        if (previousFirmwareStatusInfo == null || !previousFirmwareStatusInfo.equals(newFirmwareStatusInfo)) {
            this.eventPublisher.post((Event)FirmwareEventFactory.createFirmwareStatusInfoEvent(newFirmwareStatusInfo));
            if (newFirmwareStatusInfo.getFirmwareStatus() == FirmwareStatus.UPDATE_AVAILABLE && firmwareUpdateHandler instanceof FirmwareUpdateBackgroundTransferHandler && !firmwareUpdateHandler.isUpdateExecutable() && latestFirmware != null) {
                this.transferLatestFirmware((FirmwareUpdateBackgroundTransferHandler)firmwareUpdateHandler, latestFirmware, previousFirmwareStatusInfo);
            }
        }
    }

    private void transferLatestFirmware(final FirmwareUpdateBackgroundTransferHandler fubtHandler, final Firmware latestFirmware, final FirmwareStatusInfo previousFirmwareStatusInfo) {
        FirmwareUpdateServiceImpl.getPool().submit(new Runnable(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void run() {
                try {
                    fubtHandler.transferFirmware(latestFirmware);
                }
                catch (Exception e) {
                    FirmwareUpdateServiceImpl.this.logger.error("Exception occurred during background firmware transfer.", (Throwable)e);
                    2 var2_2 = this;
                    synchronized (var2_2) {
                        if (previousFirmwareStatusInfo == null) {
                            FirmwareUpdateServiceImpl.this.firmwareStatusInfoMap.remove((Object)fubtHandler.getThing().getUID());
                        } else {
                            FirmwareUpdateServiceImpl.this.firmwareStatusInfoMap.put(fubtHandler.getThing().getUID(), previousFirmwareStatusInfo);
                        }
                    }
                }
            }
        });
    }

    private void validateFirmwareUpdateConditions(FirmwareUpdateHandler firmwareUpdateHandler, Firmware firmware) {
        if (!firmwareUpdateHandler.isUpdateExecutable()) {
            throw new IllegalStateException(String.format("The firmware update of thing with UID %s is not executable.", new Object[]{firmwareUpdateHandler.getThing().getUID()}));
        }
        this.validateFirmwareSuitability(firmware, firmwareUpdateHandler);
    }

    private void validateFirmwareSuitability(Firmware firmware, FirmwareUpdateHandler firmwareUpdateHandler) {
        Thing thing = firmwareUpdateHandler.getThing();
        if (!firmware.isSuitableFor(thing)) {
            throw new IllegalArgumentException(String.format("Firmware %s is not suitable for thing with UID %s.", new Object[]{firmware, thing.getUID()}));
        }
    }

    private Firmware getFirmware(Thing thing, String firmwareVersion) {
        Firmware firmware = this.firmwareRegistry.getFirmware(thing, firmwareVersion);
        if (firmware == null) {
            throw new IllegalArgumentException(String.format("Firmware with version %s for thing with UID %s was not found.", new Object[]{firmwareVersion, thing.getUID()}));
        }
        return firmware;
    }

    private @Nullable FirmwareUpdateHandler getFirmwareUpdateHandler(ThingUID thingUID) {
        for (FirmwareUpdateHandler firmwareUpdateHandler : this.firmwareUpdateHandlers) {
            if (!thingUID.equals((Object)firmwareUpdateHandler.getThing().getUID())) continue;
            return firmwareUpdateHandler;
        }
        return null;
    }

    private @Nullable String getThingFirmwareVersion(FirmwareUpdateHandler firmwareUpdateHandler) {
        return firmwareUpdateHandler.getThing().getProperties().get("firmwareVersion");
    }

    private void createFirmwareUpdateStatusInfoJob() {
        if (this.firmwareStatusInfoJob == null || this.firmwareStatusInfoJob.isCancelled()) {
            this.logger.debug("Creating firmware status info job. [delay:{}, period:{}, time unit: {}]", new Object[]{this.firmwareStatusInfoJobDelay, this.firmwareStatusInfoJobPeriod, this.firmwareStatusInfoJobTimeUnit});
            this.firmwareStatusInfoJob = FirmwareUpdateServiceImpl.getPool().scheduleWithFixedDelay(this.firmwareStatusRunnable, this.firmwareStatusInfoJobDelay, this.firmwareStatusInfoJobPeriod, this.firmwareStatusInfoJobTimeUnit);
        }
    }

    private void cancelFirmwareUpdateStatusInfoJob() {
        if (this.firmwareStatusInfoJob != null && !this.firmwareStatusInfoJob.isCancelled()) {
            this.logger.debug("Cancelling firmware status info job.");
            this.firmwareStatusInfoJob.cancel(true);
            this.firmwareStatusInfoJob = null;
        }
    }

    private boolean isValid(Map<String, Object> config) {
        if (!SUPPORTED_TIME_UNITS.contains(config.get(TIME_UNIT_CONFIG_KEY))) {
            this.logger.debug("Given time unit {} is not supported. Will keep current configuration.", config.get(TIME_UNIT_CONFIG_KEY));
            return false;
        }
        try {
            this.configDescriptionValidator.validate(config, new URI(CONFIG_DESC_URI_KEY));
        }
        catch (URISyntaxException | ConfigValidationException e) {
            this.logger.debug("Validation of new configuration values failed. Will keep current configuration.", e);
            return false;
        }
        return true;
    }

    private void initializeFirmwareStatus(final FirmwareUpdateHandler firmwareUpdateHandler) {
        FirmwareUpdateServiceImpl.getPool().submit(new Runnable(){

            @Override
            public void run() {
                ThingUID thingUID = firmwareUpdateHandler.getThing().getUID();
                FirmwareStatusInfo info = FirmwareUpdateServiceImpl.this.getFirmwareStatusInfo(thingUID);
                FirmwareUpdateServiceImpl.this.logger.debug("Firmware status {} for thing {} initialized.", (Object)info.getFirmwareStatus(), (Object)thingUID);
                FirmwareUpdateServiceImpl.this.firmwareStatusInfoMap.put(thingUID, info);
            }
        });
    }

    private static ScheduledExecutorService getPool() {
        return ThreadPoolManager.getScheduledPool((String)THREAD_POOL_NAME);
    }

    public int getFirmwareStatusInfoJobPeriod() {
        return this.firmwareStatusInfoJobPeriod;
    }

    public int getFirmwareStatusInfoJobDelay() {
        return this.firmwareStatusInfoJobDelay;
    }

    public TimeUnit getFirmwareStatusInfoJobTimeUnit() {
        return this.firmwareStatusInfoJobTimeUnit;
    }

    @Reference(cardinality=ReferenceCardinality.MULTIPLE, policy=ReferencePolicy.DYNAMIC)
    protected synchronized void addFirmwareUpdateHandler(FirmwareUpdateHandler firmwareUpdateHandler) {
        if (this.firmwareUpdateHandlers.isEmpty()) {
            this.createFirmwareUpdateStatusInfoJob();
        }
        this.firmwareUpdateHandlers.add(firmwareUpdateHandler);
    }

    protected synchronized void removeFirmwareUpdateHandler(FirmwareUpdateHandler firmwareUpdateHandler) {
        this.firmwareStatusInfoMap.remove((Object)firmwareUpdateHandler.getThing().getUID());
        this.firmwareUpdateHandlers.remove(firmwareUpdateHandler);
        if (this.firmwareUpdateHandlers.isEmpty()) {
            this.cancelFirmwareUpdateStatusInfoJob();
        }
        this.progressCallbackMap.remove((Object)firmwareUpdateHandler.getThing().getUID());
    }

    @Reference
    protected void setFirmwareRegistry(FirmwareRegistry firmwareRegistry) {
        this.firmwareRegistry = firmwareRegistry;
    }

    protected void unsetFirmwareRegistry(FirmwareRegistry firmwareRegistry) {
        this.firmwareRegistry = null;
    }

    @Reference
    protected void setEventPublisher(EventPublisher eventPublisher) {
        this.eventPublisher = eventPublisher;
    }

    protected void unsetEventPublisher(EventPublisher eventPublisher) {
        this.eventPublisher = null;
    }

    @Reference
    protected void setTranslationProvider(TranslationProvider i18nProvider) {
        this.i18nProvider = i18nProvider;
    }

    protected void unsetTranslationProvider(TranslationProvider i18nProvider) {
        this.i18nProvider = null;
    }

    @Reference
    protected void setLocaleProvider(LocaleProvider localeProvider) {
        this.localeProvider = localeProvider;
    }

    protected void unsetLocaleProvider(LocaleProvider localeProvider) {
        this.localeProvider = null;
    }

    @Reference
    protected void setSafeCaller(SafeCaller safeCaller) {
        this.safeCaller = safeCaller;
    }

    protected void unsetSafeCaller(SafeCaller safeCaller) {
        this.safeCaller = null;
    }

    @Reference
    protected void setConfigDescriptionValidator(ConfigDescriptionValidator configDescriptionValidator) {
        this.configDescriptionValidator = configDescriptionValidator;
    }

    protected void unsetConfigDescriptionValidator(ConfigDescriptionValidator configDescriptionValidator) {
        this.configDescriptionValidator = null;
    }

    @Reference
    protected void setBundleResolver(BundleResolver bundleResolver) {
        this.bundleResolver = bundleResolver;
    }

    protected void unsetBundleResolver(BundleResolver bundleResolver) {
        this.bundleResolver = bundleResolver;
    }
}

