"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.GDBDebugSession = exports.base64ToHex = exports.hexToBase64 = void 0;
/*********************************************************************
 * Copyright (c) 2018 QNX Software Systems and others
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *********************************************************************/
const os = require("os");
const path = require("path");
const debugadapter_1 = require("@vscode/debugadapter");
const GDBBackend_1 = require("./GDBBackend");
const mi = require("./mi");
const data_1 = require("./mi/data");
const stoppedEvent_1 = require("./stoppedEvent");
const mi_1 = require("./mi");
class ThreadWithStatus {
    constructor(id, name, running) {
        this.id = id;
        this.name = name;
        this.running = running;
    }
}
// Allow a single number for ignore count or the form '> [number]'
const ignoreCountRegex = /\s|>/g;
const arrayRegex = /.*\[[\d]+\].*/;
const arrayChildRegex = /[\d]+/;
function hexToBase64(hex) {
    // The buffer will ignore incomplete bytes (unpaired digits), so we need to catch that early
    if (hex.length % 2 !== 0) {
        throw new Error('Received memory with incomplete bytes.');
    }
    const base64 = Buffer.from(hex, 'hex').toString('base64');
    // If the hex input includes characters that are not hex digits, Buffer.from() will return an empty buffer, and the base64 string will be empty.
    if (base64.length === 0 && hex.length !== 0) {
        throw new Error('Received ill-formed hex input: ' + hex);
    }
    return base64;
}
exports.hexToBase64 = hexToBase64;
function base64ToHex(base64) {
    const buffer = Buffer.from(base64, 'base64');
    // The caller likely passed in a value that left dangling bits that couldn't be assigned to a full byte and so
    // were ignored by Buffer. We can't be sure what the client thought they wanted to do with those extra bits, so fail here.
    if (buffer.length === 0 || !buffer.toString('base64').startsWith(base64)) {
        throw new Error('Received ill-formed base64 input: ' + base64);
    }
    return buffer.toString('hex');
}
exports.base64ToHex = base64ToHex;
class GDBDebugSession extends debugadapter_1.LoggingDebugSession {
    constructor() {
        super();
        this.gdb = this.createBackend();
        this.isAttach = false;
        // isRunning === true means there are no threads stopped.
        this.isRunning = false;
        this.supportsRunInTerminalRequest = false;
        this.supportsGdbConsole = false;
        this.frameHandles = new debugadapter_1.Handles();
        this.variableHandles = new debugadapter_1.Handles();
        this.functionBreakpoints = [];
        this.logPointMessages = {};
        this.threads = [];
        // the thread id that we were waiting for
        this.waitPausedThreadId = 0;
        // set to true if the target was interrupted where inteneded, and should
        // therefore be resumed after breakpoints are inserted.
        this.waitPausedNeeded = false;
        this.isInitialized = false;
        // Register view
        // Assume that the register name are unchanging over time, and the same across all threadsf
        this.registerMap = new Map();
        this.registerMapReverse = new Map();
        this.logger = debugadapter_1.logger;
    }
    createBackend() {
        return new GDBBackend_1.GDBBackend();
    }
    /**
     * Handle requests not defined in the debug adapter protocol.
     */
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    customRequest(command, response, args) {
        if (command === 'cdt-gdb-adapter/Memory') {
            this.memoryRequest(response, args);
            // This custom request exists to allow tests in this repository to run arbitrary commands
            // Use at your own risk!
        }
        else if (command === 'cdt-gdb-tests/executeCommand') {
            this.gdb
                .sendCommand(args.command)
                .then(() => {
                response.body = 'Ok';
                this.sendResponse(response);
            })
                .catch((e) => {
                const message = e instanceof Error
                    ? e.message
                    : `Encountered a problem executing ${args.command}`;
                this.sendErrorResponse(response, 1, message);
            });
        }
        else {
            return super.customRequest(command, response, args);
        }
    }
    initializeRequest(response, args) {
        this.supportsRunInTerminalRequest =
            args.supportsRunInTerminalRequest === true;
        this.supportsGdbConsole =
            os.platform() === 'linux' && this.supportsRunInTerminalRequest;
        response.body = response.body || {};
        response.body.supportsConfigurationDoneRequest = true;
        response.body.supportsSetVariable = true;
        response.body.supportsConditionalBreakpoints = true;
        response.body.supportsHitConditionalBreakpoints = true;
        response.body.supportsLogPoints = true;
        response.body.supportsFunctionBreakpoints = true;
        // response.body.supportsSetExpression = true;
        response.body.supportsDisassembleRequest = true;
        response.body.supportsReadMemoryRequest = true;
        response.body.supportsWriteMemoryRequest = true;
        this.sendResponse(response);
    }
    attachRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                debugadapter_1.logger.setup(args.verbose ? debugadapter_1.Logger.LogLevel.Verbose : debugadapter_1.Logger.LogLevel.Warn, args.logFile || false);
                this.gdb.on('consoleStreamOutput', (output, category) => {
                    this.sendEvent(new debugadapter_1.OutputEvent(output, category));
                });
                this.gdb.on('execAsync', (resultClass, resultData) => this.handleGDBAsync(resultClass, resultData));
                this.gdb.on('notifyAsync', (resultClass, resultData) => this.handleGDBNotify(resultClass, resultData));
                yield this.spawn(args);
                yield this.gdb.sendFileExecAndSymbols(args.program);
                yield this.gdb.sendEnablePrettyPrint();
                yield mi.sendTargetAttachRequest(this.gdb, { pid: args.processId });
                this.sendEvent(new debugadapter_1.OutputEvent(`attached to process ${args.processId}`));
                yield this.gdb.sendCommands(args.initCommands);
                this.sendEvent(new debugadapter_1.InitializedEvent());
                this.sendResponse(response);
                this.isInitialized = true;
            }
            catch (err) {
                this.sendErrorResponse(response, 1, err instanceof Error ? err.message : String(err));
            }
        });
    }
    launchRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                debugadapter_1.logger.setup(args.verbose ? debugadapter_1.Logger.LogLevel.Verbose : debugadapter_1.Logger.LogLevel.Warn, args.logFile || false);
                this.gdb.on('consoleStreamOutput', (output, category) => {
                    this.sendEvent(new debugadapter_1.OutputEvent(output, category));
                });
                this.gdb.on('execAsync', (resultClass, resultData) => this.handleGDBAsync(resultClass, resultData));
                this.gdb.on('notifyAsync', (resultClass, resultData) => this.handleGDBNotify(resultClass, resultData));
                yield this.spawn(args);
                yield this.gdb.sendFileExecAndSymbols(args.program);
                yield this.gdb.sendEnablePrettyPrint();
                if (args.initCommands) {
                    for (const command of args.initCommands) {
                        yield this.gdb.sendCommand(command);
                    }
                }
                if (args.arguments) {
                    yield mi.sendExecArguments(this.gdb, {
                        arguments: args.arguments,
                    });
                }
                this.sendEvent(new debugadapter_1.InitializedEvent());
                this.sendResponse(response);
                this.isInitialized = true;
            }
            catch (err) {
                this.sendErrorResponse(response, 1, err instanceof Error ? err.message : String(err));
            }
        });
    }
    spawn(args) {
        return __awaiter(this, void 0, void 0, function* () {
            if (args.openGdbConsole) {
                if (!this.supportsGdbConsole) {
                    debugadapter_1.logger.warn('cdt-gdb-adapter: openGdbConsole is not supported on this platform');
                }
                else if (!(yield this.gdb.supportsNewUi(args.gdb))) {
                    debugadapter_1.logger.warn(`cdt-gdb-adapter: new-ui command not detected (${args.gdb || 'gdb'})`);
                }
                else {
                    debugadapter_1.logger.verbose('cdt-gdb-adapter: spawning gdb console in client terminal');
                    return this.spawnInClientTerminal(args);
                }
            }
            return this.gdb.spawn(args);
        });
    }
    spawnInClientTerminal(args) {
        return __awaiter(this, void 0, void 0, function* () {
            return this.gdb.spawnInClientTerminal(args, (command) => __awaiter(this, void 0, void 0, function* () {
                const response = yield new Promise((resolve) => this.sendRequest('runInTerminal', {
                    kind: 'integrated',
                    cwd: process.cwd(),
                    env: process.env,
                    args: command,
                }, 5000, resolve));
                if (!response.success) {
                    const message = `could not start the terminal on the client: ${response.message}`;
                    debugadapter_1.logger.error(message);
                    throw new Error(message);
                }
            }));
        });
    }
    setBreakPointsRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            this.waitPausedNeeded = this.isRunning;
            if (this.waitPausedNeeded) {
                // Need to pause first
                const waitPromise = new Promise((resolve) => {
                    this.waitPaused = resolve;
                });
                if (this.gdb.isNonStopMode()) {
                    const threadInfo = yield mi.sendThreadInfoRequest(this.gdb, {});
                    this.waitPausedThreadId = parseInt(threadInfo['current-thread-id'], 10);
                    this.gdb.pause(this.waitPausedThreadId);
                }
                else {
                    this.gdb.pause();
                }
                yield waitPromise;
            }
            try {
                // Need to get the list of current breakpoints in the file and then make sure
                // that we end up with the requested set of breakpoints for that file
                // deleting ones not requested and inserting new ones.
                const result = yield mi.sendBreakList(this.gdb);
                const file = args.source.path;
                const gdbOriginalLocationPrefix = yield (0, mi_1.breakpointLocation)(this.gdb, file);
                const gdbbps = result.BreakpointTable.body.filter((gdbbp) => {
                    // Ignore "children" breakpoint of <MULTIPLE> entries
                    if (gdbbp.number.includes('.')) {
                        return false;
                    }
                    // Ignore other files
                    if (!gdbbp['original-location']) {
                        return false;
                    }
                    if (!gdbbp['original-location'].startsWith(gdbOriginalLocationPrefix)) {
                        return false;
                    }
                    // Ignore function breakpoints
                    return this.functionBreakpoints.indexOf(gdbbp.number) === -1;
                });
                const { resolved, deletes } = this.resolveBreakpoints(args.breakpoints || [], gdbbps, (vsbp, gdbbp) => {
                    // Always invalidate hit conditions as they have a one-way mapping to gdb ignore and temporary
                    if (vsbp.hitCondition) {
                        return false;
                    }
                    // Ensure we can compare undefined and empty strings
                    const vsbpCond = vsbp.condition || undefined;
                    const gdbbpCond = gdbbp.cond || undefined;
                    // Check with original-location so that relocated breakpoints are properly matched
                    const gdbOriginalLocation = `${gdbOriginalLocationPrefix}${vsbp.line}`;
                    return !!(gdbbp['original-location'] === gdbOriginalLocation &&
                        vsbpCond === gdbbpCond);
                });
                // Delete before insert to avoid breakpoint clashes in gdb
                if (deletes.length > 0) {
                    yield mi.sendBreakDelete(this.gdb, { breakpoints: deletes });
                    deletes.forEach((breakpoint) => delete this.logPointMessages[breakpoint]);
                }
                // Reset logPoints
                this.logPointMessages = {};
                // Set up logpoint messages and return a formatted breakpoint for the response body
                const createState = (vsbp, gdbbp) => {
                    if (vsbp.logMessage) {
                        this.logPointMessages[gdbbp.number] = vsbp.logMessage;
                    }
                    let line = 0;
                    if (gdbbp.line) {
                        line = parseInt(gdbbp.line, 10);
                    }
                    else if (vsbp.line) {
                        line = vsbp.line;
                    }
                    return {
                        id: parseInt(gdbbp.number, 10),
                        line,
                        verified: true,
                    };
                };
                const actual = [];
                for (const bp of resolved) {
                    if (bp.gdbbp) {
                        actual.push(createState(bp.vsbp, bp.gdbbp));
                        continue;
                    }
                    let temporary = false;
                    let ignoreCount;
                    const vsbp = bp.vsbp;
                    if (vsbp.hitCondition !== undefined) {
                        ignoreCount = parseInt(vsbp.hitCondition.replace(ignoreCountRegex, ''), 10);
                        if (isNaN(ignoreCount)) {
                            this.sendEvent(new debugadapter_1.OutputEvent(`Unable to decode expression: ${vsbp.hitCondition}`));
                            continue;
                        }
                        // Allow hit condition continuously above the count
                        temporary = !vsbp.hitCondition.startsWith('>');
                        if (temporary) {
                            // The expression is not 'greater than', decrease ignoreCount to match
                            ignoreCount--;
                        }
                    }
                    try {
                        const gdbbp = yield mi.sendBreakInsert(this.gdb, {
                            source: file,
                            line: vsbp.line,
                            condition: vsbp.condition,
                            temporary,
                            ignoreCount,
                            hardware: this.gdb.isUseHWBreakpoint(),
                        });
                        actual.push(createState(vsbp, gdbbp.bkpt));
                    }
                    catch (err) {
                        actual.push({
                            verified: false,
                            message: err instanceof Error ? err.message : String(err),
                        });
                    }
                }
                response.body = {
                    breakpoints: actual,
                };
                this.sendResponse(response);
            }
            catch (err) {
                this.sendErrorResponse(response, 1, err instanceof Error ? err.message : String(err));
            }
            if (this.waitPausedNeeded) {
                if (this.gdb.isNonStopMode()) {
                    mi.sendExecContinue(this.gdb, this.waitPausedThreadId);
                }
                else {
                    mi.sendExecContinue(this.gdb);
                }
            }
        });
    }
    setFunctionBreakPointsRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            this.waitPausedNeeded = this.isRunning;
            if (this.waitPausedNeeded) {
                // Need to pause first
                const waitPromise = new Promise((resolve) => {
                    this.waitPaused = resolve;
                });
                if (this.gdb.isNonStopMode()) {
                    const threadInfo = yield mi.sendThreadInfoRequest(this.gdb, {});
                    this.waitPausedThreadId = parseInt(threadInfo['current-thread-id'], 10);
                    this.gdb.pause(this.waitPausedThreadId);
                }
                else {
                    this.gdb.pause();
                }
                yield waitPromise;
            }
            try {
                const result = yield mi.sendBreakList(this.gdb);
                const gdbbps = result.BreakpointTable.body.filter((gdbbp) => {
                    // Only function breakpoints
                    return this.functionBreakpoints.indexOf(gdbbp.number) > -1;
                });
                const { resolved, deletes } = this.resolveBreakpoints(args.breakpoints, gdbbps, (vsbp, gdbbp) => {
                    // Always invalidate hit conditions as they have a one-way mapping to gdb ignore and temporary
                    if (vsbp.hitCondition) {
                        return false;
                    }
                    // Ensure we can compare undefined and empty strings
                    const vsbpCond = vsbp.condition || undefined;
                    const gdbbpCond = gdbbp.cond || undefined;
                    const originalLocation = (0, mi_1.breakpointFunctionLocation)(this.gdb, vsbp.name);
                    return !!(gdbbp['original-location'] === originalLocation &&
                        vsbpCond === gdbbpCond);
                });
                // Delete before insert to avoid breakpoint clashes in gdb
                if (deletes.length > 0) {
                    yield mi.sendBreakDelete(this.gdb, { breakpoints: deletes });
                    this.functionBreakpoints = this.functionBreakpoints.filter((fnbp) => deletes.indexOf(fnbp) === -1);
                }
                const createActual = (breakpoint) => ({
                    id: parseInt(breakpoint.number, 10),
                    verified: true,
                });
                const actual = [];
                // const actual = existing.map((bp) => createActual(bp.gdbbp));
                for (const bp of resolved) {
                    if (bp.gdbbp) {
                        actual.push(createActual(bp.gdbbp));
                        continue;
                    }
                    try {
                        const gdbbp = yield mi.sendBreakFunctionInsert(this.gdb, bp.vsbp.name, {
                            hardware: this.gdb.isUseHWBreakpoint(),
                        });
                        this.functionBreakpoints.push(gdbbp.bkpt.number);
                        actual.push(createActual(gdbbp.bkpt));
                    }
                    catch (err) {
                        actual.push({
                            verified: false,
                            message: err instanceof Error ? err.message : String(err),
                        });
                    }
                }
                response.body = {
                    breakpoints: actual,
                };
                this.sendResponse(response);
            }
            catch (err) {
                this.sendErrorResponse(response, 1, err instanceof Error ? err.message : String(err));
            }
            if (this.waitPausedNeeded) {
                if (this.gdb.isNonStopMode()) {
                    mi.sendExecContinue(this.gdb, this.waitPausedThreadId);
                }
                else {
                    mi.sendExecContinue(this.gdb);
                }
            }
        });
    }
    /**
     * Resolved which VS breakpoints needs to be installed, which
     * GDB breakpoints need to be deleted and which VS breakpoints
     * are already installed with which matching GDB breakpoint.
     * @param vsbps VS DAP breakpoints
     * @param gdbbps GDB breakpoints
     * @param matchFn matcher to compare VS and GDB breakpoints
     * @returns resolved -> array maintaining order of vsbps that identifies whether
     * VS breakpoint has a cooresponding GDB breakpoint (gdbbp field set) or needs to be
     * inserted (gdbbp field empty)
     * deletes -> GDB bps ids that should be deleted because they don't match vsbps
     */
    resolveBreakpoints(vsbps, gdbbps, matchFn) {
        const resolved = vsbps.map((vsbp) => {
            return {
                vsbp,
                gdbbp: gdbbps.find((gdbbp) => matchFn(vsbp, gdbbp)),
            };
        });
        const deletes = gdbbps
            .filter((gdbbp) => {
            return !vsbps.find((vsbp) => matchFn(vsbp, gdbbp));
        })
            .map((gdbbp) => gdbbp.number);
        return { resolved, deletes };
    }
    configurationDoneRequest(response, _args) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                if (this.isAttach) {
                    yield mi.sendExecContinue(this.gdb);
                }
                else {
                    yield mi.sendExecRun(this.gdb);
                }
                this.sendResponse(response);
            }
            catch (err) {
                this.sendErrorResponse(response, 100, err instanceof Error ? err.message : String(err));
            }
        });
    }
    convertThread(thread) {
        let name = thread.name || thread.id;
        if (thread.details) {
            name += ` (${thread.details})`;
        }
        const running = thread.state === 'running';
        return new ThreadWithStatus(parseInt(thread.id, 10), name, running);
    }
    threadsRequest(response) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                if (!this.isRunning) {
                    const result = yield mi.sendThreadInfoRequest(this.gdb, {});
                    this.threads = result.threads.map((thread) => this.convertThread(thread));
                }
                response.body = {
                    threads: this.threads,
                };
                this.sendResponse(response);
            }
            catch (err) {
                this.sendErrorResponse(response, 1, err instanceof Error ? err.message : String(err));
            }
        });
    }
    stackTraceRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                const threadId = args.threadId;
                const depthResult = yield mi.sendStackInfoDepth(this.gdb, {
                    maxDepth: 100,
                    threadId,
                });
                const depth = parseInt(depthResult.depth, 10);
                const levels = args.levels
                    ? args.levels > depth
                        ? depth
                        : args.levels
                    : depth;
                const lowFrame = args.startFrame || 0;
                const highFrame = lowFrame + levels - 1;
                const listResult = yield mi.sendStackListFramesRequest(this.gdb, {
                    lowFrame,
                    highFrame,
                    threadId,
                });
                const stack = listResult.stack.map((frame) => {
                    let source;
                    if (frame.fullname) {
                        source = new debugadapter_1.Source(path.basename(frame.file || frame.fullname), frame.fullname);
                    }
                    let line;
                    if (frame.line) {
                        line = parseInt(frame.line, 10);
                    }
                    const frameHandle = this.frameHandles.create({
                        threadId: args.threadId,
                        frameId: parseInt(frame.level, 10),
                    });
                    const name = frame.func || frame.fullname || '';
                    const sf = new debugadapter_1.StackFrame(frameHandle, name, source, line);
                    sf.instructionPointerReference = frame.addr;
                    return sf;
                });
                response.body = {
                    stackFrames: stack,
                    totalFrames: depth,
                };
                this.sendResponse(response);
            }
            catch (err) {
                this.sendErrorResponse(response, 1, err instanceof Error ? err.message : String(err));
            }
        });
    }
    nextRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                yield mi.sendExecNext(this.gdb, args.threadId);
                this.sendResponse(response);
            }
            catch (err) {
                this.sendErrorResponse(response, 1, err instanceof Error ? err.message : String(err));
            }
        });
    }
    stepInRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                yield mi.sendExecStep(this.gdb, args.threadId);
                this.sendResponse(response);
            }
            catch (err) {
                this.sendErrorResponse(response, 1, err instanceof Error ? err.message : String(err));
            }
        });
    }
    stepOutRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                yield mi.sendExecFinish(this.gdb, args.threadId);
                this.sendResponse(response);
            }
            catch (err) {
                this.sendErrorResponse(response, 1, err instanceof Error ? err.message : String(err));
            }
        });
    }
    continueRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                yield mi.sendExecContinue(this.gdb, args.threadId);
                this.sendResponse(response);
            }
            catch (err) {
                this.sendErrorResponse(response, 1, err instanceof Error ? err.message : String(err));
            }
        });
    }
    pauseRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                this.gdb.pause(args.threadId);
                this.sendResponse(response);
            }
            catch (err) {
                this.sendErrorResponse(response, 1, err instanceof Error ? err.message : String(err));
            }
        });
    }
    scopesRequest(response, args) {
        const frame = {
            type: 'frame',
            frameHandle: args.frameId,
        };
        const registers = {
            type: 'registers',
            frameHandle: args.frameId,
        };
        response.body = {
            scopes: [
                new debugadapter_1.Scope('Local', this.variableHandles.create(frame), false),
                new debugadapter_1.Scope('Registers', this.variableHandles.create(registers), true),
            ],
        };
        this.sendResponse(response);
    }
    variablesRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            const variables = new Array();
            response.body = {
                variables,
            };
            try {
                const ref = this.variableHandles.get(args.variablesReference);
                if (!ref) {
                    this.sendResponse(response);
                    return;
                }
                if (ref.type === 'registers') {
                    response.body.variables =
                        yield this.handleVariableRequestRegister(ref);
                }
                else if (ref.type === 'frame') {
                    response.body.variables = yield this.handleVariableRequestFrame(ref);
                }
                else if (ref.type === 'object') {
                    response.body.variables =
                        yield this.handleVariableRequestObject(ref);
                }
                this.sendResponse(response);
            }
            catch (err) {
                this.sendErrorResponse(response, 1, err instanceof Error ? err.message : String(err));
            }
        });
    }
    setVariableRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                const ref = this.variableHandles.get(args.variablesReference);
                if (!ref) {
                    this.sendResponse(response);
                    return;
                }
                const frame = this.frameHandles.get(ref.frameHandle);
                if (!frame) {
                    this.sendResponse(response);
                    return;
                }
                const parentVarname = ref.type === 'object' ? ref.varobjName : '';
                const varname = parentVarname +
                    (parentVarname === '' ? '' : '.') +
                    args.name.replace(/^\[(\d+)\]/, '$1');
                const stackDepth = yield mi.sendStackInfoDepth(this.gdb, {
                    maxDepth: 100,
                });
                const depth = parseInt(stackDepth.depth, 10);
                let varobj = this.gdb.varManager.getVar(frame.frameId, frame.threadId, depth, varname, ref.type);
                if (!varobj && ref.type === 'registers') {
                    const varCreateResponse = yield mi.sendVarCreate(this.gdb, {
                        expression: '$' + args.name,
                        frameId: frame.frameId,
                        threadId: frame.threadId,
                    });
                    varobj = this.gdb.varManager.addVar(frame.frameId, frame.threadId, depth, args.name, false, false, varCreateResponse, ref.type);
                    yield mi.sendVarSetFormatToHex(this.gdb, varobj.varname);
                }
                let assign;
                if (varobj) {
                    assign = yield mi.sendVarAssign(this.gdb, {
                        varname: varobj.varname,
                        expression: args.value,
                    });
                }
                else {
                    try {
                        assign = yield mi.sendVarAssign(this.gdb, {
                            varname,
                            expression: args.value,
                        });
                    }
                    catch (err) {
                        if (parentVarname === '') {
                            throw err; // no recovery possible
                        }
                        const children = yield mi.sendVarListChildren(this.gdb, {
                            name: parentVarname,
                            printValues: mi.MIVarPrintValues.all,
                        });
                        for (const child of children.children) {
                            if (this.isChildOfClass(child)) {
                                const grandchildVarname = child.name +
                                    '.' +
                                    args.name.replace(/^\[(\d+)\]/, '$1');
                                varobj = this.gdb.varManager.getVar(frame.frameId, frame.threadId, depth, grandchildVarname);
                                try {
                                    assign = yield mi.sendVarAssign(this.gdb, {
                                        varname: grandchildVarname,
                                        expression: args.value,
                                    });
                                    break;
                                }
                                catch (err) {
                                    continue; // try another child
                                }
                            }
                        }
                        if (!assign) {
                            throw err; // no recovery possible
                        }
                    }
                }
                response.body = {
                    value: assign.value,
                    type: varobj ? varobj.type : undefined,
                    variablesReference: varobj && parseInt(varobj.numchild, 10) > 0
                        ? this.variableHandles.create({
                            type: 'object',
                            frameHandle: ref.frameHandle,
                            varobjName: varobj.varname,
                        })
                        : 0,
                };
            }
            catch (err) {
                this.sendErrorResponse(response, 1, err instanceof Error ? err.message : String(err));
            }
            this.sendResponse(response);
        });
    }
    // protected async setExpressionRequest(response: DebugProtocol.SetExpressionResponse,
    //                                      args: DebugProtocol.SetExpressionArguments): Promise<void> {
    //     logger.error('got setExpressionRequest');
    //     this.sendResponse(response);
    // }
    evaluateRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            response.body = {
                result: 'Error: could not evaluate expression',
                variablesReference: 0,
            }; // default response
            try {
                if (args.frameId === undefined) {
                    throw new Error('Evaluation of expression without frameId is not supported.');
                }
                const frame = this.frameHandles.get(args.frameId);
                if (!frame) {
                    this.sendResponse(response);
                    return;
                }
                const stackDepth = yield mi.sendStackInfoDepth(this.gdb, {
                    maxDepth: 100,
                });
                const depth = parseInt(stackDepth.depth, 10);
                let varobj = this.gdb.varManager.getVar(frame.frameId, frame.threadId, depth, args.expression);
                if (!varobj) {
                    const varCreateResponse = yield mi.sendVarCreate(this.gdb, {
                        expression: args.expression,
                        frameId: frame.frameId,
                        threadId: frame.threadId,
                    });
                    varobj = this.gdb.varManager.addVar(frame.frameId, frame.threadId, depth, args.expression, false, false, varCreateResponse);
                }
                else {
                    const vup = yield mi.sendVarUpdate(this.gdb, {
                        name: varobj.varname,
                    });
                    const update = vup.changelist[0];
                    if (update) {
                        if (update.in_scope === 'true') {
                            if (update.name === varobj.varname) {
                                varobj.value = update.value;
                            }
                        }
                        else {
                            this.gdb.varManager.removeVar(frame.frameId, frame.threadId, depth, varobj.varname);
                            yield mi.sendVarDelete(this.gdb, {
                                varname: varobj.varname,
                            });
                            const varCreateResponse = yield mi.sendVarCreate(this.gdb, {
                                expression: args.expression,
                                frameId: frame.frameId,
                                threadId: frame.threadId,
                            });
                            varobj = this.gdb.varManager.addVar(frame.frameId, frame.threadId, depth, args.expression, false, false, varCreateResponse);
                        }
                    }
                }
                if (varobj) {
                    response.body = {
                        result: varobj.value,
                        type: varobj.type,
                        variablesReference: parseInt(varobj.numchild, 10) > 0
                            ? this.variableHandles.create({
                                type: 'object',
                                frameHandle: args.frameId,
                                varobjName: varobj.varname,
                            })
                            : 0,
                    };
                }
                this.sendResponse(response);
            }
            catch (err) {
                this.sendErrorResponse(response, 1, err instanceof Error ? err.message : String(err));
            }
        });
    }
    /**
     * Implement the cdt-gdb-adapter/Memory request.
     */
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    memoryRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                if (typeof args.address !== 'string') {
                    throw new Error(`Invalid type for 'address', expected string, got ${typeof args.address}`);
                }
                if (typeof args.length !== 'number') {
                    throw new Error(`Invalid type for 'length', expected number, got ${typeof args.length}`);
                }
                if (typeof args.offset !== 'number' &&
                    typeof args.offset !== 'undefined') {
                    throw new Error(`Invalid type for 'offset', expected number or undefined, got ${typeof args.offset}`);
                }
                const typedArgs = args;
                const result = yield (0, data_1.sendDataReadMemoryBytes)(this.gdb, typedArgs.address, typedArgs.length, typedArgs.offset);
                response.body = {
                    data: result.memory[0].contents,
                    address: result.memory[0].begin,
                };
                this.sendResponse(response);
            }
            catch (err) {
                this.sendErrorResponse(response, 1, err instanceof Error ? err.message : String(err));
            }
        });
    }
    disassembleRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                const meanSizeOfInstruction = 4;
                let startOffset = 0;
                let lastStartOffset = -1;
                const instructions = [];
                let oneIterationOnly = false;
                outer_loop: while (instructions.length < args.instructionCount &&
                    !oneIterationOnly) {
                    if (startOffset === lastStartOffset) {
                        // We have stopped getting new instructions, give up
                        break outer_loop;
                    }
                    lastStartOffset = startOffset;
                    const fetchSize = (args.instructionCount - instructions.length) *
                        meanSizeOfInstruction;
                    // args.memoryReference is an arbitrary expression, so let GDB do the
                    // math on resolving value rather than doing the addition in the adapter
                    try {
                        const stepStartAddress = `(${args.memoryReference})+${startOffset}`;
                        let stepEndAddress = `(${args.memoryReference})+${startOffset}+${fetchSize}`;
                        if (args.endMemoryReference && instructions.length === 0) {
                            // On the first call, if we have an end memory address use it instead of
                            // the approx size
                            stepEndAddress = args.endMemoryReference;
                            oneIterationOnly = true;
                        }
                        const result = yield (0, data_1.sendDataDisassemble)(this.gdb, stepStartAddress, stepEndAddress);
                        for (const asmInsn of result.asm_insns) {
                            const line = asmInsn.line
                                ? parseInt(asmInsn.line, 10)
                                : undefined;
                            const source = {
                                name: asmInsn.file,
                                path: asmInsn.fullname,
                            };
                            for (const asmLine of asmInsn.line_asm_insn) {
                                let funcAndOffset;
                                if (asmLine['func-name'] && asmLine.offset) {
                                    funcAndOffset = `${asmLine['func-name']}+${asmLine.offset}`;
                                }
                                else if (asmLine['func-name']) {
                                    funcAndOffset = asmLine['func-name'];
                                }
                                else {
                                    funcAndOffset = undefined;
                                }
                                const disInsn = {
                                    address: asmLine.address,
                                    instructionBytes: asmLine.opcodes,
                                    instruction: asmLine.inst,
                                    symbol: funcAndOffset,
                                    location: source,
                                    line,
                                };
                                instructions.push(disInsn);
                                if (instructions.length === args.instructionCount) {
                                    break outer_loop;
                                }
                                const bytes = asmLine.opcodes.replace(/\s/g, '');
                                startOffset += bytes.length;
                            }
                        }
                    }
                    catch (err) {
                        // Failed to read instruction -- what best to do here?
                        // in other words, whose responsibility (adapter or client)
                        // to reissue reads in smaller chunks to find good memory
                        while (instructions.length < args.instructionCount) {
                            const badDisInsn = {
                                // TODO this should start at byte after last retrieved address
                                address: `0x${startOffset.toString(16)}`,
                                instruction: err instanceof Error
                                    ? err.message
                                    : String(err),
                            };
                            instructions.push(badDisInsn);
                            startOffset += 2;
                        }
                        break outer_loop;
                    }
                }
                if (!args.endMemoryReference) {
                    while (instructions.length < args.instructionCount) {
                        const badDisInsn = {
                            // TODO this should start at byte after last retrieved address
                            address: `0x${startOffset.toString(16)}`,
                            instruction: 'failed to retrieve instruction',
                        };
                        instructions.push(badDisInsn);
                        startOffset += 2;
                    }
                }
                response.body = { instructions };
                this.sendResponse(response);
            }
            catch (err) {
                this.sendErrorResponse(response, 1, err instanceof Error ? err.message : String(err));
            }
        });
    }
    readMemoryRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                if (args.count) {
                    const result = yield (0, data_1.sendDataReadMemoryBytes)(this.gdb, args.memoryReference, args.count, args.offset);
                    response.body = {
                        data: hexToBase64(result.memory[0].contents),
                        address: result.memory[0].begin,
                    };
                    this.sendResponse(response);
                }
                else {
                    this.sendResponse(response);
                }
            }
            catch (err) {
                this.sendErrorResponse(response, 1, err instanceof Error ? err.message : String(err));
            }
        });
    }
    /**
     * Implement the memoryWrite request.
     */
    writeMemoryRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                const { memoryReference, data } = args;
                const typeofAddress = typeof memoryReference;
                const typeofContent = typeof data;
                if (typeofAddress !== 'string') {
                    throw new Error(`Invalid type for 'address', expected string, got ${typeofAddress}`);
                }
                if (typeofContent !== 'string') {
                    throw new Error(`Invalid type for 'content', expected string, got ${typeofContent}`);
                }
                const hexContent = base64ToHex(data);
                yield (0, data_1.sendDataWriteMemoryBytes)(this.gdb, memoryReference, hexContent);
                this.sendResponse(response);
            }
            catch (err) {
                this.sendErrorResponse(response, 1, err instanceof Error ? err.message : String(err));
            }
        });
    }
    disconnectRequest(response, _args) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                yield this.gdb.sendGDBExit();
                this.sendResponse(response);
            }
            catch (err) {
                this.sendErrorResponse(response, 1, err instanceof Error ? err.message : String(err));
            }
        });
    }
    sendStoppedEvent(reason, threadId, allThreadsStopped) {
        // Reset frame handles and variables for new context
        this.frameHandles.reset();
        this.variableHandles.reset();
        // Send the event
        this.sendEvent(new stoppedEvent_1.StoppedEvent(reason, threadId, allThreadsStopped));
    }
    handleGDBStopped(result) {
        const getThreadId = (resultData) => parseInt(resultData['thread-id'], 10);
        const getAllThreadsStopped = (resultData) => {
            return (!!resultData['stopped-threads'] &&
                resultData['stopped-threads'] === 'all');
        };
        switch (result.reason) {
            case 'exited':
            case 'exited-normally':
                this.sendEvent(new debugadapter_1.TerminatedEvent());
                break;
            case 'breakpoint-hit':
                if (this.logPointMessages[result.bkptno]) {
                    this.sendEvent(new debugadapter_1.OutputEvent(this.logPointMessages[result.bkptno]));
                    mi.sendExecContinue(this.gdb);
                }
                else {
                    const reason = this.functionBreakpoints.indexOf(result.bkptno) > -1
                        ? 'function breakpoint'
                        : 'breakpoint';
                    this.sendStoppedEvent(reason, getThreadId(result), getAllThreadsStopped(result));
                }
                break;
            case 'end-stepping-range':
            case 'function-finished':
                this.sendStoppedEvent('step', getThreadId(result), getAllThreadsStopped(result));
                break;
            case 'signal-received': {
                const name = result['signal-name'] || 'signal';
                this.sendStoppedEvent(name, getThreadId(result), getAllThreadsStopped(result));
                break;
            }
            default:
                this.sendStoppedEvent('generic', getThreadId(result), getAllThreadsStopped(result));
        }
    }
    handleGDBAsync(resultClass, resultData) {
        const updateIsRunning = () => {
            this.isRunning = true;
            for (const thread of this.threads) {
                if (!thread.running) {
                    this.isRunning = false;
                }
            }
        };
        switch (resultClass) {
            case 'running':
                if (this.gdb.isNonStopMode()) {
                    const id = parseInt(resultData['thread-id'], 10);
                    for (const thread of this.threads) {
                        if (thread.id === id) {
                            thread.running = true;
                        }
                    }
                }
                else {
                    for (const thread of this.threads) {
                        thread.running = true;
                    }
                }
                updateIsRunning();
                break;
            case 'stopped': {
                let suppressHandleGDBStopped = false;
                if (this.gdb.isNonStopMode()) {
                    const id = parseInt(resultData['thread-id'], 10);
                    for (const thread of this.threads) {
                        if (thread.id === id) {
                            thread.running = false;
                        }
                    }
                    if (this.waitPaused &&
                        resultData.reason === 'signal-received' &&
                        this.waitPausedThreadId === id) {
                        suppressHandleGDBStopped = true;
                    }
                }
                else {
                    for (const thread of this.threads) {
                        thread.running = false;
                    }
                    if (this.waitPaused &&
                        resultData.reason === 'signal-received') {
                        suppressHandleGDBStopped = true;
                    }
                }
                if (this.waitPaused) {
                    if (!suppressHandleGDBStopped) {
                        // if we aren't suppressing the stopped event going
                        // to the client, then we also musn't resume the
                        // target after inserting the breakpoints
                        this.waitPausedNeeded = false;
                    }
                    this.waitPaused();
                    this.waitPaused = undefined;
                }
                const wasRunning = this.isRunning;
                updateIsRunning();
                if (!suppressHandleGDBStopped &&
                    (this.gdb.isNonStopMode() ||
                        (wasRunning && !this.isRunning))) {
                    if (this.isInitialized) {
                        this.handleGDBStopped(resultData);
                    }
                }
                break;
            }
            default:
                debugadapter_1.logger.warn(`GDB unhandled async: ${resultClass}: ${JSON.stringify(resultData)}`);
        }
    }
    handleGDBNotify(notifyClass, notifyData) {
        switch (notifyClass) {
            case 'thread-created':
                this.threads.push(this.convertThread(notifyData));
                break;
            case 'thread-exited': {
                const thread = notifyData;
                const exitId = parseInt(thread.id, 10);
                this.threads = this.threads.filter((t) => t.id !== exitId);
                break;
            }
            case 'thread-selected':
            case 'thread-group-added':
            case 'thread-group-started':
            case 'thread-group-exited':
            case 'library-loaded':
            case 'breakpoint-modified':
            case 'breakpoint-deleted':
                // Known unhandled notifies
                break;
            default:
                debugadapter_1.logger.warn(`GDB unhandled notify: ${notifyClass}: ${JSON.stringify(notifyData)}`);
        }
    }
    handleVariableRequestFrame(ref) {
        return __awaiter(this, void 0, void 0, function* () {
            // initialize variables array and dereference the frame handle
            const variables = [];
            const frame = this.frameHandles.get(ref.frameHandle);
            if (!frame) {
                return Promise.resolve(variables);
            }
            // vars used to determine if we should call sendStackListVariables()
            let callStack = false;
            let numVars = 0;
            // stack depth necessary for differentiating between similarly named variables at different stack depths
            const stackDepth = yield mi.sendStackInfoDepth(this.gdb, {
                maxDepth: 100,
            });
            const depth = parseInt(stackDepth.depth, 10);
            // array of varnames to delete. Cannot delete while iterating through the vars array below.
            const toDelete = new Array();
            // get the list of vars we need to update for this frameId/threadId/depth tuple
            const vars = this.gdb.varManager.getVars(frame.frameId, frame.threadId, depth);
            if (vars) {
                for (const varobj of vars) {
                    // ignore expressions and child entries
                    if (varobj.isVar && !varobj.isChild) {
                        // request update from GDB
                        const vup = yield mi.sendVarUpdate(this.gdb, {
                            name: varobj.varname,
                        });
                        // if changelist is length 0, update is undefined
                        const update = vup.changelist[0];
                        let pushVar = true;
                        if (update) {
                            if (update.in_scope === 'true') {
                                numVars++;
                                if (update.name === varobj.varname) {
                                    // don't update the parent value to a child's value
                                    varobj.value = update.value;
                                }
                            }
                            else {
                                // var is out of scope, delete it and call sendStackListVariables() later
                                callStack = true;
                                pushVar = false;
                                toDelete.push(update.name);
                            }
                        }
                        else if (varobj.value) {
                            // value hasn't updated but it's still in scope
                            numVars++;
                        }
                        // only push entries to the result that aren't being deleted
                        if (pushVar) {
                            let value = varobj.value;
                            // if we have an array parent entry, we need to display the address.
                            if (arrayRegex.test(varobj.type)) {
                                value = yield this.getAddr(varobj);
                            }
                            variables.push({
                                name: varobj.expression,
                                value,
                                type: varobj.type,
                                memoryReference: `&(${varobj.expression})`,
                                variablesReference: parseInt(varobj.numchild, 10) > 0
                                    ? this.variableHandles.create({
                                        type: 'object',
                                        frameHandle: ref.frameHandle,
                                        varobjName: varobj.varname,
                                    })
                                    : 0,
                            });
                        }
                    }
                }
                // clean up out of scope entries
                for (const varname of toDelete) {
                    yield this.gdb.varManager.removeVar(frame.frameId, frame.threadId, depth, varname);
                }
            }
            // if we had out of scope entries or no entries in the frameId/threadId/depth tuple, query GDB for new ones
            if (callStack === true || numVars === 0) {
                const result = yield mi.sendStackListVariables(this.gdb, {
                    thread: frame.threadId,
                    frame: frame.frameId,
                    printValues: 'simple-values',
                });
                for (const variable of result.variables) {
                    let varobj = this.gdb.varManager.getVar(frame.frameId, frame.threadId, depth, variable.name);
                    if (!varobj) {
                        // create var in GDB and store it in the varMgr
                        const varCreateResponse = yield mi.sendVarCreate(this.gdb, {
                            expression: variable.name,
                            frameId: frame.frameId,
                            threadId: frame.threadId,
                        });
                        varobj = this.gdb.varManager.addVar(frame.frameId, frame.threadId, depth, variable.name, true, false, varCreateResponse);
                    }
                    else {
                        // var existed as an expression before. Now it's a variable too.
                        varobj = yield this.gdb.varManager.updateVar(frame.frameId, frame.threadId, depth, varobj);
                        varobj.isVar = true;
                    }
                    let value = varobj.value;
                    // if we have an array parent entry, we need to display the address.
                    if (arrayRegex.test(varobj.type)) {
                        value = yield this.getAddr(varobj);
                    }
                    variables.push({
                        name: varobj.expression,
                        value,
                        type: varobj.type,
                        memoryReference: `&(${varobj.expression})`,
                        variablesReference: parseInt(varobj.numchild, 10) > 0
                            ? this.variableHandles.create({
                                type: 'object',
                                frameHandle: ref.frameHandle,
                                varobjName: varobj.varname,
                            })
                            : 0,
                    });
                }
            }
            return Promise.resolve(variables);
        });
    }
    handleVariableRequestObject(ref) {
        return __awaiter(this, void 0, void 0, function* () {
            // initialize variables array and dereference the frame handle
            const variables = [];
            const frame = this.frameHandles.get(ref.frameHandle);
            if (!frame) {
                return Promise.resolve(variables);
            }
            // fetch stack depth to obtain frameId/threadId/depth tuple
            const stackDepth = yield mi.sendStackInfoDepth(this.gdb, {
                maxDepth: 100,
            });
            const depth = parseInt(stackDepth.depth, 10);
            // we need to keep track of children and the parent varname in GDB
            let children;
            let parentVarname = ref.varobjName;
            // if a varobj exists, use the varname stored there
            const varobj = this.gdb.varManager.getVarByName(frame.frameId, frame.threadId, depth, ref.varobjName);
            if (varobj) {
                children = yield mi.sendVarListChildren(this.gdb, {
                    name: varobj.varname,
                    printValues: mi.MIVarPrintValues.all,
                });
                parentVarname = varobj.varname;
            }
            else {
                // otherwise use the parent name passed in the variable reference
                children = yield mi.sendVarListChildren(this.gdb, {
                    name: ref.varobjName,
                    printValues: mi.MIVarPrintValues.all,
                });
            }
            // iterate through the children
            for (const child of children.children) {
                // check if we're dealing with a C++ object. If we are, we need to fetch the grandchildren instead.
                const isClass = this.isChildOfClass(child);
                if (isClass) {
                    const name = `${parentVarname}.${child.exp}`;
                    const objChildren = yield mi.sendVarListChildren(this.gdb, {
                        name,
                        printValues: mi.MIVarPrintValues.all,
                    });
                    for (const objChild of objChildren.children) {
                        const childName = `${name}.${objChild.exp}`;
                        variables.push({
                            name: objChild.exp,
                            value: objChild.value ? objChild.value : objChild.type,
                            type: objChild.type,
                            variablesReference: parseInt(objChild.numchild, 10) > 0
                                ? this.variableHandles.create({
                                    type: 'object',
                                    frameHandle: ref.frameHandle,
                                    varobjName: childName,
                                })
                                : 0,
                        });
                    }
                }
                else {
                    // check if we're dealing with an array
                    let name = `${ref.varobjName}.${child.exp}`;
                    let varobjName = name;
                    let value = child.value ? child.value : child.type;
                    const isArrayParent = arrayRegex.test(child.type);
                    const isArrayChild = varobj !== undefined
                        ? arrayRegex.test(varobj.type) &&
                            arrayChildRegex.test(child.exp)
                        : false;
                    if (isArrayChild) {
                        // update the display name for array elements to have square brackets
                        name = `[${child.exp}]`;
                    }
                    if (isArrayParent || isArrayChild) {
                        // can't use a relative varname (eg. var1.a.b.c) to create/update a new var so fetch and track these
                        // vars by evaluating their path expression from GDB
                        const exprResponse = yield mi.sendVarInfoPathExpression(this.gdb, child.name);
                        // create or update the var in GDB
                        let arrobj = this.gdb.varManager.getVar(frame.frameId, frame.threadId, depth, exprResponse.path_expr);
                        if (!arrobj) {
                            const varCreateResponse = yield mi.sendVarCreate(this.gdb, {
                                expression: exprResponse.path_expr,
                                frameId: frame.frameId,
                                threadId: frame.threadId,
                            });
                            arrobj = this.gdb.varManager.addVar(frame.frameId, frame.threadId, depth, exprResponse.path_expr, true, false, varCreateResponse);
                        }
                        else {
                            arrobj = yield this.gdb.varManager.updateVar(frame.frameId, frame.threadId, depth, arrobj);
                        }
                        // if we have an array parent entry, we need to display the address.
                        if (isArrayParent) {
                            value = yield this.getAddr(arrobj);
                        }
                        arrobj.isChild = true;
                        varobjName = arrobj.varname;
                    }
                    const variableName = isArrayChild ? name : child.exp;
                    variables.push({
                        name: variableName,
                        value,
                        type: child.type,
                        variablesReference: parseInt(child.numchild, 10) > 0
                            ? this.variableHandles.create({
                                type: 'object',
                                frameHandle: ref.frameHandle,
                                varobjName,
                            })
                            : 0,
                    });
                }
            }
            return Promise.resolve(variables);
        });
    }
    handleVariableRequestRegister(ref) {
        return __awaiter(this, void 0, void 0, function* () {
            // initialize variables array and dereference the frame handle
            const variables = [];
            const frame = this.frameHandles.get(ref.frameHandle);
            if (!frame) {
                return Promise.resolve(variables);
            }
            if (this.registerMap.size === 0) {
                const result_names = yield mi.sendDataListRegisterNames(this.gdb, {
                    frameId: frame.frameId,
                    threadId: frame.threadId,
                });
                let idx = 0;
                const registerNames = result_names['register-names'];
                for (const regs of registerNames) {
                    if (regs !== '') {
                        this.registerMap.set(regs, idx);
                        this.registerMapReverse.set(idx, regs);
                    }
                    idx++;
                }
            }
            const result_values = yield mi.sendDataListRegisterValues(this.gdb, {
                fmt: 'x',
                frameId: frame.frameId,
                threadId: frame.threadId,
            });
            const reg_values = result_values['register-values'];
            for (const n of reg_values) {
                const id = n.number;
                const reg = this.registerMapReverse.get(parseInt(id));
                if (reg) {
                    const val = n.value;
                    const res = {
                        name: reg,
                        evaluateName: '$' + reg,
                        value: val,
                        variablesReference: 0,
                    };
                    variables.push(res);
                }
                else {
                    throw new Error('Unable to parse response for reg. values');
                }
            }
            return Promise.resolve(variables);
        });
    }
    getAddr(varobj) {
        return __awaiter(this, void 0, void 0, function* () {
            const addr = yield mi.sendDataEvaluateExpression(this.gdb, `&(${varobj.expression})`);
            return addr.value ? addr.value : varobj.value;
        });
    }
    isChildOfClass(child) {
        return (child.type === undefined &&
            child.value === '' &&
            (child.exp === 'public' ||
                child.exp === 'protected' ||
                child.exp === 'private'));
    }
}
exports.GDBDebugSession = GDBDebugSession;
//# sourceMappingURL=GDBDebugSession.js.map