#!/usr/bin/env Rscript

# Example script demonstrating Base R bar plot with MAIDR

library(devtools)
# Get the directory where this script is located
script_path <- commandArgs(trailingOnly = FALSE)
script_file <- sub("--file=", "", script_path[grep("--file=", script_path)])
script_dir <- dirname(normalizePath(script_file))
# Script is in inst/examples/, so go up two levels to reach package root
maidr_dir <- dirname(dirname(script_dir))
load_all(maidr_dir)

# Create output directory inside inst/ if it doesn't exist
# script_dir is inst/examples, so go up one level to inst/
output_dir <- file.path(dirname(script_dir), "output")
if (!dir.exists(output_dir)) {
  dir.create(output_dir, recursive = TRUE)
}

cat("=== Base R Bar Plot Example ===\n")

# Create a Base R barplot
bar_values <- c(30, 25, 15, 10)
barplot(bar_values,
  names.arg = c("A", "B", "C", "D"),
  main = "Simple Base R Bar Plot",
  sub = "Sample bar chart with four categories",
  xlab = "Categories",
  ylab = "Values",
  col = "lightblue",
  border = "black",
  yaxt = "n"
)

# Add y-axis with comma formatting
axis(2, at = pretty(range(bar_values)), labels = scales::label_comma())

# Generate interactive HTML
html_file <- file.path(output_dir, "example_bar_plot_base_r.html")
save_html(file = html_file)

cat("✓ Base R bar plot example completed\n")
cat("Generated:", html_file, "\n")

cat("\n=== Base R Dodged Bar Plot Example ===\n")

# Create a Base R dodged barplot with simple, non-zero values
test_matrix <- matrix(c(
  10, 20, 15, # Series A values
  15, 25, 20, # Series B values
  20, 30, 25 # Series C values
), nrow = 3, byrow = TRUE)

rownames(test_matrix) <- c("A", "B", "C")
colnames(test_matrix) <- c("Cat1", "Cat2", "Cat3")

# Create dodged barplot
barplot(test_matrix,
  beside = TRUE,
  col = c("red", "blue", "green"),
  legend.text = TRUE,
  args.legend = list(x = "topright", cex = 0.8),
  main = "Base R Dodged Bar Plot",
  xlab = "Categories",
  ylab = "Values",
  border = "black",
  yaxt = "n"
)

# Add y-axis with comma formatting
axis(2, at = pretty(range(test_matrix)), labels = scales::label_comma())

# Generate interactive HTML
dodged_html_file <- file.path(output_dir, "example_dodged_bar_plot_base_r.html")
save_html(file = dodged_html_file)

cat("✓ Base R dodged bar plot example completed\n")
cat("Generated:", dodged_html_file, "\n")

cat("\n=== Base R Stacked Bar Plot Example ===\n")

# Create a Base R stacked barplot (matrix + beside = FALSE)
stacked_matrix <- matrix(c(
  10, 20, 30, # Type1
  15, 25, 35 # Type2
), nrow = 2, byrow = TRUE)

rownames(stacked_matrix) <- c("Type1", "Type2")
colnames(stacked_matrix) <- c("A", "B", "C")

barplot(stacked_matrix,
  beside = FALSE,
  col = c("steelblue", "orange"),
  legend.text = rownames(stacked_matrix),
  args.legend = list(x = "topright", bty = "n"),
  main = "Base R Stacked Bar Plot",
  xlab = "Category",
  ylab = "Value",
  border = "black",
  yaxt = "n"
)

# Add y-axis with comma formatting
axis(2, at = pretty(c(0, colSums(stacked_matrix))), labels = scales::label_comma())

# Generate interactive HTML
stacked_html_file <- file.path(output_dir, "example_stacked_bar_plot_base_r.html")
save_html(file = stacked_html_file)

cat("✓ Base R stacked bar plot example completed\n")
cat("Generated:", stacked_html_file, "\n")

cat("\n=== Base R Histogram Example ===\n")

# Create a Base R histogram
set.seed(123)
hist_data <- rnorm(100, mean = 0, sd = 1)

h <- hist(hist_data,
  main = "Base R Histogram",
  sub = "Normal distribution (mean=0, sd=1)",
  xlab = "Values",
  ylab = "Frequency",
  col = "steelblue",
  border = "black",
  yaxt = "n"
)

# Add y-axis with integer formatting for frequency counts
axis(2, at = pretty(c(0, max(h$counts))), labels = scales::label_number(accuracy = 1))

# Generate interactive HTML
hist_html_file <- file.path(output_dir, "example_histogram_base_r.html")
save_html(file = hist_html_file)

cat("✓ Base R histogram example completed\n")
cat("Generated:", hist_html_file, "\n")

cat("\n=== Base R Density/Smooth Plot Example ===\n")

# Create a Base R density plot
set.seed(456)
density_data <- rnorm(100, mean = 0, sd = 1)

dens <- density(density_data)
plot(dens,
  main = "Base R Density Plot",
  xlab = "Value",
  ylab = "Density",
  col = "darkblue",
  lwd = 2,
  yaxt = "n"
)

# Add y-axis with decimal formatting for density values
axis(2, at = pretty(c(0, max(dens$y))), labels = scales::label_number(accuracy = 0.001))

# Generate interactive HTML
density_html_file <- file.path(output_dir, "example_density_plot_base_r.html")
save_html(file = density_html_file)

cat("✓ Base R density plot example completed\n")
cat("Generated:", density_html_file, "\n")

cat("\n=== Base R Histogram + Density Multi-Layer Example ===\n")

# Create histogram with density overlay
set.seed(42)
data <- rnorm(200, mean = 10, sd = 3)

hist(data,
  probability = TRUE,
  main = "Histogram with Density Curve",
  xlab = "Value",
  ylab = "Density",
  col = "lightblue",
  border = "white",
  breaks = 15,
  yaxt = "n"
)

# Add y-axis with decimal formatting for density values
dens_data <- density(data)
axis(2, at = pretty(c(0, max(dens_data$y))), labels = scales::label_number(accuracy = 0.01))

lines(dens_data,
  col = "darkred",
  lwd = 3
)

# Generate interactive HTML
multilayer_html_file <- file.path(output_dir, "example_histogram_density_base_r.html")
save_html(file = multilayer_html_file)

cat("✓ Base R histogram + density multi-layer example completed\n")
cat("Generated:", multilayer_html_file, "\n")

cat("\n=== Base R Single Line Plot Example ===\n")

# Create a Base R single line plot
x <- 1:10
y <- c(5, 7, 3, 8, 6, 9, 4, 7, 10, 8)

plot(x, y,
  type = "l",
  main = "Base R Single Line Plot",
  xlab = "X values",
  ylab = "Y values",
  col = "steelblue",
  lwd = 2,
  yaxt = "n"
)

# Add y-axis with decimal formatting
axis(2, at = pretty(range(y)), labels = scales::label_number(accuracy = 0.1))

# Generate interactive HTML
line_html_file <- file.path(output_dir, "example_line_plot_base_r.html")
save_html(file = line_html_file)

cat("✓ Base R single line plot example completed\n")
cat("Generated:", line_html_file, "\n")

cat("\n=== Base R Multiline Plot Example ===\n")

# Create a Base R multiline plot using matplot
set.seed(123)
x <- 1:12
y1 <- c(10, 12, 11, 14, 13, 15, 14, 16, 15, 17, 16, 18)
y2 <- c(8, 10, 9, 11, 10, 12, 11, 13, 12, 14, 13, 15)
y3 <- c(15, 17, 16, 18, 17, 19, 18, 20, 19, 21, 20, 22)

y_matrix <- cbind(y1, y2, y3)
colnames(y_matrix) <- c("Product A", "Product B", "Product C")

matplot(x, y_matrix,
  type = "l",
  main = "Base R Multiline Plot (3 Series)",
  xlab = "Month",
  ylab = "Sales",
  col = c("red", "green", "blue"),
  lty = 1,
  lwd = 2,
  yaxt = "n"
)

# Add y-axis with comma formatting for sales values
axis(2, at = pretty(range(y_matrix)), labels = scales::label_comma())

legend("topright",
  legend = colnames(y_matrix),
  col = c("red", "green", "blue"),
  lty = 1,
  lwd = 2
)

# Generate interactive HTML
multiline_html_file <- file.path(output_dir, "example_multiline_plot_base_r.html")
save_html(file = multiline_html_file)

cat("✓ Base R multiline plot example completed\n")
cat("Generated:", multiline_html_file, "\n")

cat("\n=== Base R Heatmap Example ===\n")

# Create a Base R heatmap
heatmap_data <- matrix(c(
  4, 3,
  2, 1
), nrow = 2, ncol = 2, byrow = TRUE)

colnames(heatmap_data) <- c("A", "B")
rownames(heatmap_data) <- c("2", "1")

# Generate color palette
n_colors <- 100
color_palette <- colorRampPalette(c("darkblue", "blue", "lightblue"))(n_colors)

# Create the heatmap using heatmap() function
heatmap(heatmap_data,
  Rowv = NA, Colv = NA, # No dendrograms
  col = color_palette,
  scale = "none",
  main = "Base R Heatmap Example",
  xlab = "Columns", ylab = "Rows",
  margins = c(5, 8)
)

# Generate interactive HTML
heatmap_html_file <- file.path(output_dir, "example_heatmap_base_r.html")
save_html(file = heatmap_html_file)

cat("✓ Base R heatmap example completed\n")
cat("Generated:", heatmap_html_file, "\n")

cat("\n=== Base R Vertical Boxplot Example ===\n")

# Create a Base R vertical boxplot
set.seed(789)
boxplot_data <- list(
  Group1 = rnorm(30, mean = 100, sd = 15),
  Group2 = rnorm(30, mean = 120, sd = 20),
  Group3 = rnorm(30, mean = 110, sd = 18)
)

boxplot(boxplot_data,
  horizontal = FALSE,
  col = c("lightblue", "lightgreen", "lightcoral"),
  main = "Base R Vertical Boxplot",
  xlab = "Group",
  ylab = "Value",
  border = "black",
  yaxt = "n"
)

# Add y-axis with 2 decimal precision
all_values <- unlist(boxplot_data)
axis(2, at = pretty(range(all_values)), labels = scales::label_number(accuracy = 0.01))

# Generate interactive HTML
vertical_boxplot_html_file <- file.path(output_dir, "example_boxplot_vertical_base_r.html")
save_html(file = vertical_boxplot_html_file)

cat("✓ Base R vertical boxplot example completed\n")
cat("Generated:", vertical_boxplot_html_file, "\n")

cat("\n=== Base R Horizontal Boxplot Example ===\n")

# Create a Base R horizontal boxplot
set.seed(890)
boxplot_data_h <- list(
  Category_A = rnorm(25, mean = 50, sd = 10),
  Category_B = rnorm(25, mean = 70, sd = 12),
  Category_C = rnorm(25, mean = 60, sd = 11)
)

boxplot(boxplot_data_h,
  horizontal = TRUE,
  col = c("steelblue", "orange", "purple"),
  main = "Base R Horizontal Boxplot",
  xlab = "Value",
  ylab = "Category",
  border = "black",
  xaxt = "n"
)

# Add x-axis with 2 decimal precision (values are on x-axis for horizontal boxplot)
all_values_h <- unlist(boxplot_data_h)
axis(1, at = pretty(range(all_values_h)), labels = scales::label_number(accuracy = 0.01))

# Generate interactive HTML
horizontal_boxplot_html_file <- file.path(output_dir, "example_boxplot_horizontal_base_r.html")
save_html(file = horizontal_boxplot_html_file)

cat("✓ Base R horizontal boxplot example completed\n")
cat("Generated:", horizontal_boxplot_html_file, "\n")

cat("\n=== Base R Scatter Plot Example ===\n")

# Create a Base R scatter plot with multiple y values per x
set.seed(123)
x_values <- rep(1:5, each = 3) # 3 measurements per x value
y_values <- c(
  rnorm(3, 10, 1), rnorm(3, 15, 2), rnorm(3, 12, 1.5),
  rnorm(3, 18, 1.8), rnorm(3, 14, 0.8)
)
groups <- rep(c("A", "B", "C"), times = 5)
colors <- rep(c("red", "green", "blue"), times = 5)

plot(x_values, y_values,
  col = colors,
  main = "Base R Scatter Plot (Multiple Y per X)",
  sub = "Three groups with repeated measurements",
  xlab = "X Values",
  ylab = "Y Values",
  pch = 19,
  yaxt = "n"
)

# Add y-axis with decimal formatting
axis(2, at = pretty(range(y_values)), labels = scales::label_number(accuracy = 0.1))

# Generate interactive HTML
scatter_html_file <- file.path(output_dir, "example_scatter_plot_base_r.html")
save_html(file = scatter_html_file)

cat("✓ Base R scatter plot example completed\n")
cat("Generated:", scatter_html_file, "\n")

cat("\n=== Base R Scatter + Linear Regression Example ===\n")

# Create a scatter plot with linear regression line
set.seed(42)
x <- 1:30
y <- 3 * x + 10 + rnorm(30, sd = 5)

plot(x, y,
  main = "Base R Scatter with Linear Regression",
  xlab = "X Variable",
  ylab = "Y Variable",
  pch = 19,
  col = "darkblue",
  yaxt = "n"
)

# Add y-axis with decimal formatting
axis(2, at = pretty(range(y)), labels = scales::label_number(accuracy = 0.1))

# Add linear regression line using abline
model <- lm(y ~ x)
abline(model, col = "red", lwd = 2)

# Generate interactive HTML
scatter_lm_html_file <- file.path(output_dir, "example_scatter_linear_regression_base_r.html")
save_html(file = scatter_lm_html_file)

cat("✓ Base R scatter + linear regression example completed\n")
cat("Generated:", scatter_lm_html_file, "\n")

cat("\n=== Base R Scatter + LOESS Smooth Example ===\n")

# Create a scatter plot with LOESS smooth curve
set.seed(42)
x <- seq(0, 10, length.out = 50)
y <- sin(x) * 10 + x * 2 + rnorm(50, sd = 2)

plot(x, y,
  main = "Base R Scatter with LOESS Smooth",
  xlab = "X Variable",
  ylab = "Y Variable",
  pch = 16,
  col = "darkgreen",
  yaxt = "n"
)

# Add y-axis with decimal formatting
axis(2, at = pretty(range(y)), labels = scales::label_number(accuracy = 0.1))

# Add LOESS smooth curve using lines and predict
lo <- loess(y ~ x, span = 0.5)
x_seq <- seq(min(x), max(x), length.out = 100)
y_pred <- predict(lo, x_seq)
lines(x_seq, y_pred, col = "purple", lwd = 3)

# Generate interactive HTML
scatter_loess_html_file <- file.path(output_dir, "example_scatter_loess_smooth_base_r.html")
save_html(file = scatter_loess_html_file)

cat("✓ Base R scatter + LOESS smooth example completed\n")
cat("Generated:", scatter_loess_html_file, "\n")

cat("\n=== Base R Multipanel 2x2 (MFROW) Example ===\n")

# Create a 2x2 multipanel layout using mfrow (row-major)
par(mfrow = c(2, 2))

# Panel 1: Scatter plot
set.seed(123)
scatter_x <- 1:10
scatter_y <- rnorm(10, mean = 10, sd = 2)
plot(scatter_x, scatter_y,
  main = "Scatter Plot",
  xlab = "X Values",
  ylab = "Y Values",
  pch = 19,
  col = "steelblue"
)

# Panel 2: Line plot
line_x <- 1:10
line_y <- c(5, 7, 3, 8, 6, 9, 4, 7, 10, 8)
plot(line_x, line_y,
  type = "l",
  main = "Line Plot",
  xlab = "Time",
  ylab = "Value",
  col = "darkgreen",
  lwd = 2
)

# Panel 3: Bar plot
bar_categories <- c("A", "B", "C", "D")
bar_values <- c(30, 25, 15, 10)
barplot(bar_values,
  names.arg = bar_categories,
  main = "Bar Plot",
  xlab = "Category",
  ylab = "Count",
  col = "coral",
  border = "black"
)

# Panel 4: Histogram
set.seed(456)
hist_data <- rnorm(100, mean = 0, sd = 1)
hist(hist_data,
  main = "Histogram",
  xlab = "Value",
  ylab = "Frequency",
  col = "lightblue",
  border = "black"
)

# Generate interactive HTML
multipanel_2x2_html_file <- file.path(output_dir, "example_multipanel_2x2_mfrow_base_r.html")
save_html(file = multipanel_2x2_html_file)

cat("✓ Base R multipanel 2x2 (mfrow) example completed\n")
cat("Generated:", multipanel_2x2_html_file, "\n")

if (grDevices::dev.cur() > 1) dev.off()

cat("\n=== Base R Multipanel 2x2 (MFCOL) Example ===\n")

# Create a 2x2 multipanel layout using mfcol (column-major)
par(mfcol = c(2, 2))

# Panel 1: Top-Left
set.seed(111)
p1_x <- 1:10
p1_y <- rnorm(10, mean = 10, sd = 2)
plot(p1_x, p1_y,
  main = "Panel 1 (Top-Left)",
  xlab = "X",
  ylab = "Y",
  pch = 19,
  col = "red"
)

# Panel 2: Bottom-Left
set.seed(222)
p2_x <- 1:10
p2_y <- rnorm(10, mean = 15, sd = 2)
plot(p2_x, p2_y,
  main = "Panel 2 (Bottom-Left)",
  xlab = "X",
  ylab = "Y",
  pch = 19,
  col = "blue"
)

# Panel 3: Top-Right
set.seed(333)
p3_x <- 1:10
p3_y <- rnorm(10, mean = 20, sd = 2)
plot(p3_x, p3_y,
  main = "Panel 3 (Top-Right)",
  xlab = "X",
  ylab = "Y",
  pch = 19,
  col = "green"
)

# Panel 4: Bottom-Right
set.seed(444)
p4_x <- 1:10
p4_y <- rnorm(10, mean = 25, sd = 2)
plot(p4_x, p4_y,
  main = "Panel 4 (Bottom-Right)",
  xlab = "X",
  ylab = "Y",
  pch = 19,
  col = "purple"
)

# Generate interactive HTML
multipanel_2x2_mfcol_html_file <- file.path(output_dir, "example_multipanel_2x2_mfcol_base_r.html")
save_html(file = multipanel_2x2_mfcol_html_file)

cat("✓ Base R multipanel 2x2 (mfcol) example completed\n")
cat("Generated:", multipanel_2x2_mfcol_html_file, "\n")

if (grDevices::dev.cur() > 1) dev.off()

cat("\n=== Base R Multipanel 3x2 Example ===\n")

# Create a 3x2 multipanel layout
par(mfrow = c(3, 2))

# Create 6 scatter plots with different colors
set.seed(789)
for (i in 1:6) {
  px <- 1:10
  py <- rnorm(10, mean = i * 5, sd = 2)
  plot(px, py,
    main = paste("Panel", i),
    xlab = "X Values",
    ylab = "Y Values",
    pch = 19,
    col = rainbow(6)[i]
  )
}

# Generate interactive HTML
multipanel_3x2_html_file <- file.path(output_dir, "example_multipanel_3x2_base_r.html")
save_html(file = multipanel_3x2_html_file)

cat("✓ Base R multipanel 3x2 example completed\n")
cat("Generated:", multipanel_3x2_html_file, "\n")

# ============================================================================
# FACET-LIKE EXAMPLES (using par(mfrow) + manual subsetting)
# ============================================================================

cat("\n=== Base R Facet-like Plot: Point Plot by Species (1x3) ===\n")

# Example: Point plot faceted by Species (like ggplot2 facet_wrap)
species_levels <- unique(iris$Species)
par(mfrow = c(1, 3))

for (species in species_levels) {
  subset_data <- iris[iris$Species == species, ]
  plot(subset_data$Petal.Length, subset_data$Petal.Width,
    main = paste("Species:", species),
    xlab = "Petal Length",
    ylab = "Petal Width",
    pch = 19,
    col = "steelblue"
  )
}

# Generate interactive HTML
facet_point_html_file <- file.path(output_dir, "example_facet_point_1x3_base_r.html")
save_html(file = facet_point_html_file)

cat("✓ Base R facet point plot (1x3) example completed\n")
cat("Generated:", facet_point_html_file, "\n")

cat("\n=== Base R Facet-like Plot: Multi-layer by Species (1x3) ===\n")

# Example: Multi-layer facet (scatter + regression line)
par(mfrow = c(1, 3))

for (species in unique(iris$Species)) {
  subset_data <- iris[iris$Species == species, ]

  # Layer 1: Scatter plot
  plot(subset_data$Petal.Length, subset_data$Petal.Width,
    main = paste(species),
    xlab = "Petal Length",
    ylab = "Petal Width",
    pch = 19,
    col = rgb(0.2, 0.4, 0.8, 0.5)
  )

  # Layer 2: Add linear regression line (LOW-level call)
  fit <- lm(Petal.Width ~ Petal.Length, data = subset_data)
  abline(fit, col = "red", lwd = 2)
}

# Generate interactive HTML
facet_multilayer_html_file <- file.path(output_dir, "example_facet_multilayer_1x3_base_r.html")
save_html(file = facet_multilayer_html_file)

cat("✓ Base R facet multi-layer plot (1x3) example completed\n")
cat("Generated:", facet_multilayer_html_file, "\n")

if (grDevices::dev.cur() > 1) dev.off()

cat("\n=== Base R Bar Plot with scales:: Formatting Example ===\n")

# Example: Bar plot with scales::label_dollar for y-axis
# This demonstrates format extraction from scales:: closure

# Revenue data in dollars
revenue <- c(5000, 12000, 8500, 15000, 9500)
names(revenue) <- c("Product A", "Product B", "Product C", "Product D", "Product E")

# Create barplot without y-axis (we'll add it manually with custom formatting)
bp <- barplot(revenue,
  main = "Product Revenue",
  xlab = "Product",
  ylab = "Revenue (USD)",
  col = "steelblue",
  border = "black",
  yaxt = "n" # Suppress default y-axis
)

# Add custom y-axis with scales::label_dollar formatting
# The axis() wrapper will detect the scales:: function and extract format config
axis(2, at = pretty(range(revenue)), labels = scales::label_dollar())

# Generate interactive HTML
format_html_file <- file.path(output_dir, "example_bar_plot_scales_format_base_r.html")
save_html(file = format_html_file)

cat("✓ Base R bar plot with scales:: formatting example completed\n")
cat("Generated:", format_html_file, "\n")

cat("\n=== Base R Bar Plot with scales:: Percent Formatting Example ===\n")

# Example: Bar plot with scales::label_percent for y-axis
percentages <- c(0.15, 0.25, 0.35, 0.18, 0.07)
names(percentages) <- c("Category A", "Category B", "Category C", "Category D", "Category E")

# Create barplot without y-axis
bp2 <- barplot(percentages,
  main = "Market Share by Category",
  xlab = "Category",
  ylab = "Market Share",
  col = "coral",
  border = "black",
  yaxt = "n"
)

# Add custom y-axis with scales::label_percent formatting
axis(2, at = pretty(range(percentages)), labels = scales::label_percent(accuracy = 1))

# Generate interactive HTML
percent_format_html_file <- file.path(output_dir, "example_bar_plot_percent_format_base_r.html")
save_html(file = percent_format_html_file)

cat("✓ Base R bar plot with scales:: percent formatting example completed\n")
cat("Generated:", percent_format_html_file, "\n")
