## ----setup, include=FALSE-----------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  fig.width = 7,
  fig.height = 4,
  dpi = 96,
  message = FALSE,
  warning = FALSE,
  fig.alt = "Figure generated by tikatuwq package"
)

## ----setup-package------------------------------------------------------------
library(tikatuwq)
library(dplyr)

data("wq_demo", package = "tikatuwq")

# Inspect structure
str(wq_demo)
head(wq_demo)

## ----read-data, eval=FALSE----------------------------------------------------
# # Example: reading from a CSV file
# # df <- read_wq("path/to/your/data.csv")

## ----validate-data------------------------------------------------------------
df <- wq_demo

# Validate required columns
df <- validate_wq(df)

# Check structure
str(df)

## ----censored-values----------------------------------------------------------
# Example with censored values
# If your CSV contains values like "<0.01", "<LD", or "ND", read_wq() handles them
# with the nd_policy parameter:

# df_with_nd <- read_wq("data.csv", nd_policy = "ld2")

## ----clean-units--------------------------------------------------------------
# Clean and validate units
df_clean <- clean_units(df)

# If units need conversion (e.g., ug/L to mg/L for phosphorus):
# df_clean <- clean_units(df, units_map = list(p_total = "ug/L"))

## ----compute-iqa--------------------------------------------------------------
# Compute IQA
df_iqa <- iqa(df_clean, na_rm = TRUE)

# View results
head(df_iqa[, c("ponto", "IQA", "IQA_status")])

# Summary
summary(df_iqa$IQA)
table(df_iqa$IQA_status)

## ----compute-iet, eval=FALSE--------------------------------------------------
# # Carlson IET (requires secchi depth, chlorophyll, total phosphorus)
# # df_iet <- iet_carlson(df, .keep_ids = TRUE)
# 
# # Lamparelli IET
# # df_iet <- iet_lamparelli(df, .keep_ids = TRUE)

## ----conama-check-------------------------------------------------------------
# Check compliance for class 2 (default)
df_conama <- conama_check(df_iqa, classe = "2")

# View compliance columns (one per parameter)
head(df_conama[, grep("_ok$", names(df_conama))])

## ----conama-summary-----------------------------------------------------------
# Long-format summary
summary_long <- conama_summary(df_conama, classe = "2")
head(summary_long)

# Report table (violations only, formatted)
report_tab <- conama_report(df_conama, classe = "2", only_violations = TRUE, pretty = TRUE)
print(report_tab)

# Textual summary
summary_text <- conama_text(df_conama, classe = "2", only_violations = TRUE)
cat(summary_text, sep = "\n")

## ----plot-iqa-----------------------------------------------------------------
library(ggplot2)

# Bar plot of IQA by point
p1 <- plot_iqa(df_iqa)
print(p1)

## ----plot-series--------------------------------------------------------------
# Time series of a parameter
p2 <- plot_series(df_iqa, "turbidez", facet = "ponto")
print(p2)

## ----plot-box-----------------------------------------------------------------
# Box plots by point
p3 <- plot_box(df_iqa, "od", by = "ponto")
print(p3)

## ----plot-heatmap-------------------------------------------------------------
library(tidyr)

# Reshape to long format
df_long <- df_iqa %>%
  dplyr::select(data, ponto, turbidez, od, dbo, ph) %>%
  pivot_longer(cols = c(turbidez, od, dbo, ph),
               names_to = "parametro",
               values_to = "valor")

# Heatmap
p4 <- plot_heatmap(df_long)
print(p4)

## ----generate-analysis--------------------------------------------------------
# Generate analytical text
analysis_text <- generate_analysis(
  df_iqa,
  classe_conama = "2",
  incluir_tendencia = FALSE,  # Set TRUE if you have temporal data
  contexto = list(river = "Demo River", period = "2025")
)

cat(paste(analysis_text, collapse = "\n\n"))

## ----render-report, eval=FALSE------------------------------------------------
# # Generate HTML report (requires rmarkdown)
# # report_path <- render_report(
# #   df_iqa,
# #   meta = list(river = "Demo River", period = "2025"),
# #   output_dir = tempdir()
# # )
# #
# # # Open in browser
# # browseURL(report_path)

