/**
 * Authors: Frederik Leyvraz, David Degenhardt
 * License: GNU General Public License v3.0 only
 * Version: 1.0.1
 */

package ch.bfh.ti.latexindexer;

import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.List;

/**
 * Authors: Frederik Leyvraz, David Degenhardt
 * License: GNU General Public License v3.0 only
 */

import static org.junit.jupiter.api.Assertions.*;

public class ParserTest {


    private Path tempFile;
    private IndexWriter indexWriter;

    @BeforeEach
    public void setUp() throws IOException {
        tempFile = Files.createTempFile("testlatex", ".tex");
        List<String> lines = Arrays.asList(
                "\\documentclass{article}",
                "",
                "\\begin{document}",
                "",
                "\\title{Minimal LaTeX Document}",
                "\\author{Your Name}",
                "\\date{\\today}",
                "",
                "\\maketitle",
                "",
                "\\section{Introduction}",
                "        This is a minimal LaTeX document created by ChatGPT.",
                "\\section{Another One}",
                "        This is a second section created by by ChatGPT.",
                "",
                "\\end{document}"
        );

        Files.write(tempFile, lines);
    }

    @AfterEach
    public void tearDown() throws IOException {
        Files.deleteIfExists(tempFile);
    }


    @Test
    void testDetex() throws IOException {

        Parser parser = new DetexParser(tempFile.toString());
        List<Word> words = parser.parseDocument();
        assertFalse(words.isEmpty());
        assertTrue(words.contains(new Word("LaTeX")));

    }

    @Test
    void testPandoc() throws IOException {

        Parser parser = new PandocParser(tempFile.toString());
        List<Word> words = parser.parseDocument();
        assertFalse(words.isEmpty());

        Word latex = new Word("LaTeX");
        Word created = new Word("created", 2);
        assertTrue(words.contains(latex));
        assertTrue(words.contains(created));
        assertEquals(2, words.get(words.indexOf(created)).getFrequency());

    }

    @Test
    void testNoEmptyWords() throws IOException {
        Parser parser = new PandocParser(tempFile.toString());
        List<Word> words = parser.parseDocument();
        for (Word word : words) {
            assertFalse(word.getValue().isEmpty());
        }
    }

}
