## -*-Tcl-*-
 # ###################################################################
 #  Alpha - new Tcl folder configuration
 # 
 #  FILE: "www.tcl"
 #                                    created: 4/9/97 {11:37:57 am} 
 #                                last update: 06/16/1999 {11:55:04 AM} 
 #  Author: Vince Darley
 #  E-mail: <vince@santafe.edu>
 #    mail: Division of Engineering and Applied Sciences, Harvard University
 #          Oxford Street, Cambridge MA 02138, USA
 #     www: <http://www.santafe.edu/~vince/>
 #  
 #  modified by  rev reason
 #  -------- --- --- -----------
 #  4/9/97   VMD 1.0 original
 # ###################################################################
 ##

namespace eval url {}

proc url::parse {url} {
    if {![regexp {^([^:]+)://(.*)$} $url dmy type rest]} {
	alertnote "I couldn't understand that url: '$url'"
	error ""
    }
    return [list $type $rest]
}

proc url::parseFtp {p array} {
    # format is user:pass@host/path
    if {[set at [string first "@" $p]] != -1} {
	# have user etc.
	if {[string first ":" $p] < $at} {
	    # have password
	    regexp {([^:]+):([^@]+)@(.*)$} $p dummy user pass p
	} else {
	    # no password
	    regexp {([^@]+)@(.*)$} dummy user p
	    set pass ""
	}
    } else {
	set user "anonymous"
	if {[catch {set pass [icGetPref Email]}]} {
	    set pass "anonymous"
	}
    }
    regexp {([^/]+)($|/$|/(.*/)([^/]*)$)} $p dummy host dummy path file
    upvar $array a
    array set a [list user $user pass $pass host $host path $path file $file]
}

proc url::store {url file} {
    set t [url::parse $url]
    set type [lindex $t 0]
    set rest [lindex $t 1]	
    switch -- $type {
	"ftp" {
	    url::parseFtp $rest i
	    set i(file) [file tail $file]
	    ftpStore "$file" $i(host) "$i(path)$i(file)" $i(user) $i(pass)
	}
	default {
	    alertnote "Don't know how to put '$type' url's"
	    error ""
	}
    }
}

proc url::fetchFrom {url localdir {file ""}} {
    url::fetch ${url}${file} $localdir $file	
}


## 
 # -------------------------------------------------------------------------
 # 
 # "url::fetch" --
 # 
 #  Get a precise url into a localdir/file.  The url may be a directory,
 #  in which case we retrieve a listing.
 #  
 #  Use url::fetchFrom to fetch a file from a given url-location.
 #  
 #  Note 'Geni' is the sig of a wish applet I wrote which is augmented
 #  with a few procedures to download files via http. 
 #  Of course it needs the user to install Sun's latest
 #  release of Tcl/Tk
 # -------------------------------------------------------------------------
 ##
proc url::fetch {url localdir {file ""}} {
    set t [url::parse $url]
    set type [lindex $t 0]
    set rest [lindex $t 1]
    switch -- $type {
	"ftp" {
	    url::parseFtp $rest i
	    catch {file mkdir [file dirname $localdir]}
	    if {[regexp "/$" "$i(path)$i(file)"]} {
		# directory
		ftpList $localdir$file $i(host) $i(path) $i(user) $i(pass)
	    } else {
		ftpFetch "$localdir$file" $i(host) "$i(path)$i(file)" $i(user) $i(pass)
	    }
	}
	"http" {
	    global httpDownloadSig httpDownloadSigs
	    url::parseFtp $rest i
	    app::launchAnyOfThese $httpDownloadSigs httpDownloadSig
	    if {[file exists "$localdir$file"]} {
		if {[dialog::yesno "Replace $file?"]} {
		    file delete "$localdir$file"
		} else {
		    error "Abort download."
		}
	    }
	    set fid [open "$localdir$file" w]
	    close $fid
	    if {$httpDownloadSig == "Geni"} {
		switchTo '$httpDownloadSig'
		set res [AEBuild -r -t 30000 '$httpDownloadSig' misc dosc ---- \
		  "[list Http_Copy ${url} $localdir$file]"]
		switchTo 'ALFA'
		if {[string match "*Not found*" $res]} {
		    catch {file delete $localdir$file}
		    error "File not found on http server."
		}
	    } else {
		AEBuild -r -t 30000 '$httpDownloadSig' WWW! OURL ---- "${url}" \
		  INTO [makeAlis "$localdir$file"]
	    }
	}
	default {
	    alertnote "Don't know how to fetch '$type' url's"
	    error ""
	}
    }
    return $type
}

## 
 # -------------------------------------------------------------------------
 # 
 # "ftpFetch" --
 # 
 #  Downloads a remote file to your disk. 
 #  
 # -------------------------------------------------------------------------
 ##
proc ftpFetch {localName host path user password} {
    global useTclServiceForFtp
    if {[info exists useTclServiceForFtp] && $useTclServiceForFtp} {
	package require FTP
	FTP::Open $host $user $password
	FTP::Type binary
	FTP::Get $path $localName
	FTP::Close
	return
    }
    global ftpSig ftpSigs
    watchCursor
    app::launchAnyOfThese $ftpSigs ftpSig "Please locate your ftp application:"
    if {$ftpSig == "FTCh"} {
	set localName "[file dirname $localName]:"
    } else {
	set fd [open $localName "w"]
	close $fd
    }
    switch -- $ftpSig {
	Arch -
	FTCh {AEBuild -r -t 30000 '$ftpSig' Arch Ftch FTPh "$host" FTPc "$path" ArGU "$user" ArGp "$password" ---- [makeAlis $localName]}
	Woof {AEBuild -r -t 30000 'Woof' GURL GURL ---- "ftp://${user}:${password}@${host}/${path}" dest [makeAlis $localName]}
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "ftpStore" --
 # 
 #  Uploads a file to a remote ftp server.
 #  
 # -------------------------------------------------------------------------
 ##
proc ftpStore {localName host path user password} {
    global ftpSig ftpSigs
    watchCursor
    app::launchAnyOfThese $ftpSigs ftpSig "Please locate your ftp application:"
    currentReplyHandler ftpHandleReply
    switch -- $ftpSig {
	Arch -
	FTCh {AEBuild -q -t 30000 '$ftpSig' Arch Stor ---- [makeAlis $localName] FTPh "$host" FTPc "$path" ArGU "$user" ArGp "$password"}
	Woof {
	    set path [string range $path 0 [expr {[string last / $path] - 1}]]
	    AEBuild -r -q -t 30000 '$ftpSig' PURL PURL ---- [makeAlis $localName] dest "ftp://${user}:${password}@${host}/${path}"
	}
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "ftpList" --
 # 
 #  Saves the file listing of a remote directory to a file. Uses a trick 
 #  for Fetch when saving the file. First the files are listed in a text
 #  window in Fetch. This window is then saved to the disk.
 #  
 #  This function doesn't work with NetFinder.
 #   
 # -------------------------------------------------------------------------
 ##
proc ftpList {localName host path user password} {
    global useTclServiceForFtp
    if {[info exists useTclServiceForFtp] && $useTclServiceForFtp} {
	package require FTP
	FTP::Open $host $user $password
	FTP::Type binary
	set res [FTP::List $path]
	FTP::Close
	set fd [open $localName "w"]
	puts $fd [join [concat "dummy" $res "dummy"] "\n"]
	close $fd
	return
    }
    global ftpSig
    watchCursor
    app::launchAnyOfThese [list Arch FTCh] ftpSig "Please locate your ftp application:"
    if {[lsearch {Arch FTCh} $ftpSig] < 0} {alertnote "This only works with Anarchie and Fetch."; error ""}
    set fd [open $localName "w"]
    close $fd
    AEBuild -r -t 30000 '$ftpSig' Arch List FTPh "$host" FTPc "$path" ArGU "$user" ArGp "$password" {----} [makeAlis $localName]
    if {$ftpSig == "FTCh"} {
	AEBuild -r -t 30000 'FTCh' FTCh VwFL ---- "obj{want:type(cFWA), from:'null'(), form:name, seld:$host}"
	AEBuild -r -t 30000 'FTCh' core save ---- "obj{want:type(cFWC), from:'null'(), form:indx, seld:long(1)}" kfil [makeAlis $localName]
	AEBuild -r -t 30000 'FTCh' core clos ---- "obj{want:type(cFWC), from:'null'(), form:indx, seld:long(1)}" savo "yes"
    }
}


## 
 # -------------------------------------------------------------------------
 # 
 # "ftpHandleReply" --
 # 
 #  Handles the reply when using ftpStore.
 #  
 # -------------------------------------------------------------------------
 ##
proc ftpHandleReply {reply} {
    set ans [string range $reply 11 end]
    if {[regexp {^errs:([^]+)} $ans dum err]} {
	# Fetch error
	if {[regexp {Error: (.*)} $err dum err2]} {set err $err2}
	alertnote "Ftp error: $err"
    } elseif {[regexp {^'----':(-?[0-9]*)} $ans dum err]} {
	if {$err != "0"} {
	    # Anarchie error.
	    message "Ftp error."
	} else {
	    message "Document uploaded to ftp server."
	}
    } elseif {$ans == "\\\}"} {
	message "Document uploaded to ftp server."
    } else {
	return 0
    }
    return 1
}


## 
 # -------------------------------------------------------------------------
 #	 
 # "GURLHandler" --
 #	
 #  Handle general GURL events by extracting the type 'ftp', 'http', and
 #  calling a procedure ${type}GURLHandler with a single parameter which is
 #  the extracted resource.  Can be put to more general use.  You must
 #  register this proc as an event handler if you want to use it.  Do this
 #  with:
 #   
 #    eventHandler GURL GURL GURLHandler
 #    
 # -------------------------------------------------------------------------
 ##
proc GURLHandler {msg} {
    if {![regsub {.*(.*).*} $msg {\1} gurl]} {
	alertnote "Didn't understand GURL: $msg"
	return
    }
    set GURLtype [lindex [split $gurl ":"] 0]
    set GURLvalue [string range $gurl [expr 1+[string length $GURLtype]] end]
    if {[catch {${GURLtype}GURLHandler $GURLvalue} msg]} {
	message $msg
    }
}


