/*
 * Copyright (c) 2016 Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.sun.ts.tests.jsonp.api.jsonarraytests;

import com.sun.ts.lib.util.TestUtil;
import com.sun.ts.tests.jsonp.api.common.ArrayBuilder;
import com.sun.ts.tests.jsonp.api.common.JsonIO;
import com.sun.ts.tests.jsonp.api.common.JsonValueType;
import com.sun.ts.tests.jsonp.api.common.TestResult;
import javax.json.Json;
import javax.json.JsonArrayBuilder;
import javax.json.JsonValue;

import static com.sun.ts.tests.jsonp.api.common.JsonAssert.*;
import static com.sun.ts.tests.jsonp.api.common.SimpleValues.*;

// $Id: ArrayBuildRemove.java 74753 2016-12-19 11:13:34Z tkraus $
/**
 * JavaScript Object Notation (JSON) compatibility tests:
 * {@link JsonArrayBuilder} API remove() methods added in JSON-P 1.1.<br>
 */
public class ArrayBuildRemove extends ArrayCommon {

    /**
     * Creates an instance of {@link JsonArrayBuilder} API remove() methods added in JSON-P 1.1 test.
     */
    ArrayBuildRemove() {
        super();
    }

    /**
     * {@link JsonArrayBuilder} API remove() methods added in JSON-P 1.1.
     * @return Result of all tests in this suite.
     */
    TestResult test() {
        final TestResult result = new TestResult("JsonArrayBuilder API remove() methods added in JSON-P 1.1.");
        TestUtil.logMsg("JsonArrayBuilder API remove() methods added in JSON-P 1.1.");
        testRemove(result);
        testRemoveOutOfBounds(result);
        return result;
    }
    
    /**
     * Test {@code default JsonArrayBuilder remove(int, Object)} method on {@code String} array.
     * @param result Test suite result.
     */
    private void testRemove(final TestResult result) {
        final Object[] values = new Object[] {
            OBJ_VALUE,    // remove(int,JsonValue)
            STR_VALUE,    // remove(int,String)
            INT_VALUE,    // remove(int,int)
            LNG_VALUE,    // remove(int,long)
            DBL_VALUE,    // remove(int,double)
            BIN_VALUE,    // remove(int,BigInteger)
            BDC_VALUE,    // remove(int,BigDecimal)
            BOOL_VALUE    // remove(int,boolean)
        };
        for (Object value : values) {
            final String typeName = JsonValueType.getType(value).name();
            TestUtil.logMsg(" - remove(int,"+typeName+")");
            final String json = "[]";
            final JsonValue check = JsonIO.read(json);
            JsonArrayBuilder builder = ArrayBuilder.add(Json.createArrayBuilder(), value);
            builder = updateOperationBuilder(builder, 0, null);
            final JsonValue out = builder.build();
            if (operationFailed(check, out)) {
                result.fail("remove(" + typeName + ")",
                        "Builder output " + valueToString(out) + " value shall be " + valueToString(check));
            }
        }
    }

    /**
     * Test {@code default JsonArrayBuilder remove(int, Object)} method on {@code String} array with index being
     * out of range ({@code index < 0 || index > array size}).
     * @param result Test suite result.
     */
    private void testRemoveOutOfBounds(final TestResult result) {
        final Object[] values = new Object[] {
            OBJ_VALUE,    // remove(int,JsonValue)
            STR_VALUE,    // remove(int,String)
            INT_VALUE,    // remove(int,int)
            LNG_VALUE,    // remove(int,long)
            DBL_VALUE,    // remove(int,double)
            BIN_VALUE,    // remove(int,BigInteger)
            BDC_VALUE,    // remove(int,BigDecimal)
            BOOL_VALUE    // remove(int,boolean)
        };
        final int[] indexes = new int[] {-1, 2, 3};
        for (Object value : values) {
            final String typeName = JsonValueType.getType(value).name();
            TestUtil.logMsg(" - remove(int,"+typeName+")");
            final String json = "[" + JsonValueType.toStringValue(value) + "]";
            // Add value into the array for the first time to het array of size 1.
            JsonArrayBuilder builder = ArrayBuilder.add(Json.createArrayBuilder(), value);
            for (int index : indexes) {
                try {
                    // Add value on out of bounds index
                    builder = updateOperationBuilder(builder, index, null);
                    result.fail("remove(int,"+typeName+")",
                            "Calling method with out of bounds index="
                            + index + " argument shall throw IndexOutOfBoundsException");
                } catch (IndexOutOfBoundsException e) {
                TestUtil.logMsg("    - Expected exception for index="+index+": "+e.getMessage());
                } catch (Throwable t) {
                    result.fail("remove(int,("+typeName+")null)",
                            "Calling method with with out of bounds index="
                            + index + " argument shall throw IndexOutOfBoundsException, not "
                            + t.getClass().getSimpleName());
                }
            }
        }
    }

    /**
     * Create and initialize array builder.
     * Unsupported method call for remove() method.
     * @param value JSON value stored in the builder. Value of {@code null} is stored as JSON {@code null} keyword.
     * @return JSON array builder containing value.
     */
    @Override
    protected JsonArrayBuilder createArrayBuilder(Object value) {
        throw new UnsupportedOperationException("Method remove is not implemented.");
    }

    /**
     * Create and initialize array builder.
     * Unsupported method call for remove() method.
     * @param index Position in the array where value is added.
     * @param value JSON value stored in the builder. Value of {@code null} is stored as JSON {@code null} keyword.
     * @return JSON array builder containing value.
     */
    @Override
    protected JsonArrayBuilder createArrayBuilder(final int index, final Object value) {
        throw new UnsupportedOperationException("Method remove is not implemented.");
    }

    /**
     * Update array builder.
     * Unsupported method call for remove() method.
     * @param builder JSON array builder to update.
     * @param value JSON value stored in the builder. Value of {@code null} is stored as JSON {@code null} keyword.
     * @return JSON array builder with value updated.
     */
    @Override
    protected JsonArrayBuilder updateOperationBuilder(JsonArrayBuilder builder, Object value) {
        throw new UnsupportedOperationException("Method remove is not implemented.");
    }

    /**
     * Update array builder with value removal at specified index.
     * Child class callback.
     * @param builder JSON array builder to update.
     * @param index Position in the array where value is added.
     * @param value JSON value argument is ignored.
     * @return JSON array builder with value updated.
     */
    @Override
    protected JsonArrayBuilder updateOperationBuilder(
            final JsonArrayBuilder builder, final int index, final Object value) {
        return ArrayBuilder.remove(builder, index);
    }

}
