/*
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.lsat.scheduler.product;

import static org.slf4j.LoggerFactory.getLogger;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.tuple.ImmutablePair;
import org.eclipse.lsat.common.graph.directed.Node;
import org.eclipse.lsat.common.scheduler.graph.Task;
import org.eclipse.lsat.common.scheduler.schedule.Schedule;
import org.eclipse.lsat.common.scheduler.schedule.ScheduledTask;
import org.eclipse.lsat.product.productdata.Action;
import org.eclipse.lsat.product.productdata.EntryEvent;
import org.eclipse.lsat.product.productdata.ExitEvent;
import org.eclipse.lsat.product.productdata.LocationInformation;
import org.eclipse.lsat.product.productdata.ProductInstance;
import org.eclipse.lsat.product.productdata.TransferEvent;
import org.slf4j.Logger;

import activity.Move;
import activity.PeripheralAction;
import lsat_graph.PeripheralActionTask;

public class ProductUtil {
    /**
     *
     */
    private static final String SEPARATOR = System.lineSeparator() + "=".repeat(80);

    private static final Logger LOGGER = getLogger(ProductUtil.class);

    private ProductUtil() {
        // No instances allowed
    }

    public static List<ProductChange> createProductChanges(Schedule<Task> schedule) {
        var products = new ArrayList<ProductChange>();
        var peripheralActions = schedule.allNodesInTopologicalOrder().stream() //
                .map(s -> ImmutablePair.of(s, getAction(s))) //
                .filter(e -> e.right != null).toList();
        int numDigits = String.valueOf(peripheralActions.stream().map(ImmutablePair::getRight).distinct().count() - 1).length();
        var actionDataMap = peripheralActions.stream().map(ImmutablePair::getRight).distinct()
                .collect(Collectors.toMap(a -> a, ProductUtil::createActionData));
        var nameCount = new LinkedHashMap<String, Integer>();
        for (var entry: peripheralActions) {
            var scheduledTask = entry.left;
            var peripheralAction = entry.right;
            var peripheralActionData = actionDataMap.get(peripheralAction);
            var startTime = scheduledTask.getStartTime();
            var endTime = scheduledTask.getEndTime();
            var taskAction = getAction(scheduledTask);
            var index = nameCount.computeIfAbsent(scheduledTask.getName(), (v) -> 0);
            nameCount.put(scheduledTask.getName(), index + 1);
            var actionName = taskAction.getGraph().getName() + "." + scheduledTask.getName();
            var actionId = String.format("%0" + numDigits + "d%s%s", index, "#", actionName);
            var taskData = new org.eclipse.lsat.product.productdata.Task(scheduledTask, actionId, startTime, endTime,
                    peripheralActionData);

            if (!peripheralAction.getProductChanges().isEmpty()) {
                for (var productChange: peripheralAction.getProductChanges()) {
                    var productContext = new ProductChange(scheduledTask, taskData, productChange);
                    products.add(productContext);
                }
            } else {
                var productContext = new ProductChange(scheduledTask, taskData, null);
                products.add(productContext);
            }
        }
        return products;
    }

    public static PeripheralAction getAction(Node node) {
        if (node instanceof ScheduledTask<?> scheduledTask) {
            node = scheduledTask.getTask();
        }
        if (node instanceof PeripheralActionTask pTask) {
            if (pTask.getAction() instanceof PeripheralAction pAction) {
                return pAction;
            }
        }
        return null;
    }

    /**
     * Post-processing the schedule and the product information to infer the symbolic locations of the products at each
     * event in the life cycle. Currently decided against using this, keeping the method for later.
     *
     * @param initialScheduledTask - The scheduled task of the product
     * @param schedule - The complete schedule
     * @return The symbolic location of the product
     */
    public static String getSymbolicLocation(ScheduledTask<Task> initialScheduledTask, Schedule<Task> schedule) {
        var orderedMoveList = schedule.allNodesInTopologicalOrder().stream()
                .filter(scheduledTask -> scheduledTask.getTask() instanceof PeripheralActionTask)
                .filter(scheduledTask -> ((PeripheralActionTask)scheduledTask.getTask()).getAction() instanceof Move)
                .collect(Collectors.toList());

        var scheduledTaskMove = orderedMoveList.stream()
                .filter(scheduledTask -> ((PeripheralActionTask)scheduledTask.getTask()).getAction().fqn()
                        .equals(((PeripheralActionTask)initialScheduledTask.getTask()).getAction().fqn()))
                .filter(scheduledTask -> scheduledTask.getStartTime().doubleValue() > initialScheduledTask
                        .getStartTime().doubleValue())
                .findFirst();

        if (scheduledTaskMove.isEmpty()) {
            return "N/A";
        }
        var move = (Move)((PeripheralActionTask)scheduledTaskMove.get().getTask()).getAction();
        return move.getSourcePosition().getName();
    }

    public static <T extends Node> boolean isSuccessor(T start, T target) {
        if (start == null || target == null) {
            return false;
        }

        // Use a helper method with a visited set to prevent cycles
        return isSuccessor(start, target, new LinkedHashSet<>());
    }

    private static <T extends Node> boolean isSuccessor(T current, T target, Set<T> visited) {
        // Base case: found the target
        if (current.equals(target)) {
            return true;
        }

        // Mark current node as visited
        if (visited.add(current)) {
            // Check all successors recursively
            @SuppressWarnings("unchecked")
            List<T> successors = current.getOutgoingEdges().stream().map(e -> (T)e.getTargetNode()).toList();

            for (T successor: successors) {
                if (isSuccessor(successor, target, visited)) {
                    return true;
                }
            }
        }

        // Target not found in any path from current node
        return false;
    }

    public static void logProductLocations(Map<ProductInstance, ProductChange> activeProducts) {
        if (activeProducts.isEmpty()) {
            return;
        }
        LOGGER.info(SEPARATOR + System.lineSeparator() + getProductLocationsString(activeProducts) + SEPARATOR);
    }

    public static void logProductFlow(Collection<ProductInstance> instances) {
        LOGGER.info(SEPARATOR + System.lineSeparator() + getProductFlowString(instances) + SEPARATOR);
    }

    public static String getProductLocationsString(Map<ProductInstance, ProductChange> activeProducts) {
        if (activeProducts.isEmpty()) {
            return "";
        }

        StringBuilder sb = new StringBuilder();
        sb.append(String.format("%-20s, %s", "ProductId", "Owner")).append(System.lineSeparator());

        sb.append(activeProducts.entrySet().stream() //
                .map(e -> String.format("%-20s, %s%s", e.getKey().getProductID(),
                        e.getValue().getProductOwner().getPeripheral(),
                        e.getValue().getSlot().isEmpty() ? "" : "." + e.getValue().getSlot()))
                .collect(Collectors.joining(System.lineSeparator())));

        return sb.toString();
    }

    public static String getProductFlowString(Collection<ProductInstance> instances) {
        StringBuilder sb = new StringBuilder();
        sb.append(String.format("%-9s, %-7s, %s", "ProductId", "Ready", "Flow"));

        for (var instance: instances) {
            var locs = instance.getLifeCycle().stream().map(ProductUtil::getLocationInformation).map(lis -> {
                var slot = lis.stream() //
                        .map(LocationInformation::getSlot) //
                        .distinct().collect(Collectors.joining(","));
                return lis.stream() //
                        .map(li -> li.getTask().getAction().getPeripheral()) //
                        .collect(Collectors.joining(" -" + slot + "-> ", "[", "]"));
            }).toList();
            sb.append(System.lineSeparator());
            sb.append(String.format("%9s, %-7s, %s", instance.getProductID(),
                    instance.getLifeCycle().getLast() instanceof ExitEvent ? "yes" : "no", String.join("->", locs)));
        }

        return sb.toString();
    }

    /** find the product that has not yet passed the slot */
    public static ProductInstance getProductInstanceThatHasNotPassedSlot(
            Map<ProductInstance, ProductChange> activeProducts, ProductOwner productOwner, String slot)
    {
        var result = activeProducts.entrySet().stream().filter(e -> e.getValue().getProductOwner().equals(productOwner)) //
                .map(Map.Entry::getKey) //
                // check if the product already passed
                .filter(pi -> slot.isEmpty() || !pi.getLifeCycle().stream().map(ProductUtil::getLocationInformation)
                        .flatMap(Collection::stream).anyMatch(l -> l.getSlot().equals(slot)))
                .findFirst().orElse(null);
        if (result == null) {
            var action = productOwner.getAction();
            var activity = action.getGraph();
            throw new IllegalArgumentException("Action: '" + activity.getName() + "." + action.getName()
                    + "' cannot find an existing product owner for " + productOwner + " using slot: '" + slot + "'");
        }
        return result;
    }

    public static ProductInstance getProductInstanceForSlot(Map<ProductInstance, ProductChange> activeProducts,
            ProductOwner owner, String slot)
    {
        var productInstance = activeProducts.entrySet().stream() //
                .filter(e -> e.getValue().getProductOwner().equals(owner)) //
                .filter(e -> slot == null || slot.isEmpty() || e.getValue().getSlot().equals(slot)) //
                .map(Map.Entry::getKey).reduce((a, b) -> b).orElse(null); // get last or else null
        return productInstance;
    }

    private static Action createActionData(PeripheralAction peripheralAction) {
        var activityName = peripheralAction.getGraph().getName();
        var actionName = peripheralAction.getName();
        return new Action(activityName, actionName, peripheralAction.fqn());
    }

    public static List<LocationInformation> getLocationInformation(Object event) {
        return switch (event) {
            case EntryEvent entry -> List.of(entry.getEntry());
            case ExitEvent exit -> List.of(exit.getExit());
            case TransferEvent transfer -> List.of(transfer.getTransfer().getOut(), transfer.getTransfer().getIn());
            default -> List.of();
        };
    }
}
