/**
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.lsat.timing.util;

import activity.Move;
import activity.SimpleAction;
import com.google.common.collect.Iterables;
import distributions.CalculationMode;
import distributions.Distribution;
import distributions.DistributionsAdapter;
import expressions.BigDecimalConstant;
import expressions.Expression;
import java.math.BigDecimal;
import java.math.MathContext;
import java.math.RoundingMode;
import java.util.Arrays;
import java.util.Collection;
import java.util.IdentityHashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Function;
import machine.ActionType;
import machine.Axis;
import machine.IResource;
import machine.Peripheral;
import machine.Profile;
import machine.SetPoint;
import org.apache.commons.math3.random.JDKRandomGenerator;
import org.apache.commons.math3.random.RandomGenerator;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.lsat.common.graph.directed.editable.Node;
import org.eclipse.lsat.common.util.IterableUtil;
import org.eclipse.lsat.common.xtend.annotations.IntermediateProperty;
import org.eclipse.lsat.motioncalculator.MotionException;
import org.eclipse.lsat.motioncalculator.MotionProfile;
import org.eclipse.lsat.motioncalculator.MotionProfileParameter;
import org.eclipse.lsat.motioncalculator.MotionSegment;
import org.eclipse.lsat.motioncalculator.PositionInfo;
import org.eclipse.lsat.motioncalculator.util.MotionSegmentUtilities;
import org.eclipse.lsat.timing.calculator.MotionCalculatorExtension;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.Functions.Function2;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.MapExtensions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.eclipse.xtext.xbase.lib.XbaseGenerated;
import setting.MoveAdjustments;
import setting.PhysicalSettings;
import setting.Settings;
import timing.Array;
import timing.Scalar;

@SuppressWarnings("all")
public class TimingCalculator implements ITimingCalculator {
  private static final MathContext MICRO_SECONDS = new MathContext(6, RoundingMode.HALF_UP);

  private static final double HALF_MICRO_SECOND = (0.5 * Math.pow(10, (-TimingCalculator.MICRO_SECONDS.getPrecision())));

  @IntermediateProperty(Array.class)
  private final Map<Array, Integer> _IntermediateProperty_pointer = new java.util.WeakHashMap<>();

  private final RandomGenerator distributionRandom = new JDKRandomGenerator(1618033989);

  private final MotionCalculatorHelper motionCalculatorHelper;

  private final CalculationMode mode;

  private final boolean synchronizeAxes;

  private final Map<Node, BigDecimal> nodeTimes;

  private final Map<Move, Map<SetPoint, BigDecimal>> motionTimes;

  private final DistributionsAdapter distributions;

  private final Map<Node, Map<String, MotionData>> motionData;

  public TimingCalculator(final Settings settings, final MotionCalculatorExtension motionCalculator) {
    this(settings, motionCalculator, CalculationMode.MEAN, true, true);
  }

  public TimingCalculator(final Settings settings, final MotionCalculatorExtension motionCalculator, final CalculationMode mode) {
    this(settings, motionCalculator, mode, true, true);
  }

  public TimingCalculator(final Settings settings, final MotionCalculatorExtension motionCalculator, final CalculationMode mode, final boolean synchronizeAxes, final boolean useCache) {
    MotionCalculatorHelper _motionCalculatorHelper = new MotionCalculatorHelper(settings, motionCalculator);
    this.motionCalculatorHelper = _motionCalculatorHelper;
    this.mode = mode;
    this.synchronizeAxes = synchronizeAxes;
    IdentityHashMap<Node, BigDecimal> _xifexpression = null;
    if ((useCache && Objects.equals(mode, CalculationMode.MEAN))) {
      _xifexpression = new IdentityHashMap<Node, BigDecimal>();
    }
    this.nodeTimes = _xifexpression;
    IdentityHashMap<Move, Map<SetPoint, BigDecimal>> _xifexpression_1 = null;
    if (useCache) {
      _xifexpression_1 = new IdentityHashMap<Move, Map<SetPoint, BigDecimal>>();
    }
    this.motionTimes = _xifexpression_1;
    this.distributions = DistributionsAdapter.getAdapter(settings);
    IdentityHashMap<Node, Map<String, MotionData>> _xifexpression_2 = null;
    if (useCache) {
      _xifexpression_2 = new IdentityHashMap<Node, Map<String, MotionData>>();
    }
    this.motionData = _xifexpression_2;
    this.distributions.setCalculationMode(mode);
    this.distributions.setRandomGenerator(this.distributionRandom);
  }

  public Settings getSettings() {
    return this.motionCalculatorHelper.getSettings();
  }

  public MotionCalculatorExtension getMotionCalculator() {
    return this.motionCalculatorHelper.getMotionCalculator();
  }

  @Override
  public void reset() {
    this._IntermediateProperty_pointer.clear();
    if ((this.nodeTimes != null)) {
      this.nodeTimes.clear();
    }
    if ((this.motionTimes != null)) {
      this.motionTimes.clear();
    }
    Resource _eResource = this.getSettings().eResource();
    ResourceSet _resourceSet = null;
    if (_eResource!=null) {
      _resourceSet=_eResource.getResourceSet();
    }
    DistributionsAdapter.resetDistributions(_resourceSet);
    if ((this.motionData != null)) {
      this.motionData.clear();
    }
  }

  @Override
  public BigDecimal calculateDuration(final Node node) throws MotionException {
    BigDecimal _xifexpression = null;
    if ((null == this.nodeTimes)) {
      _xifexpression = this.doCalculateDuration(node);
    } else {
      final Function<Node, BigDecimal> _function = (Node it) -> {
        return this.doCalculateDuration(it);
      };
      _xifexpression = this.nodeTimes.computeIfAbsent(node, _function);
    }
    return _xifexpression;
  }

  protected BigDecimal _doCalculateDuration(final Node node) {
    return BigDecimal.ZERO;
  }

  protected BigDecimal _doCalculateDuration(final SimpleAction action) {
    return this.doCalculateValue(action.getType(), action.getResource(), action.getPeripheral());
  }

  protected BigDecimal doCalculateValue(final ActionType actionType, final IResource resource, final Peripheral peripheral) {
    try {
      return this.doCalculateValue(this.getPhysicalSettings(resource, peripheral).getTimingSettings().get(actionType));
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  protected BigDecimal _doCalculateValue(final Scalar scalar) {
    return this.doCalculateValue(scalar.getValueExp());
  }

  protected BigDecimal _doCalculateValue(final Array array) {
    BigDecimal _switchResult = null;
    final CalculationMode mode = this.mode;
    if (mode != null) {
      switch (mode) {
        case LINEAIR:
          BigDecimal _xblockexpression = null;
          {
            Integer _pointer = this.getPointer(array);
            int _plus = ((_pointer).intValue() + 1);
            int _size = array.getValues().size();
            int _modulo = (_plus % _size);
            this.setPointer(array, Integer.valueOf(_modulo));
            _xblockexpression = this.doCalculateValue(array.getValuesExp().get((this.getPointer(array)).intValue()));
          }
          _switchResult = _xblockexpression;
          break;
        default:
          final Function1<Expression, BigDecimal> _function = (Expression it) -> {
            return this.doCalculateValue(it);
          };
          _switchResult = TimingCalculator.getAverage(ListExtensions.<Expression, BigDecimal>map(array.getValuesExp(), _function));
          break;
      }
    } else {
      final Function1<Expression, BigDecimal> _function = (Expression it) -> {
        return this.doCalculateValue(it);
      };
      _switchResult = TimingCalculator.getAverage(ListExtensions.<Expression, BigDecimal>map(array.getValuesExp(), _function));
    }
    return _switchResult;
  }

  protected BigDecimal _doCalculateValue(final Distribution distribution) {
    return TimingCalculator.normalize(distribution.evaluate());
  }

  protected BigDecimal _doCalculateValue(final Expression expression) {
    return expression.evaluate();
  }

  protected BigDecimal _doCalculateDuration(final Move move) {
    try {
      return IterableUtil.<BigDecimal>max(this.calculateMotionTime(move).values(), BigDecimal.ZERO);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  @Override
  public Map<SetPoint, BigDecimal> calculateMotionTime(final Move move) throws MotionException {
    Map<SetPoint, BigDecimal> motionTime = null;
    if ((null == this.motionTimes)) {
      motionTime = this.doCalculateMotionTimes(move).get(move);
    } else {
      boolean _containsKey = this.motionTimes.containsKey(move);
      boolean _not = (!_containsKey);
      if (_not) {
        this.motionTimes.putAll(this.doCalculateMotionTimes(move));
      }
      motionTime = this.motionTimes.get(move);
    }
    final String profileName = move.getProfile().getName();
    final LinkedHashMap<SetPoint, BigDecimal> motionTimeCopy = CollectionLiterals.<SetPoint, BigDecimal>newLinkedHashMap();
    final PhysicalSettings physicalSettings = this.motionCalculatorHelper.getSettings().getPhysicalSettings(move.getResource(), move.getPeripheral());
    final Function1<MoveAdjustments, Boolean> _function = (MoveAdjustments it) -> {
      return Boolean.valueOf(((it.getProfile() != null) && Objects.equals(it.getProfile().getName(), profileName)));
    };
    MoveAdjustments adjustments = IterableExtensions.<MoveAdjustments>findFirst(physicalSettings.getMoveAdjustments(), _function);
    if ((adjustments == null)) {
      final Function1<MoveAdjustments, Boolean> _function_1 = (MoveAdjustments it) -> {
        Profile _profile = it.getProfile();
        return Boolean.valueOf((_profile == null));
      };
      adjustments = IterableExtensions.<MoveAdjustments>findFirst(physicalSettings.getMoveAdjustments(), _function_1);
    }
    if ((adjustments != null)) {
      Expression _expression = adjustments.getTimeDeclaration().getExpression();
      final BigDecimalConstant bdConst = ((BigDecimalConstant) _expression);
      try {
        Set<Map.Entry<SetPoint, BigDecimal>> _entrySet = motionTime.entrySet();
        for (final Map.Entry<SetPoint, BigDecimal> element : _entrySet) {
          {
            bdConst.setValue(element.getValue());
            motionTimeCopy.put(element.getKey(), adjustments.getAdjustExpression().evaluate());
          }
        }
      } finally {
        bdConst.setValue(BigDecimal.ZERO);
      }
      return motionTimeCopy;
    }
    return motionTime;
  }

  @Override
  public Map<String, MotionData> calculateMotionData(final Node node) throws MotionException {
    if ((null == this.motionData)) {
      return this.doCalculateMotionData(node).get(node);
    }
    boolean _containsKey = this.motionData.containsKey(node);
    boolean _not = (!_containsKey);
    if (_not) {
      this.motionData.putAll(this.doCalculateMotionData(node));
    }
    return this.motionData.get(node);
  }

  /**
   * Compute the duration of each segment in the (concatenated) move
   */
  protected Map<Move, ? extends Map<SetPoint, BigDecimal>> doCalculateMotionTimes(final Move move) {
    try {
      Map<EList<Axis>, ? extends List<SetPoint>> _xifexpression = null;
      if (this.synchronizeAxes) {
        EList<Axis> _axes = move.getPeripheral().getType().getAxes();
        EList<SetPoint> _setPoints = move.getPeripheral().getType().getSetPoints();
        Pair<EList<Axis>, EList<SetPoint>> _mappedTo = Pair.<EList<Axis>, EList<SetPoint>>of(_axes, _setPoints);
        _xifexpression = CollectionLiterals.<EList<Axis>, EList<SetPoint>>newHashMap(_mappedTo);
      } else {
        final Function1<SetPoint, EList<Axis>> _function = (SetPoint it) -> {
          return it.getAxes();
        };
        _xifexpression = IterableExtensions.<EList<Axis>, SetPoint>groupBy(move.getPeripheral().getType().getSetPoints(), _function);
      }
      final Map<EList<Axis>, ? extends List<SetPoint>> axesAndSetPoints = _xifexpression;
      final List<Move> concatenatedMove = this.motionCalculatorHelper.getConcatenatedMove(move);
      int _size = concatenatedMove.size();
      final IdentityHashMap<Move, IdentityHashMap<SetPoint, BigDecimal>> result = new IdentityHashMap<Move, IdentityHashMap<SetPoint, BigDecimal>>(_size);
      Set<? extends Map.Entry<EList<Axis>, ? extends List<SetPoint>>> _entrySet = axesAndSetPoints.entrySet();
      for (final Map.Entry<EList<Axis>, ? extends List<SetPoint>> e : _entrySet) {
        {
          final List<MotionSegment> motionSegments = this.motionCalculatorHelper.createMotionSegments(concatenatedMove, e.getKey());
          final List<Double> motionTimes = this.motionCalculatorHelper.getMotionCalculator().calculateTimes(motionSegments);
          BigDecimal startTime = BigDecimal.ZERO;
          for (int i = 0; (i < motionTimes.size()); i++) {
            {
              final BigDecimal endTime = TimingCalculator.normalize(motionTimes.get(i));
              final Function<Move, IdentityHashMap<SetPoint, BigDecimal>> _function_1 = (Move it) -> {
                int _size_1 = move.getPeripheral().getType().getSetPoints().size();
                return new IdentityHashMap<SetPoint, BigDecimal>(_size_1);
              };
              TimingCalculator.<SetPoint, BigDecimal>fill(result.computeIfAbsent(concatenatedMove.get(i), _function_1), e.getValue(), endTime.subtract(startTime));
              startTime = endTime;
            }
          }
        }
      }
      return result;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  protected Map<? extends Node, ? extends Map<String, MotionData>> _doCalculateMotionData(final Node node) {
    final IdentityHashMap<Node, IdentityHashMap<String, MotionData>> result = new IdentityHashMap<Node, IdentityHashMap<String, MotionData>>();
    IdentityHashMap<String, MotionData> _identityHashMap = new IdentityHashMap<String, MotionData>();
    result.put(node, _identityHashMap);
    return result;
  }

  /**
   * Compute the motion data (position, velocity, acceleration, etc.) over time for the given move.
   */
  protected Map<? extends Node, ? extends Map<String, MotionData>> _doCalculateMotionData(final Move move) {
    try {
      Map<EList<Axis>, ? extends List<SetPoint>> _xifexpression = null;
      if (this.synchronizeAxes) {
        EList<Axis> _axes = move.getPeripheral().getType().getAxes();
        EList<SetPoint> _setPoints = move.getPeripheral().getType().getSetPoints();
        Pair<EList<Axis>, EList<SetPoint>> _mappedTo = Pair.<EList<Axis>, EList<SetPoint>>of(_axes, _setPoints);
        _xifexpression = CollectionLiterals.<EList<Axis>, EList<SetPoint>>newHashMap(_mappedTo);
      } else {
        final Function1<SetPoint, EList<Axis>> _function = (SetPoint it) -> {
          return it.getAxes();
        };
        _xifexpression = IterableExtensions.<EList<Axis>, SetPoint>groupBy(move.getPeripheral().getType().getSetPoints(), _function);
      }
      final Map<EList<Axis>, ? extends List<SetPoint>> axesAndSetPoints = _xifexpression;
      final List<Move> concatenatedMove = this.motionCalculatorHelper.getConcatenatedMove(move);
      final IdentityHashMap<Move, IdentityHashMap<String, MotionData>> result = new IdentityHashMap<Move, IdentityHashMap<String, MotionData>>();
      Set<? extends Map.Entry<EList<Axis>, ? extends List<SetPoint>>> _entrySet = axesAndSetPoints.entrySet();
      for (final Map.Entry<EList<Axis>, ? extends List<SetPoint>> e : _entrySet) {
        {
          final List<MotionSegment> motionSegments = this.motionCalculatorHelper.createMotionSegments(concatenatedMove, e.getKey());
          final MotionProfile motionProfile = MotionSegmentUtilities.getMotionProfiles(motionSegments).iterator().next();
          final Function1<MotionProfileParameter, String> _function_1 = (MotionProfileParameter it) -> {
            return it.getName();
          };
          final List<String> parameterNames = IterableExtensions.<String>toList(IterableExtensions.<MotionProfileParameter, String>map(motionProfile.getParameters(), _function_1));
          Collection<PositionInfo> _positionInfo = this.motionCalculatorHelper.getMotionCalculator().getPositionInfo(motionSegments);
          for (final PositionInfo positionInfo : _positionInfo) {
            {
              final List<double[]> allData = positionInfo.getAllData();
              BigDecimal startTime = BigDecimal.ZERO;
              for (final Move moveSegment : concatenatedMove) {
                {
                  final double startTimeD = startTime.doubleValue();
                  final Function2<SetPoint, BigDecimal, Boolean> _function_2 = (SetPoint key, BigDecimal value) -> {
                    String _name = key.getName();
                    String _setPointId = positionInfo.getSetPointId();
                    return Boolean.valueOf(Objects.equals(_name, _setPointId));
                  };
                  BigDecimal _get = ((BigDecimal[])Conversions.unwrapArray(MapExtensions.<SetPoint, BigDecimal>filter(this.calculateMotionTime(moveSegment), _function_2).values(), BigDecimal.class))[0];
                  final double endTimeD = startTime.add(_get).doubleValue();
                  final Function1<double[], Boolean> _function_3 = (double[] array) -> {
                    return Boolean.valueOf(((array[0] >= (startTimeD - TimingCalculator.HALF_MICRO_SECOND)) && (array[0] <= (endTimeD + TimingCalculator.HALF_MICRO_SECOND))));
                  };
                  final Function1<double[], double[]> _function_4 = (double[] array) -> {
                    double[] _xblockexpression = null;
                    {
                      double _get_1 = array[0];
                      double _minus = (_get_1 - startTimeD);
                      array[0] = _minus;
                      _xblockexpression = array;
                    }
                    return _xblockexpression;
                  };
                  final List<double[]> filteredData = IterableExtensions.<double[]>toList(IterableExtensions.<double[], double[]>map(IterableExtensions.<double[]>filter(allData, _function_3), _function_4));
                  MotionData motionData = new MotionData();
                  motionData.setTimeData(filteredData);
                  List<String> _asList = IterableUtil.<String>asList("Time", "Position");
                  motionData.setParameterNames(IterableExtensions.<String>toList(Iterables.<String>concat(_asList, parameterNames)));
                  final Function<Move, IdentityHashMap<String, MotionData>> _function_5 = (Move it) -> {
                    return new IdentityHashMap<String, MotionData>();
                  };
                  result.computeIfAbsent(moveSegment, _function_5).put(positionInfo.getSetPointId(), motionData);
                  BigDecimal _startTime = startTime;
                  final Function2<SetPoint, BigDecimal, Boolean> _function_6 = (SetPoint key, BigDecimal value) -> {
                    String _name = key.getName();
                    String _setPointId = positionInfo.getSetPointId();
                    return Boolean.valueOf(Objects.equals(_name, _setPointId));
                  };
                  BigDecimal _get_1 = ((BigDecimal[])Conversions.unwrapArray(MapExtensions.<SetPoint, BigDecimal>filter(this.calculateMotionTime(moveSegment), _function_6).values(), BigDecimal.class))[0];
                  startTime = _startTime.add(_get_1);
                }
              }
            }
          }
        }
      }
      return result;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  protected PhysicalSettings getPhysicalSettings(final IResource resource, final Peripheral peripheral) throws SpecificationException {
    final PhysicalSettings physicalSettings = this.getSettings().getPhysicalSettings(resource, peripheral);
    if ((null == physicalSettings)) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Physical settings not specified for peripheral: ");
      String _fqn = peripheral.fqn();
      _builder.append(_fqn);
      Settings _settings = this.getSettings();
      throw new SpecificationException(_builder.toString(), _settings);
    }
    return physicalSettings;
  }

  /**
   * Fills the map with the specified value for all specified keys.
   */
  public static <K extends Object, V extends Object> void fill(final Map<K, V> map, final Collection<? extends K> keys, final V value) {
    final Consumer<K> _function = (K it) -> {
      map.put(it, value);
    };
    keys.forEach(_function);
  }

  public static BigDecimal getAverage(final Collection<BigDecimal> values) {
    BigDecimal sum = IterableExtensions.<BigDecimal>head(values);
    Iterable<BigDecimal> _tail = IterableExtensions.<BigDecimal>tail(values);
    for (final BigDecimal value : _tail) {
      BigDecimal _sum = sum;
      sum = _sum.add(value);
    }
    int _size = values.size();
    BigDecimal _bigDecimal = new BigDecimal(_size);
    return sum.divide(_bigDecimal, TimingCalculator.MICRO_SECONDS);
  }

  public static BigDecimal normalize(final Number number) {
    final BigDecimal result = BigDecimal.valueOf(number.doubleValue());
    return result.max(BigDecimal.ZERO).setScale(TimingCalculator.MICRO_SECONDS.getPrecision(), TimingCalculator.MICRO_SECONDS.getRoundingMode());
  }

  @XbaseGenerated
  protected BigDecimal doCalculateDuration(final Node move) {
    if (move instanceof Move) {
      return _doCalculateDuration((Move)move);
    } else if (move instanceof SimpleAction) {
      return _doCalculateDuration((SimpleAction)move);
    } else if (move != null) {
      return _doCalculateDuration(move);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(move).toString());
    }
  }

  @XbaseGenerated
  protected BigDecimal doCalculateValue(final EObject distribution) {
    if (distribution instanceof Distribution) {
      return _doCalculateValue((Distribution)distribution);
    } else if (distribution instanceof Expression) {
      return _doCalculateValue((Expression)distribution);
    } else if (distribution instanceof Array) {
      return _doCalculateValue((Array)distribution);
    } else if (distribution instanceof Scalar) {
      return _doCalculateValue((Scalar)distribution);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(distribution).toString());
    }
  }

  @XbaseGenerated
  protected Map<? extends Node, ? extends Map<String, MotionData>> doCalculateMotionData(final Node move) {
    if (move instanceof Move) {
      return _doCalculateMotionData((Move)move);
    } else if (move != null) {
      return _doCalculateMotionData(move);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(move).toString());
    }
  }

  private static final Integer _DEFAULT_ARRAY_POINTER = Integer.valueOf((-1));

  private Integer getPointer(final Array owner) {
    Integer value = _IntermediateProperty_pointer.get(owner);
    return value == null ? _DEFAULT_ARRAY_POINTER : value;
  }

  private void setPointer(final Array owner, final Integer value) {
    if (value == _DEFAULT_ARRAY_POINTER) {
        _IntermediateProperty_pointer.remove(owner);
    } else {
        _IntermediateProperty_pointer.put(owner, value);
    }
  }

  private void disposePointer() {
    _IntermediateProperty_pointer.clear();
  }
}
