/**
 * Copyright (c) 2020 CEA LIST.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr
 */
package org.eclipse.papyrus.robotics.codegen.common.utils;

import com.google.common.base.Objects;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.papyrus.robotics.codegen.common.component.CodeSkeleton;
import org.eclipse.papyrus.robotics.core.utils.FunctionUtils;
import org.eclipse.papyrus.robotics.profile.robotics.components.Activity;
import org.eclipse.papyrus.robotics.profile.robotics.components.ActivityPort;
import org.eclipse.papyrus.robotics.profile.robotics.components.ComponentDefinition;
import org.eclipse.papyrus.robotics.profile.robotics.components.PeriodicTimer;
import org.eclipse.papyrus.robotics.profile.robotics.functions.Function;
import org.eclipse.papyrus.robotics.profile.robotics.functions.FunctionKind;
import org.eclipse.papyrus.uml.tools.utils.ConnectorUtil;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.ConnectableElement;
import org.eclipse.uml2.uml.Connector;
import org.eclipse.uml2.uml.ConnectorEnd;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.util.UMLUtil;

@SuppressWarnings("all")
public class ActivityUtils {
  public static List<Activity> getActivities(final org.eclipse.uml2.uml.Class component) {
    final ComponentDefinition cd = UMLUtil.<ComponentDefinition>getStereotypeApplication(component, ComponentDefinition.class);
    return cd.getActivities();
  }

  /**
   * Return the activity port connected with the passed component port.
   */
  public static ActivityPort getActivityForPort(final org.eclipse.uml2.uml.Class component, final Port componentPortUml) {
    final Port activityPortUml = ActivityUtils.getActivityForPortUml(component, componentPortUml);
    if ((activityPortUml != null)) {
      return UMLUtil.<ActivityPort>getStereotypeApplication(activityPortUml, ActivityPort.class);
    }
    return null;
  }

  /**
   * Return the activity connected with the passed component port.
   */
  public static org.eclipse.uml2.uml.Class getActivity(final org.eclipse.uml2.uml.Class component, final Port componentPortUml) {
    final Port activityPortUml = ActivityUtils.getActivityForPortUml(component, componentPortUml);
    if ((activityPortUml != null)) {
      return activityPortUml.getClass_();
    }
    return null;
  }

  /**
   * Return the activity port connected with the passed component port.
   */
  public static Port getActivityForPortUml(final org.eclipse.uml2.uml.Class component, final Port componentPortUml) {
    List<Activity> _activities = ActivityUtils.getActivities(component);
    for (final Activity activity : _activities) {
      {
        final org.eclipse.uml2.uml.Class activityCl = activity.getBase_Class();
        EList<Connector> _ownedConnectors = component.getOwnedConnectors();
        for (final Connector connector : _ownedConnectors) {
          boolean _connectsPort = ConnectorUtil.connectsPort(connector, componentPortUml);
          if (_connectsPort) {
            final ConnectorEnd end1 = connector.getEnds().get(0);
            final ConnectorEnd end2 = connector.getEnds().get(1);
            Element _owner = end1.getRole().getOwner();
            boolean _equals = Objects.equal(_owner, activityCl);
            if (_equals) {
              ConnectableElement _role = end1.getRole();
              return ((Port) _role);
            }
            Element _owner_1 = end2.getRole().getOwner();
            boolean _equals_1 = Objects.equal(_owner_1, activityCl);
            if (_equals_1) {
              ConnectableElement _role_1 = end2.getRole();
              return ((Port) _role_1);
            }
          }
        }
      }
    }
    return null;
  }

  /**
   * return true, if the component has external functions, i.e. empty functions
   * whose code is not supplied by fragments (opaque behavior) stored in the model
   */
  public static boolean hasExternalCode(final org.eclipse.uml2.uml.Class component) {
    List<Activity> _activities = ActivityUtils.getActivities(component);
    for (final Activity activity : _activities) {
      EList<Function> _functions = activity.getFunctions();
      for (final Function function : _functions) {
        boolean _isCodeInModel = function.isCodeInModel();
        boolean _not = (!_isCodeInModel);
        if (_not) {
          return true;
        }
      }
    }
    return false;
  }

  /**
   * return the name of the component class, in case of a component
   * with externally defined functions with an _impl postfix
   */
  public static String getPostfix(final org.eclipse.uml2.uml.Class component) {
    boolean _hasExternalCode = ActivityUtils.hasExternalCode(component);
    if (_hasExternalCode) {
      return CodeSkeleton.POSTFIX;
    }
    return "";
  }

  public static boolean hasPeriodicActivities(final org.eclipse.uml2.uml.Class component) {
    List<Activity> _activities = ActivityUtils.getActivities(component);
    for (final Activity activity : _activities) {
      int _size = FunctionUtils.getFunctions(activity, FunctionKind.PERIODIC).size();
      boolean _greaterThan = (_size > 0);
      if (_greaterThan) {
        return true;
      }
    }
    return false;
  }

  /**
   * return period length of an activity or null, if non-specified
   */
  public static String getPeriod(final Activity activity) {
    EList<Classifier> _nestedClassifiers = activity.getBase_Class().getNestedClassifiers();
    for (final Classifier cl : _nestedClassifiers) {
      {
        final PeriodicTimer pt = UMLUtil.<PeriodicTimer>getStereotypeApplication(cl, PeriodicTimer.class);
        if ((pt != null)) {
          return pt.getPeriod();
        }
      }
    }
    return null;
  }
}
