/*****************************************************************************
 * Copyright (c) 2020, 2023 CEA LIST.
 *
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * Contributors:
 *  Matteo MORELLI (CEA LIST) <matteo.morelli@cea.fr> - Bug #566899
 *  Matteo MORELLI (CEA LIST) <matteo.morelli@cea.fr> - Bug #581690
 *  Matteo MORELLI (CEA LIST) <matteo.morelli@cea.fr> - Bug #581812
 *
 *****************************************************************************/

package org.eclipse.papyrus.robotics.ros2.codegen.cpp.skillrealization

import java.util.Map
import org.eclipse.papyrus.designer.transformation.base.utils.TransformationException
import org.eclipse.papyrus.designer.languages.common.base.file.IPFileSystemAccess
import org.eclipse.papyrus.robotics.profile.robotics.skills.SkillDefinition
import org.eclipse.papyrus.robotics.profile.robotics.skills.SkillOperationalState
import org.eclipse.papyrus.robotics.profile.robotics.skills.SkillSemantic
import org.eclipse.papyrus.robotics.ros2.codegen.common.utils.SkillUtils
import org.eclipse.uml2.uml.Interface

import static extension org.eclipse.papyrus.robotics.core.utils.InteractionUtils.*
import static extension org.eclipse.papyrus.robotics.ros2.codegen.common.utils.SkillUtils.*
import org.eclipse.papyrus.designer.languages.common.base.file.ProtSection

/**
 * Create the C++ code of a skill realization executed
 * on the sequencing layer (uses a config&coord interface to SW components)
 */
class CreateSkillRealizationCppCode {

	private static def msgAbortMultipleOperationalStatesIsUnsopported() {
		return String.format(
				"abort transformation, code-generation of skill FSMs with many operational states is not supported"
				)
	}

	private static def msgAbortSystemHasNotCompIf(String ifName) {
		return String.format(
				"abort transformation, there are no components in the system with the required coordination interface (%s)",
					ifName
				)
	}

	private static def msgAbortCompIfNotActionNorQuery(String ifName) {
		return String.format(
				"abort transformation, the coordination interface has incompatible type (%s)",
					ifName
				)
	}
/*
	static def createCppSkillFSMManager(SkillDefinition definition, SkillSemantic semantics) '''
	'''
*/

	static def createCppSimplestActionSkill(SkillDefinition skill, SkillOperationalState ops, String actionName) '''
		// Generated by Papyrus4Robotics
		//

		// «ProtSection.protSection("BtActionNode header files")»
		// «ProtSection.protSection»
		#include "bt_utils/generic_types_conversions.hpp"
		«FOR tp : skill.uniqueSkillParameterTypes»
			#include "«tp.ROS2TypeIncludePath»"
		«ENDFOR»
		#include "«ops.coordinationIfIncludePath»"
		#include "nav2_behavior_tree/bt_action_node.hpp"

		class «skill.name»Action : public nav2_behavior_tree::BtActionNode<«ops.coordinationIfQn»>
		{
		public:
		  «skill.name»Action(
		      const std::string& name,
		      const std::string & action_name,
		      const BT::NodeConfiguration& conf)
		  : nav2_behavior_tree::BtActionNode<«ops.coordinationIfQn»>(name, action_name, conf)
		    {
		    }

		  «IF !skill.ins.isNullOrEmpty»
		  void on_tick() override
		  {
		  	// «ProtSection.protSection("BtActionNode on_tick()")»
		    «FOR param : skill.ins»
		    «param.type.getROS2TypeFromMsgName» «param.name»;
		    getInput("«param.name»", «param.name»);
		    «FOR prop : param.type.ownedAttributes»
		      goal_.«prop.name» = «param.name».«prop.name»;
		    «ENDFOR»
		  «ENDFOR»
		  	// «ProtSection.protSection»
		  }
		  «ENDIF»

		  «IF !skill.outs.isNullOrEmpty»
		  BT::NodeStatus on_success() override
		  {
		  	// «ProtSection.protSection("BtActionNode on_success()")»
		  	«FOR param : skill.outs»
		  	«param.type.getROS2TypeFromMsgName» «param.name»;
		  	«FOR prop : param.type.ownedAttributes»
		  	  «param.name».«prop.name» = result_.«prop.name»;
		  	«ENDFOR»
		  	setOutput("«param.name»", «param.name»);
		  	«ENDFOR»
		  	return BT::NodeStatus::SUCCESS;
		  	«ProtSection.protSection»
		  }
		  «ENDIF»

		  «createProvidedPortsMethod(skill,true)»

		// «ProtSection.protSection("- BtActionNode other class methods and attributes (if any)")»
		// «ProtSection.protSection»
		};

		#include "behaviortree_cpp_v3/bt_factory.h"
		BT_REGISTER_NODES(factory)
		{
		  BT::NodeBuilder builder =
		    [](const std::string & name, const BT::NodeConfiguration & config)
		    {
		      return std::make_unique<«skill.name»Action>(name, "«actionName»", config);
		    };
		
		  factory.registerBuilder<«skill.name»Action>("«skill.name»", builder);
		}
	'''

	static def createCppSimplestQuerySkill(SkillDefinition skill, SkillOperationalState ops, String serviceName) '''
		// Generated by Papyrus4Robotics
		//

		// «ProtSection.protSection("BtConditionNode header files")»
		// «ProtSection.protSection»
		#include "bt_utils/generic_types_conversions.hpp"
		«FOR tp : skill.uniqueSkillParameterTypes»
			#include "«tp.ROS2TypeIncludePath»"
		«ENDFOR»
		#include "«ops.coordinationIfIncludePath»"
		#include "bt_utils/bt_condition_node.hpp"

		class «skill.name»Condition : public bt_utils::BtConditionNode<«ops.coordinationIfQn»>
		{
		public:
		  «skill.name»Condition(
		      const std::string& name,
		      const std::string & service_name,
		      const BT::NodeConfiguration& conf)
		  : bt_utils::BtConditionNode<«ops.coordinationIfQn»>(name, service_name, conf)
		    {
		    }

		  «IF !skill.ins.isNullOrEmpty»
		  void on_tick() override
		  {
		  	// «ProtSection.protSection("BtConditionNode on_tick()")»
		    «FOR param : skill.ins»
		    «param.type.getROS2TypeFromMsgName» «param.name»;
		    getInput("«param.name»", «param.name»);
		    «FOR prop : param.type.ownedAttributes»
		      request_->«prop.name» = «param.name».«prop.name»;
		    «ENDFOR»
		  «ENDFOR»
		  	// «ProtSection.protSection»
		  }
		  «ENDIF»

		  BT::NodeStatus check_future(
		      std::shared_future<typename «ops.coordinationIfQn»::Response::SharedPtr> future_result) override
		  {
		    rclcpp::FutureReturnCode rc;
		    rc = rclcpp::spin_until_future_complete(
		      node_,
		      future_result, server_timeout_);
		    if (rc == rclcpp::FutureReturnCode::SUCCESS) {
		      auto res = future_result.get()->data;

		      return (res == true) ? BT::NodeStatus::SUCCESS : BT::NodeStatus::FAILURE;
		    } else if (rc == rclcpp::FutureReturnCode::TIMEOUT) {
		      RCLCPP_WARN(
		        node_->get_logger(),
		        "Node timed out while executing service call to %s.", service_name_.c_str());
		      on_wait_for_result();
		    }
		    return BT::NodeStatus::FAILURE;
		  }

		  «createProvidedPortsMethod(skill,true)»

		// «ProtSection.protSection("- BtConditionNode other class methods and attributes (if any)")»
		// «ProtSection.protSection»
		};

		#include "behaviortree_cpp_v3/bt_factory.h"
		BT_REGISTER_NODES(factory)
		{
		  BT::NodeBuilder builder =
		    [](const std::string & name, const BT::NodeConfiguration & config)
		    {
		      return std::make_unique<«skill.name»Condition>(name, "«serviceName»", config);
		    };

		  factory.registerBuilder<«skill.name»Condition>("«skill.name»", builder);
		}
	'''

	static def createCppSimplestSkillNoInteraction(SkillDefinition skill) '''
		// Generated by Papyrus4Robotics
		// ...

		#include <string>
		// «ProtSection.protSection("ActionNodeBase header files")»
		// «ProtSection.protSection»
		#include "rclcpp/rclcpp.hpp"
		#include "bt_utils/generic_types_conversions.hpp"
		«FOR tp : skill.uniqueSkillParameterTypes»
			#include "«tp.ROS2TypeIncludePath»"
		«ENDFOR»
		#include "behaviortree_cpp_v3/action_node.h"
		
		class «skill.name»Action : public BT::ActionNodeBase
		{
		
		public:
		  // Any TreeNode with ports must have a constructor with this signature
		  «skill.name»Action(
		    const std::string& name,
		    const BT::NodeConfiguration& config)
		  : ActionNodeBase(name, config)
		  {
		  }
		
		  «createProvidedPortsMethod(skill,false)»
		
		  BT::NodeStatus tick() override
		  {
		  	// «ProtSection.protSection("ActionNodeBase tick()")»
			«IF !skill.ins.isNullOrEmpty»
			  // Read from input ports
			  //
			  «FOR param : skill.ins»
			  	«param.type.getROS2TypeFromMsgName» «param.name»;
			  	getInput("«param.name»", «param.name»);
			  «ENDFOR»

			  /* ----------------------------------------------
			   * USER CODE HERE USING READINGS FROM INPUT PORTS
			   * ----------------------------------------------
			   */
			«ENDIF»

			«IF skill.ins.isNullOrEmpty && skill.outs.isNullOrEmpty»
				/* -------------------
				 * SOME USER CODE HERE
				 * -------------------
				 */
			«ENDIF»

			«IF !skill.outs.isNullOrEmpty»
				// Share through the blackboard
				//
				«FOR param : skill.outs»
					«param.type.getROS2TypeFromMsgName» «param.name»;
					/* -----------------------------------------
					 * USER CODE HERE TO INITIALIZE THE VARIABLE
					 * -----------------------------------------
					 */
					setOutput("«param.name»", «param.name»);

				«ENDFOR»
			«ENDIF»
		  	return BT::NodeStatus::SUCCESS;
		  	// «ProtSection.protSection»
		  }

		// «ProtSection.protSection("- ActionNodeBase other class methods and attributes (if any)")»
		// «ProtSection.protSection»

		};
		
		#include "behaviortree_cpp_v3/bt_factory.h"
		
		// This function must be implemented in the .cpp file to create
		// a plugin that can be loaded at run-time
		BT_REGISTER_NODES(factory)
		{
		    factory.registerNodeType<«skill.name»Action>("«skill.name»");
		}
	'''

	static def createProvidedPortsMethod(SkillDefinition skill, boolean need_basic_ports) '''
		// «skill.name» has a constructor in the form (const std::string&, const NodeConfiguration&) => must provide a providedPorts method
		static BT::PortsList providedPorts()
		{
		  return«IF need_basic_ports» providedBasicPorts(«ENDIF»
		  {
		  	«createPortConstructionCommands(skill)»
		  }«IF need_basic_ports»)«ENDIF»;
		}
	'''

    static def createPortConstructionCommands(SkillDefinition skill) '''
		«FOR param : skill.ins + skill.outs SEPARATOR ','»
		«IF skill.ins.contains(param)»BT::InputPort<«ELSE»BT::OutputPort<«ENDIF»«param.type.getROS2TypeFromMsgName»>("«param.name»")
		«ENDFOR»
	'''

	static def genCode(IPFileSystemAccess fileAccess, Map<SkillDefinition, SkillSemantic> skdefToSemanticsMap, Map<Interface, String> serviceToNameMap) {
		for (entry : skdefToSemanticsMap.entrySet) {
			val definition  = entry.key
			val semantics   = entry.value
			// Generate a skill manager to manage complex realization semantics with more than 1 operational state
			if (semantics.operational.size > 1) {
				throw new TransformationException(
						msgAbortMultipleOperationalStatesIsUnsopported()
					)
			}
			else {
			// here it is guaranteed that semantics.operational.size == 1
				val ops = semantics.firstOpState
				val skillRealizationFileName = definition.realizationFileName+".cpp"
				if (SkillUtils.doesConfigAndCoordOfComponents(ops)) {
					val cIf = SkillUtils.getCompInterface(ops)
					if (!serviceToNameMap.containsKey(cIf))
						throw new
							TransformationException(
								msgAbortSystemHasNotCompIf(
									cIf.name
								)
							);
					if (!cIf.communicationPattern.isAction && !cIf.communicationPattern.isQuery)
						throw new
							TransformationException(
								msgAbortCompIfNotActionNorQuery(
									cIf.name
								)
							)
					if (cIf.communicationPattern.isAction) {
						val actionName = serviceToNameMap.get(cIf)
						fileAccess.generateFile(skillRealizationFileName, createCppSimplestActionSkill(definition, ops, actionName).toString)
					}
					else {// cIf.communicationPattern.isQuery
						val serviceName = serviceToNameMap.get(cIf)
						fileAccess.generateFile(skillRealizationFileName, createCppSimplestQuerySkill(definition, ops, serviceName).toString)
					}
				}
				else
					fileAccess.generateFile(skillRealizationFileName, createCppSimplestSkillNoInteraction(definition).toString)
			}
		}
	}
}