/**
 * Copyright (c) 2010, 2013 Darmstadt University of Technology.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Marcel Bruch - initial API and implementation.
 */
package org.eclipse.recommenders.stats.rcp.events;

import static org.apache.commons.lang3.StringUtils.isEmpty;
import static org.eclipse.recommenders.stats.rcp.events.CompletionEvent.ProposalKind.toKind;

import java.util.List;

import javax.inject.Inject;

import org.eclipse.jdt.core.CompletionProposal;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.internal.compiler.ast.ASTNode;
import org.eclipse.jface.text.contentassist.ICompletionProposal;
import org.eclipse.recommenders.completion.rcp.IRecommendersCompletionContext;
import org.eclipse.recommenders.completion.rcp.processable.IProcessableProposal;
import org.eclipse.recommenders.completion.rcp.processable.SessionProcessor;
import org.eclipse.recommenders.rcp.IRcpService;
import org.eclipse.recommenders.rcp.JavaElementResolver;
import org.eclipse.recommenders.stats.rcp.events.CompletionEvent.ProposalKind;
import org.eclipse.recommenders.stats.rcp.interfaces.IDataPersistenceService;
import org.eclipse.recommenders.stats.rcp.persistence.Constants;

@SuppressWarnings("restriction")
public class StatisticsSessionProcessor extends SessionProcessor implements IRcpService {

    private static final ASTNode NULL = new NullAstNode();

    private static class NullAstNode extends ASTNode {
        @Override
        public StringBuffer print(int indent, StringBuffer output) {
            return output;
        }
    }

    private final JavaElementResolver resolver;
    private final IDataPersistenceService persistenceService;

    // protected for test purposes
    protected CompletionEvent event;
    private IRecommendersCompletionContext ctx;

    @Inject
    public StatisticsSessionProcessor(JavaElementResolver resolver, IDataPersistenceService persistenceService) {
        this.resolver = resolver;
        this.persistenceService = persistenceService;
    }

    @Override
    public boolean startSession(IRecommendersCompletionContext ctx) {
        this.ctx = ctx;
        flushCurrentEvent();
        initEvent();
        return true;
    }

    @Override
    public void applied(ICompletionProposal proposal) {
        event.sessionEnded = System.currentTimeMillis();

        if (proposal instanceof IProcessableProposal) {
            IProcessableProposal p = (IProcessableProposal) proposal;
            CompletionProposal coreProposal = p.getCoreProposal().orNull();
            if (coreProposal != null) {
                event.prefix = p.getPrefix();
                event.applied = toKind(coreProposal.getKind());
                event.completion = new String(coreProposal.getCompletion());
                if (ProposalKind.UNKNOWN == event.applied && isEmpty(event.completion)) {
                    event.error = coreProposal.toString();
                }
            }
        } else {
            event.applied = ProposalKind.UNKNOWN;
        }
        flushCurrentEvent();
    }

    @Override
    public void aboutToClose() {
        if (event != null) {
            event.sessionEnded = System.currentTimeMillis();
        }
    }

    @Override
    public void endSession(List<ICompletionProposal> proposals) {
        event.numberOfProposals = proposals.size();
    }

    private void initEvent() {
        event = new CompletionEvent();
        event.sessionStarted = System.currentTimeMillis();
        event.completionKind = ctx.getCompletionNode().or(NULL).getClass().getSimpleName();
        event.completionParentKind = ctx.getCompletionNodeParent().or(NULL).getClass().getSimpleName();

        IType receiverType = ctx.getReceiverType().orNull();
        if (receiverType != null) {
            event.receiverType = resolver.toRecType(receiverType);
        }
        event.prefix = ctx.getPrefix();
    }

    private void flushCurrentEvent() {
        persistenceService.record(event, Constants.COMPLETION_EVENTS_CHANNEL);
        event = null;
    }
}
