/**
 * Copyright (c) 2013 Timur Achmetow.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Timur Achmetow - Initial API and implementation
 */
package org.eclipse.recommenders.stats.rcp.ui;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.recommenders.stats.rcp.interfaces.ICategoryItem;
import org.eclipse.recommenders.stats.rcp.interfaces.IPageContent;
import org.eclipse.recommenders.stats.rcp.model.TreeCategory;
import org.eclipse.recommenders.stats.rcp.model.TreeItem;
import org.eclipse.swt.graphics.Image;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.annotations.VisibleForTesting;

public class ExtensionPointReader {

    private static final Logger LOG = LoggerFactory.getLogger(ExtensionPointReader.class);
    private static final String EXTENSION_POINT_ID = "org.eclipse.recommenders.stats.rcp.pages"; //$NON-NLS-1$

    private static final String PAGE = "page";
    private static final String CATEGORY = "category";
    private static final String NAME = "name";
    private static final String IMAGE = "image";

    private static ExtensionPointReader instance = new ExtensionPointReader();

    public static ExtensionPointReader getInstance() {
        return instance;
    }

    public List<ICategoryItem> createTreeObjectsFromExtensions() {
        final IConfigurationElement[] elements = Platform.getExtensionRegistry().getConfigurationElementsFor(
                EXTENSION_POINT_ID);
        return createTreeObjectsFromExtensions(elements);
    }

    @VisibleForTesting
    protected List<ICategoryItem> createTreeObjectsFromExtensions(IConfigurationElement[] elements) {
        if (elements == null) {
            return new ArrayList<ICategoryItem>();
        }
        final Map<String, ICategoryItem> catMap = buildDataStructure(elements);
        return new ArrayList<ICategoryItem>(catMap.values());
    }

    private Map<String, ICategoryItem> buildDataStructure(IConfigurationElement[] elements) {
        final Map<String, ICategoryItem> catMap = new HashMap<String, ICategoryItem>();
        for (final IConfigurationElement element : elements) {
            final String elementType = element.getName();
            final String pluginId = element.getContributor().getName();
            ImageDescriptor imageDescriptor = AbstractUIPlugin.imageDescriptorFromPlugin(pluginId,
                    element.getAttribute(IMAGE));
            final Image image = imageDescriptor == null ? null : imageDescriptor.createImage();
            final String id = element.getAttribute("id");
            final String name = element.getAttribute(NAME);

            if (elementType.equals(CATEGORY)) {
                if (!catMap.containsKey(id)) {
                    TreeCategory cat = new TreeCategory(name, image, element.getAttribute("description"));
                    catMap.put(id, cat);
                }
            }
        }

        for (final IConfigurationElement element : elements) {
            final String elementType = element.getName();
            final String pluginId = element.getContributor().getName();
            ImageDescriptor imageDescriptor = AbstractUIPlugin.imageDescriptorFromPlugin(pluginId,
                    element.getAttribute(IMAGE));
            final Image image = imageDescriptor == null ? null : imageDescriptor.createImage();
            final String categoryID = element.getAttribute(CATEGORY);
            final String name = element.getAttribute(NAME);

            if (elementType.equals(PAGE)) {
                ICategoryItem catItem = catMap.get(categoryID);
                if (catItem != null) {
                    TreeItem item = new TreeItem(name, image, getExecutableObject(element));
                    catItem.addItem(item);
                }
            }
        }

        return catMap;
    }

    private IPageContent getExecutableObject(final IConfigurationElement element) {
        IPageContent object = null;
        try {
            object = (IPageContent) element.createExecutableExtension("class");
        } catch (CoreException e) {
            LOG.debug("Failed to instantiate executable " + element.getAttribute("class"));
        }
        return object;
    }
}
