/**
 * Copyright (c) 2015 Akif Etkue.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Akif Etkue - Initial API and implementation
 */
package org.eclipse.recommenders.stats.rcp.ui.data;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;

import org.eclipse.core.runtime.preferences.IEclipsePreferences;
import org.eclipse.core.runtime.preferences.InstanceScope;
import org.eclipse.jgit.lib.Ref;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.revwalk.RevCommit;
import org.eclipse.jgit.revwalk.RevWalk;
import org.eclipse.jgit.storage.file.FileRepositoryBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Splitter;

public class GitCommitData {

    private static final Logger LOG = LoggerFactory.getLogger(GitCommitData.class);
    private static final IEclipsePreferences PREFS = InstanceScope.INSTANCE.getNode("org.eclipse.egit.core");
    private static final String PREFS_DIRECTORIES = "GitRepositoriesView.GitDirectories";
    private static final int COMMITER_COUNT = 6;

    public static Set<String> readAllGitRepositories() {
        String dirs;
        synchronized (PREFS) {
            dirs = PREFS.get(PREFS_DIRECTORIES, ""); //$NON-NLS-1$
        }
        return getGitRepositories(dirs);
    }

    public static Map<String, Double> readAllCommits(String repoPath) {
        Map<String, AtomicInteger> committerData = new HashMap<>();
        try {
            List<RevCommit> commits = readCommits(repoPath);
            for (RevCommit commit : commits) {
                String author = commit.getAuthorIdent().getName();
                if (!committerData.containsKey(author)) {
                    committerData.put(author, new AtomicInteger(0));
                }
                committerData.get(author).incrementAndGet();
            }
            LOG.debug("Repo: " + repoPath + " Commit Number: " + commits.size());

        } catch (IOException e) {
            LOG.error(e.getMessage());
        }

        Map<String, Double> result = new HashMap<>();
        for (String key : committerData.keySet()) {
            result.put(key, committerData.get(key).doubleValue());
        }
        return result;
    }

    public static String extractRepoName(String repoPath) {
        if (repoPath == null || repoPath.isEmpty()) {
            return "";
        }

        String repo = repoPath;
        if (repo.contains(".git")) {
            repo = repo.replace(".git", "");
        }
        if (repo.contains("\\")) {
            repo = repo.replace("\\", "/");
        }

        File dir = new File(repo);
        return dir.getName();
    }

    public static int getCommitNumber(Map<String, Double> commitMap) {
        int totalNumber = 0;
        for (String key : commitMap.keySet()) {
            totalNumber += commitMap.get(key);
        }
        return totalNumber;
    }

    public static <K, V extends Comparable<? super V>> Map<K, V> sortByValue(Map<K, V> map) {
        return sortByValue(map, COMMITER_COUNT);
    }

    private static <K, V extends Comparable<? super V>> Map<K, V> sortByValue(Map<K, V> map, int committer) {
        List<Map.Entry<K, V>> list = new ArrayList<>(map.entrySet());
        Collections.sort(list, new Comparator<Map.Entry<K, V>>() {
            @Override
            public int compare(Map.Entry<K, V> o1, Map.Entry<K, V> o2) {
                return o2.getValue().compareTo(o1.getValue());
            }
        });
        int count = 0;
        Map<K, V> result = new LinkedHashMap<>();
        for (Map.Entry<K, V> entry : list) {
            if (count >= committer) {
                break;
            }
            result.put(entry.getKey(), entry.getValue());
            count++;
        }
        return result;
    }

    private static List<RevCommit> readCommits(String repoPath) throws IOException {
        List<RevCommit> commitList = new ArrayList<>();
        Repository repository = openRepository(repoPath);
        Ref head = repository.getRef("refs/heads/master");

        if (head != null) {
            // if no repository found, head will be null
            LOG.debug("Read commit data from GIT repo: " + repoPath);

            RevWalk walk = new RevWalk(repository);
            RevCommit commit = walk.parseCommit(head.getObjectId());
            walk.markStart(commit);

            for (RevCommit rev : walk) {
                commitList.add(rev);
            }
            walk.dispose();
        }
        repository.close();
        return commitList;
    }

    private static Repository openRepository(String repoPath) throws IOException {
        FileRepositoryBuilder repositoryBuilder = new FileRepositoryBuilder();
        repositoryBuilder.setMustExist(true);
        repositoryBuilder.setGitDir(new File(repoPath));
        return repositoryBuilder.build();
    }

    @VisibleForTesting
    protected static Set<String> getGitRepositories(String gitRepos) {
        if (gitRepos == null || gitRepos.isEmpty()) {
            return Collections.emptySet();
        }

        Set<String> repoURL = new HashSet<String>();
        Iterable<String> split = Splitter.on(File.pathSeparator).split(gitRepos);
        for (String repo : split) {
            if (!repo.isEmpty()) {
                repoURL.add(repo);
            }
        }
        return repoURL;
    }
}
