/**
 * Copyright (c) 2013 Timur Achmetow.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Timur Achmetow - Initial API and implementation
 */
package org.eclipse.recommenders.stats.rcp.ui.data;

import static com.google.common.base.Predicates.not;

import java.util.Collection;

import org.eclipse.recommenders.stats.rcp.events.CompletionEvent;
import org.eclipse.recommenders.stats.rcp.events.CompletionEvent.ProposalKind;
import org.eclipse.recommenders.stats.rcp.interfaces.IDataPersistenceService;
import org.eclipse.recommenders.stats.rcp.persistence.Constants;

import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;
import com.google.common.collect.HashMultiset;
import com.google.common.collect.Multiset;

public final class StatisticData {

    private Collection<CompletionEvent> okayEvents;
    private Collection<CompletionEvent> appliedEvents;
    private Collection<CompletionEvent> abortedEvents;
    private final IDataPersistenceService persistenceService;

    private static class BuggyEventsPredicate implements Predicate<CompletionEvent> {
        @Override
        public boolean apply(CompletionEvent input) {
            return input.numberOfProposals < 1 || input.sessionEnded < input.sessionStarted;
        }
    }

    private static class HasAppliedProposalPredicate implements Predicate<CompletionEvent> {
        @Override
        public boolean apply(CompletionEvent e) {
            return e.applied != null;
        }
    }

    private StatisticData(IDataPersistenceService persistenceService) {
        this.persistenceService = persistenceService;
        loadEventData();
    }

    public static StatisticData create(IDataPersistenceService persistenceService) {
        return new StatisticData(persistenceService);
    }

    private void loadEventData() {
        Collection<CompletionEvent> eventList = persistenceService.retrieve(CompletionEvent.class,
                Constants.COMPLETION_EVENTS_CHANNEL);

        okayEvents = Collections2.filter(eventList, not(new BuggyEventsPredicate()));
        appliedEvents = Collections2.filter(okayEvents, new HasAppliedProposalPredicate());
        abortedEvents = Collections2.filter(okayEvents, not(new HasAppliedProposalPredicate()));
    }

    public Multiset<Object> getStatsData() {
        Multiset<Object> proposalKindBag = HashMultiset.create();
        for (final ProposalKind kind : ProposalKind.values()) {
            Collection<CompletionEvent> byKind = Collections2.filter(okayEvents, new Predicate<CompletionEvent>() {
                @Override
                public boolean apply(CompletionEvent input) {
                    if (kind == input.applied) {
                        return true;
                    }
                    return false;
                }
            });
            if (byKind.size() > 0) {
                proposalKindBag.add(kind, byKind.size());
            }
        }
        return proposalKindBag;
    }

    public Collection<CompletionEvent> getOkayEvents() {
        return okayEvents;
    }

    public Collection<CompletionEvent> getAppliedEvents() {
        return appliedEvents;
    }

    public Collection<CompletionEvent> getAbortedEvents() {
        return abortedEvents;
    }
}
