/**
 * Copyright (c) 2015 Akif Etkue.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Akif Etkue - Initial API and implementation
 */
package org.eclipse.recommenders.stats.rcp.ui.data;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectNatureDescriptor;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.recommenders.stats.rcp.model.WorkspaceProjectType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WorkspaceProjectData {

    private static final Logger LOG = LoggerFactory.getLogger(WorkspaceProjectData.class);
    private int totalProjects;
    private long totalSize;

    public List<WorkspaceProjectType> readAllProjects() {
        List<WorkspaceProjectType> modelList = new ArrayList<WorkspaceProjectType>();
        IWorkspaceRoot workspaceRoot = ResourcesPlugin.getWorkspace().getRoot();
        for (IProject project : workspaceRoot.getProjects()) {
            if (project.exists()) {

                List<File> collectedFiles = new ArrayList<File>();
                IPath path = project.getLocation();
                WorkspaceDataHelper.collectAllFilesFor(path.toFile(), collectedFiles);

                try {
                    String name = project.getName();
                    int fileSize = collectedFiles.size();
                    long bytes = WorkspaceDataHelper.getSizeFor(collectedFiles);
                    List<String> labelType = getLabelType(workspaceRoot, project);
                    modelList.add(new WorkspaceProjectType(name, labelType, fileSize, bytes));
                } catch (CoreException e) {
                    LOG.debug(e.getMessage());
                }
            }
        }

        totalProjects = modelList.size();
        totalSize = getSumOfAllProjects(modelList);

        return modelList;
    }

    protected List<String> getLabelType(IWorkspaceRoot workspaceRoot, IProject project) throws CoreException {
        List<String> labelList = new ArrayList<>();
        if (project.getDescription() != null) {
            String[] natureIds = project.getDescription().getNatureIds();
            if (natureIds != null) {
                for (String natureID : natureIds) {
                    IProjectNatureDescriptor natureDescriptor = workspaceRoot.getWorkspace().getNatureDescriptor(
                            natureID);
                    if (natureDescriptor != null && natureDescriptor.getLabel() != null) {
                        labelList.add(natureDescriptor.getLabel());
                    }
                }
            }
        }
        return labelList;
    }

    public int getTotalProjects() {
        return totalProjects;
    }

    public long getTotalSize() {
        return totalSize;
    }

    private long getSumOfAllProjects(List<WorkspaceProjectType> modelList) {
        long sumAllProjects = 0L;
        for (WorkspaceProjectType project : modelList) {
            sumAllProjects += project.getBytes();
        }
        return sumAllProjects;
    }
}
