/*******************************************************************************
 * Copyright (c) 2017 Red Hat, Inc and others.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Red Hat, Inc - initial API and implementation
 *******************************************************************************/
package org.eclipse.reddeer.eclipse.test.ui.views.log;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

import java.util.List;

import org.eclipse.core.runtime.ILog;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.reddeer.common.condition.AbstractWaitCondition;
import org.eclipse.reddeer.common.wait.AbstractWait;
import org.eclipse.reddeer.common.wait.TimePeriod;
import org.eclipse.reddeer.common.wait.WaitUntil;
import org.eclipse.reddeer.eclipse.test.Activator;
import org.eclipse.reddeer.eclipse.ui.views.log.LogMessage;
import org.eclipse.reddeer.eclipse.ui.views.log.LogView;
import org.eclipse.reddeer.junit.runner.RedDeerSuite;
import org.eclipse.reddeer.workbench.impl.menu.WorkbenchPartMenuItem;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;

@RunWith(RedDeerSuite.class)
public class LogViewTest {

	public static final String OK_ID_1="fake_ID_OK_1";
	public static final String OK_MESSAGE_1="fake_OK_1";
	
	public static final String OK_ID_2="fake_ID_OK_2";
	public static final String OK_MESSAGE_2="fake_OK_2";
	public static final String OK_STACK_2="fake_OK_2";
	
	public static final String ERROR_ID_1="fake_ID_ERROR_1";
	public static final String ERROR_MESSAGE_1="fake_ERROR_1";
	public static final String ERROR_STACK_1="fake_ERROR_1";
	
	public static final String ERROR_ID_2="fake_ID_ERROR_2";
	public static final String ERROR_MESSAGE_2="fake_ERROR_2";
	public static final String ERROR_STACK_2="fake_ERROR_2";
	
	public static final String WARNING_ID_1="fake_ID_WARNING_1";
	public static final String WARNING_MESSAGE_1="fake_WARNING_1";
	public static final String WARNING_STACK_1="fake_WARNING_1";
	
	public static final String WARNING_ID_2="fake_ID_WARNING_2";
	public static final String WARNING_MESSAGE_2="fake_WARNING_2";
	public static final String WARNING_STACK_2="fake_WARNING_2";
	
	public static final String INFO_ID_1="fake_ID_INFO_1";
	public static final String INFO_MESSAGE_1="fake_INFO_1";
	
	public static final String INFO_ID_2="fake_ID_INFO_2";
	public static final String INFO_MESSAGE_2="fake_INFO_2";
	public static final String INFO_STACK_2="fake_INFO_2";
	
	
	
	
	private LogView logView;
	
	@Before
	public void setup(){
		ILog log= Platform.getLog(Platform.getBundle(Activator.PLUGIN_ID));
		
		log.log(new Status(IStatus.ERROR,ERROR_ID_1,ERROR_MESSAGE_1,new NullPointerException(ERROR_STACK_1)));
		log.log(new Status(IStatus.ERROR,ERROR_ID_2,ERROR_MESSAGE_2,new NullPointerException(ERROR_STACK_2)));
		
		log.log(new Status(IStatus.OK,OK_ID_1,OK_MESSAGE_1,null));
		log.log(new Status(IStatus.OK,OK_ID_2,OK_MESSAGE_2,new NullPointerException(OK_STACK_2)));
		
		log.log(new Status(IStatus.WARNING,WARNING_ID_1,WARNING_MESSAGE_1,new IllegalArgumentException(WARNING_STACK_1)));
		log.log(new Status(IStatus.WARNING,WARNING_ID_2,WARNING_MESSAGE_2,new NullPointerException(WARNING_STACK_2)));
		
		log.log(new Status(IStatus.INFO,INFO_ID_1,INFO_MESSAGE_1,null));
		log.log(new Status(IStatus.INFO,INFO_ID_2,INFO_MESSAGE_2,new NullPointerException(INFO_STACK_2)));
		AbstractWait.sleep(TimePeriod.getCustom(3));

		logView = new LogView();
		logView.open();
	}
	
	@Test
	public void getOKMessage(){
		List<LogMessage> messages = logView.getOKMessages();
		assertTrue("No OK messages found!", !messages.isEmpty());
		
		//test OK Message no.1
		assertTrue(messageIsAvailable(messages, IStatus.OK, OK_ID_1,
				OK_MESSAGE_1, "An exception stack trace is not available."));

		//test OK Message no.2
		assertTrue(messageIsAvailable(messages, IStatus.OK, OK_ID_2,
				OK_MESSAGE_2, OK_STACK_2));
	}
	
	@Test
	public void getInfoMessage(){
		List<LogMessage> messages = logView.getInfoMessages();
		assertTrue("No INFO messages found!", !messages.isEmpty());
		
		//test INFO Message no.1
		assertTrue(messageIsAvailable(messages, IStatus.INFO, INFO_ID_1,
				INFO_MESSAGE_1, "An exception stack trace is not available."));
				
		//test INFO Message no.2
		assertTrue(messageIsAvailable(messages, IStatus.INFO, INFO_ID_2,
				INFO_MESSAGE_2, INFO_STACK_2));
	}
	
	@Test
	public void getWarningMessage(){
		List<LogMessage> messages = logView.getWarningMessages();
		assertTrue("No WARNING messages found!", !messages.isEmpty());
		
		//test WARNING Message no.1
		assertTrue(messageIsAvailable(messages, IStatus.WARNING, WARNING_ID_1,
				WARNING_MESSAGE_1, WARNING_STACK_1));
				
		//test WARNING Message no.2
		assertTrue(messageIsAvailable(messages, IStatus.WARNING, WARNING_ID_2,
				WARNING_MESSAGE_2, WARNING_STACK_2));
	}
	
	@Test
	public void getErrorMessage(){
		List<LogMessage> messages = logView.getErrorMessages();
		assertTrue("No ERROR messages found!", !messages.isEmpty());
		
		//test ERROR Message no.1
		assertTrue(messageIsAvailable(messages, IStatus.ERROR, ERROR_ID_1,
				ERROR_MESSAGE_1, ERROR_STACK_1));
				
		//test ERROR Message no.2
		assertTrue(messageIsAvailable(messages, IStatus.ERROR, ERROR_ID_2,
				ERROR_MESSAGE_2, ERROR_STACK_2));
	}

	@Test
	public void testClearAndRestore(){
		assertFalse("There must be messages", logView.getErrorMessages().isEmpty());
		logView.clearLog();				
		assertTrue("There should be messages", logView.getErrorMessages().isEmpty());
		logView.restoreLog();				
		assertFalse("There should be messages", logView.getErrorMessages().isEmpty());			
	}
	
	@Test
	public void testDelete(){
		assertFalse("There must be messages", logView.getErrorMessages().isEmpty());
		logView.deleteLog();				
		assertTrue("There should be no messages", logView.getErrorMessages().isEmpty());
		logView.deleteLog(); //https://github.com/jboss-reddeer/reddeer/pull/953
		logView.restoreLog();				
		assertTrue("There should be no messages", logView.getErrorMessages().isEmpty());
	}

	@Test
	public void testSetActivateOnNewEvents() {
		logView.setActivateOnNewEvents(true);
		WorkbenchPartMenuItem menu = new WorkbenchPartMenuItem("Activate on new events");
		assertTrue("'Activate on new events' option should be selected", menu.isSelected());
		logView.setActivateOnNewEvents(false);
		assertFalse("'Activate on new events' option should NOT be selected", menu.isSelected());
		logView.setActivateOnNewEvents(true);
		assertTrue("'Activate on new events' option should be selected", menu.isSelected());
	}
	
	private boolean messageIsAvailable(List<LogMessage> messages, int severity, String plugin, String message, String stackTrace) {
		for (LogMessage m : messages) {
			if (m.getSeverity() == severity && m.getPlugin().equals(plugin) && m.getMessage().equals(message)) {
				return true;
			}
		}
		return false;
	}
	
	@After
	public void cleanup() throws Exception {
//		if (!Platform.getLogFileLocation().toFile().delete()) {
//			System.out.println("Log file not deleted properly");
//		}

		LogView logView2 = new LogView();
		logView2.open();
		logView2.deleteLog();
		new WaitUntil(new LogViewIsEmpty(), TimePeriod.DEFAULT);
	}
	
	private class LogViewIsEmpty extends AbstractWaitCondition {

		@Override
		public boolean test() {
			LogView view = new LogView();
			if (!view.isOpen()) {
				view.open();
			}
			if (view.getOKMessages().isEmpty() 
					&& view.getErrorMessages().isEmpty() 
					&& view.getWarningMessages().isEmpty()
					&& view.getInfoMessages().isEmpty()) {
				return true;
			}
			return false;
		}
	}
	
}
