/*******************************************************************************
 * Copyright (c) 2024 École Polytechnique de Montréal
 *
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License 2.0 which
 * accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *******************************************************************************/
package org.eclipse.tracecompass.incubator.internal.rocm.core.analysis;

/**
 * Event Layout for ROCm events generated by the CTF plugin
 *
 * @author Arnaud Fiorini
 */
public class RocmEventLayout {

    /* Event name variables */
    private static final String HIP_PREFIX = "hip"; //$NON-NLS-1$
    private static final String HIP_BEGIN_SUFFIX = "Begin"; //$NON-NLS-1$
    private static final String HIP_END_SUFFIX = "End"; //$NON-NLS-1$
    private static final String HSA_PREFIX = "hsa"; //$NON-NLS-1$
    private static final String HSA_BEGIN_SUFFIX = "_begin"; //$NON-NLS-1$
    private static final String HSA_END_SUFFIX = "_end"; //$NON-NLS-1$
    private static final String HIP_OPERATION_BEGIN = "hip_op_begin"; //$NON-NLS-1$
    private static final String HIP_OPERATION_END = "hip_op_end"; //$NON-NLS-1$
    private static final String HSA_OPERATION_BEGIN = "hsa_op_begin"; //$NON-NLS-1$
    private static final String HSA_OPERATION_END = "hsa_op_end"; //$NON-NLS-1$

    private static final String HIP_LAUNCH_PREFIX = "hipLaunch"; //$NON-NLS-1$
    private static final String HIP_EXT_LAUNCH_PREFIX = "hipExtLaunch"; //$NON-NLS-1$
    private static final String HIP_MEMCPY_PREFIX = "hipMemcpy"; //$NON-NLS-1$
    private static final String HIP_MEMCPY_BEGIN = "hipMemcpyBegin"; //$NON-NLS-1$
    private static final String HIP_MEMCPY_END = "hipMemcpyEnd"; //$NON-NLS-1$
    private static final String HIP_LAUNCH_KERNEL_BEGIN = "hipLaunchKernelBegin"; //$NON-NLS-1$
    private static final String HIP_LAUNCH_KERNEL_END = "hipLaunchKernelEnd"; //$NON-NLS-1$
    private static final String HIP_STREAM_SYNCHRONIZE_BEGIN = "hipStreamSynchronizeBegin"; //$NON-NLS-1$
    private static final String HIP_STREAM_SYNCHRONIZE_END = "hipStreamSynchronizeEnd"; //$NON-NLS-1$

    private static final String HSA_HANDLE_TYPE = "hsa_handle_type"; //$NON-NLS-1$

    /* Common event field names */
    private static final String THREAD_ID = "context._thread_id"; //$NON-NLS-1$
    private static final String QUEUE_ID = "context._queue_id"; //$NON-NLS-1$
    private static final String AGENT_ID = "context._agent_id"; //$NON-NLS-1$
    private static final String CORRELATION_ID = "context._correlation_id"; //$NON-NLS-1$

    /* HIP event field names */
    private static final String KERNEL_NAME = "context._kernel_name"; //$NON-NLS-1$
    private static final String MEMCPY_KIND = "kind"; //$NON-NLS-1$

    /* HSA event field names */

    /* HIP operation event field names */
    private static final String OP_KERNEL_NAME = "kernel_name"; //$NON-NLS-1$

    /* HSA operation event field names */

    // ------------------------------------------------------------------------
    // Event names
    // ------------------------------------------------------------------------

    /**
     * This event occurs after a call to hipMemcpy which initiates a memory copy
     * that involves a GPU.
     *
     * @return the event name
     */
    public String hipMemcpyBegin() {
        return HIP_MEMCPY_BEGIN;
    }

    /**
     * This event occurs right before the call to hipMemcpy ended.
     *
     * @return the event name
     */
    public String hipMemcpyEnd() {
        return HIP_MEMCPY_END;
    }

    /**
     * This event occurs after a call to hipMemcpy which initiates a kernel
     * launch to a ROCm Agent.
     *
     * @return the event name
     */
    public String hipLaunchKernelBegin() {
        return HIP_LAUNCH_KERNEL_BEGIN;
    }

    /**
     * This event occurs right before the call to hipLaunchKernel ended.
     *
     * @return the event name
     */
    public String hipLaunchKernelEnd() {
        return HIP_LAUNCH_KERNEL_END;
    }

    /**
     * This event occurs after a call to hipStreamSynchronize which initiates a
     * barrier call on a HIP stream.
     *
     * @return the event name
     */
    public String hipStreamSynchronizeBegin() {
        return HIP_STREAM_SYNCHRONIZE_BEGIN;
    }

    /**
     * This event occurs right before the call to hipStreamSynchronize ended.
     *
     * @return the event name
     */
    public String hipStreamSynchronizeEnd() {
        return HIP_STREAM_SYNCHRONIZE_END;
    }

    /**
     * This function returns the prefix of each HIP API event name.
     *
     * @return the prefix
     */
    public String getHipPrefix() {
        return HIP_PREFIX;
    }

    /**
     * This function returns the suffix of each HIP API begin event name.
     *
     * @return the suffix
     */
    public String getHipBeginSuffix() {
        return HIP_BEGIN_SUFFIX;
    }

    /**
     * This function returns the suffix of each HIP API end event name.
     *
     * @return the suffix
     */
    public String getHipEndSuffix() {
        return HIP_END_SUFFIX;
    }

    /**
     * This function returns the prefix of each HSA API event name.
     *
     * @return the prefix
     */
    public String getHsaPrefix() {
        return HSA_PREFIX;
    }

    /**
     * This function returns the suffix of each HSA API begin event name.
     *
     * @return the suffix
     */
    public String getHsaBeginSuffix() {
        return HSA_BEGIN_SUFFIX;
    }

    /**
     * This function returns the suffix of each HSA API end event name.
     *
     * @return the suffix
     */
    public String getHsaEndSuffix() {
        return HSA_END_SUFFIX;
    }

    /**
     * This event occurs when a HIP operation has begun on one of the ROCm
     * Agents.
     *
     * @return the event name
     */
    public String getHipOperationBegin() {
        return HIP_OPERATION_BEGIN;
    }

    /**
     * This event occurs when a HIP operation has ended on one of the ROCm
     * Agents.
     *
     * @return the event name
     */
    public String getHipOperationEnd() {
        return HIP_OPERATION_END;
    }

    /**
     * This event occurs when a HSA operation has begun on one of the ROCm
     * Agents.
     *
     * @return the event name
     */
    public String getHsaOperationBegin() {
        return HSA_OPERATION_BEGIN;
    }

    /**
     * This event occurs when a HSA operation has ended on one of the ROCm
     * Agents.
     *
     * @return the event name
     */
    public String getHsaOperationEnd() {
        return HSA_OPERATION_END;
    }

    /**
     * This event is emitted during the initialization phase to identify the
     * different agents during the execution.
     *
     * @return the event name
     */
    public String getHsaHandleType() {
        return HSA_HANDLE_TYPE;
    }

    /**
     * This event is a call that will result in a memory copy between the host
     * and a device or between two devices.
     *
     * @param eventName
     *            the event name
     * @return whether the event corresponds to a memory copy
     */
    public boolean isMemcpyBegin(String eventName) {
        return eventName.startsWith(HIP_MEMCPY_PREFIX) && eventName.endsWith(HIP_BEGIN_SUFFIX);
    }

    /**
     * This event is a call that will result in a kernel launch that will happen
     * on a ROCm agent.
     *
     * @param eventName
     *            the event name
     * @return whether the event corresponds to a kernel launch
     */
    public boolean isLaunchBegin(String eventName) {
        return (eventName.startsWith(HIP_LAUNCH_PREFIX) || eventName.startsWith(HIP_EXT_LAUNCH_PREFIX)) && eventName.endsWith(HIP_BEGIN_SUFFIX);
    }

    // ------------------------------------------------------------------------
    // Event field names
    // ------------------------------------------------------------------------

    /**
     * Field indicating the Thread ID where the event described happened.
     *
     * @return the field name
     */
    public String fieldThreadId() {
        return THREAD_ID;
    }

    /**
     * Field indicating the Queue ID where the event was dispatched.
     *
     * @return the field name
     */
    public String fieldQueueId() {
        return QUEUE_ID;
    }

    /**
     * Field indicating the Agent ID where the event has happened.
     *
     * @return the field name
     */
    public String fieldAgentId() {
        return AGENT_ID;
    }

    /**
     * Field indicating an identification number that match one API call to a
     * HIP or HSA operation.
     *
     * @return the field name
     */
    public String fieldCorrelationId() {
        return CORRELATION_ID;
    }

    /**
     * Field indicating the kernel function name.
     *
     * @return the field name
     */
    public String fieldKernelName() {
        return KERNEL_NAME;
    }

    /**
     * Field indicating the kind of memory copy, i.e. whether it is from or to
     * the host or between GPUs.
     *
     * @return the field name
     */
    public String fieldMemcpyKind() {
        return MEMCPY_KIND;
    }

    /**
     * Field indicating the kernel function name for operation events.
     *
     * @return the field name
     */
    public String fieldOperationName() {
        return OP_KERNEL_NAME;
    }
}
