/*
 * hotkey.c
 *
 * Hot key functions
 *
 */

#include "frotz.h"
#include "s5api.h"

extern short restore_undo (struct sg *g);

extern short read_number (struct sg *g);

extern short read_yes_or_no (struct sg *g, const char *);

extern void replay_open (struct sg *g);
extern void replay_close (struct sg *g);
extern void record_open (struct sg *g);
extern void record_close (struct sg *g);

//extern void seed_random (short);

/*
 * hot_key_debugging
 *
 * ...allows user to toggle cheating options on/off.
 *
 */

short hot_key_debugging (struct sg *g)
{

    print_string (g, "Debugging options\n");

    g->option_attribute_assignment = read_yes_or_no (g, "Watch attribute assignment");
    g->option_attribute_testing = read_yes_or_no (g, "Watch attribute testing");
    g->option_object_movement = read_yes_or_no (g, "Watch object movement");
    g->option_object_locating = read_yes_or_no (g, "Watch object locating");

    return FALSE;

}/* hot_key_debugging */

/*
 * hot_key_help
 *
 * ...displays a list of all hot keys.
 *
 */

short hot_key_help (struct sg *g) {

    print_string (g, "Help\n");

    print_string (g,
	"\n"
	"Ctrl-D  debugging options\n"
	"Ctrl-S  help\n"
	"Ctrl-N  new game\n"
	"Ctrl-P  playback on\n"
	"Ctrl-R  recording on/off\n"
	"Ctrl-U  undo one turn\n"
	"Ctrl-X  exit game\n");

    return FALSE;

}/* hot_key_help */

/*
 * hot_key_playback
 *
 * ...allows user to turn playback on.
 *
 */

short hot_key_playback (struct sg *g)
{

    print_string (g, "Playback on\n");

    if (!g->istream_replay)
	replay_open (g);

    return FALSE;

}/* hot_key_playback */

/*
 * hot_key_recording
 *
 * ...allows user to turn recording on/off.
 *
 */

short hot_key_recording (struct sg *g)
{

    if (g->istream_replay) {
	print_string (g,"Playback off\n");
	replay_close (g);
    } else if (g->ostream_record) {
	print_string (g,"Recording off\n");
	record_close (g);
    } else {
	print_string (g,"Recording on\n");
	record_open (g);
    }

    return FALSE;

}/* hot_key_recording */

/*
 * hot_key_seed
 *
 * ...allows user to seed the random number seed.
 *
 */

/*static short hot_key_seed (void)
{

    print_string ("Seed random numbers\n");

    print_string ("Enter seed value (or return to randomize): ");
    //seed_random (read_number ());

    return FALSE;

}
*/
/* hot_key_seed */

/*
 * hot_key_undo
 *
 * ...allows user to undo the previous turn.
 *
 */

short hot_key_undo (struct sg *g)
{

    print_string (g,"Undo one turn\n");

    if (restore_undo (g)) {

	if (g->h_version >= V5) {          /* for V5+ games we must */
	    store (g, 2);                  /* store 2 (for success) */
	    return TRUE;                /* and abort the input   */
	}

	if (g->h_version <= V3) {          /* for V3- games we must */
	    z_show_status (g);           /* draw the status line  */
	    return FALSE;               /* and continue input    */
	}

    } else print_string (g, "No more undo information available.\n");

    return FALSE;

}/* hot_key_undo */

/*
 * hot_key_restart
 *
 * ...allows user to start a new game.
 *
 */

short hot_key_restart (struct sg *g)
{

    print_string (g,"New game\n");

    if (read_yes_or_no (g,"Do you wish to restart")) {

	z_restart (g);
	return TRUE;

    } else return FALSE;

}/* hot_key_restart */

/*
 * hot_key_quit
 *
 * ...allows user to exit the game.
 *
 */

short hot_key_quit (struct sg *g)
{

    print_string (g, "Exit game\n");

    if (read_yes_or_no (g, "Do you wish to quit")) {

	z_quit (g);
	return TRUE;

    } else return FALSE;

}/* hot_key_quit */

/*
 * handle_hot_key
 *
 * Perform the action associated with a so-called hot key. Return
 * true to abort the current input action.
 *
 */

short handle_hot_key (struct sg *g, zchar key)
{

    if (g->cwin == 0) {

	short aborting;

	print_string (g, "\nHot key -- ");

	switch (key) {
	    case ZC_HKEY_RECORD: aborting = hot_key_recording (g); break;
	    case ZC_HKEY_PLAYBACK: aborting = hot_key_playback (g); break;
	    case ZC_HKEY_UNDO: aborting = hot_key_undo (g); break;
	    case ZC_HKEY_RESTART: aborting = hot_key_restart (g); break;
	    case ZC_HKEY_QUIT: aborting = hot_key_quit (g); break;
	    case ZC_HKEY_DEBUG: aborting = hot_key_debugging (g); break;
	    case ZC_HKEY_HELP: aborting = hot_key_help (g); break;
	}

	if (aborting)
	    return TRUE;

	print_string (g,"\nContinue input...\n");

    }

    return FALSE;

}/* handle_hot_key */
