/* Copyright (c) 2000 - 2001 by Kevin Forchione.  All Rights Reserved. */
/*
 *  TADS LIBRARY EXTENSION
 *  COMPOSITE.T				
 *  version 3.0
 *
 *		The module facilitates the creation of Composite objects by
 *		providing the parser with the appropriate component to handle
 *		the action. 
 *
 *		A desk, for example, can be constructed as a composite object 
 *		consisting of a desk parent object, a surface component, and an
 *		openable component. Actions are automatically redirected to the
 *		appropriate object by working down the parent / component tree
 *		and assessing which object is capable of completing the action.
 *
 *		The model recursively works down the Composite / Component tree
 *		allowing an author to build Composite objects with several
 *		levels of Components.
 *
 *----------------------------------------------------------------------
 *  REQUIREMENTS
 *
 *      + HTML TADS 2.5.0 or later
 *      + Should be #included after ADV.T and STD.T 
 *
 *----------------------------------------------------------------------
 *  IMPORTANT LIBRARY INTERFACE AND MODIFICATION
 *
 *      This module replaces:
 *
 *			initSearch()
 *
 *----------------------------------------------------------------------
 *  COPYRIGHT NOTICE
 *
 *  	You may modify and use this file in any way you want, provided that
 *		if you redistribute modified copies of this file in source form, the
 *   	copies must include the original copyright notice (including this
 *   	paragraph), and must be clearly marked as modified from the original
 *   	version.
 *
 *------------------------------------------------------------------------------
 *  REVISION HISTORY
 *
 *		06-Oct-00:	Creation.
 *		11-Oct-00:	Renamed classes such as SurfCont to SurfaceContainer
 *				and removed defined() check from iteration loop in 
 *				disambigXobj() methods. 
 *				    Added preparseCmd() and reparsePronoun() functions
 *				to handle pronoun disambiguation for Composite class
 *				objects.
 *      03-Nov-00:  Changed the way makeComponents() saves vocabulary
 *              from parent objects so that components can be called
 *              "desk top" or "top of desk".
 *                  Changed Component sdesc to point to self.componentParent
 *              instead of self.location to allow for components
 *              separated from their parents to be properly described.
 *                  Moved the API for disambiguating Composite class
 *              objects to the componentParser object.          
 *      24-Mar-01:  Overhauled the system completely to take advantage
 *              of dobjCheck() and iobjCheck(), simplifying the 
 *              process.
 */

#ifndef COMPOSITE_MODULE 
#define COMPOSITE_MODULE

#pragma C+

/* Behavior Levels */
#define BEHAVIOR_ACTIONED       2
#define BEHAVIOR_VERIFIED       1
#define BEHAVIOR_NONE           0

getVerbInfo: function;
superclassCount: function;
initComposite: function;
subList: function;

class Component: object
    sdesc = { 
        global.useCompositeDesc ? self.compositeDesc :
            self.componentDesc;
    }
    
    compositeDesc = { 
        self.myComposite ? self.myComposite.sdesc :
            self.componentDesc;
    }
    
    componentDesc = "Component Class Object"
    
    getBehaviorLevel(verification, action) = {        
        if (verification && defined(self, verification) != nil)
            if (action && defined(self, action) != nil)
                return BEHAVIOR_ACTIONED;
            else return BEHAVIOR_VERIFIED;
        else return BEHAVIOR_NONE;
    }
;

class Composite: Component
    componentList = []
    
    dobjCheck( actor, verb, iobj, prep ) = {
        local i, len, ret, nounList;
        
        ret = self.buildLists(true, actor, verb, iobj, prep);
        
        if (length(ret) == 0)
            return;
            
        nounList = ret[1][2];
        
        /*
         *  We'll loop through all the objects in the noun list and
         *  perform execCommand for each.
         */
        len = length(nounList);
        for (i = 1; i <= len; i++)
        {
            local dobj = nounList[i];
            
            if (len > 1)
                "\n<<dobj.multisdesc>>: ";
                
            if (superclassCount(nounList, dobj) == 1)
                global.useCompositeDesc = true;
            else global.useCompositeDesc = nil;
                
            execCommand(actor, verb, dobj, prep, iobj);
            
            global.useCompositeDesc = nil;
        }
        
        /* exit this object if we've processed a noun list */
        if (len > 0)
            exitobj;
    }
    
    iobjCheck(actor, verb, dobj, prep) = {
        local i, len, ret, nounList, verbList, iobj;
        
        ret = self.buildLists(nil, actor, verb, self, prep);
        
        
        if (length(ret) == 0)
            return;
            
        nounList = ret[1][2];
        verbList = ret[2];

        len = length(nounList);
            
        if (len == 0)
            return;
            
        if (superclassCount(nounList, dobj) == 1)
            global.useCompositeDesc = true;
        else global.useCompositeDesc = nil;
    
        /*
         *  More than one object defines a verification or action method
         *  for the verb. We let the parser disambiguate in this case.
         */
        if (len > 1)
        {
retry:;
            nounList = self.disambiguate(actor, verb, prep, dobj, PRO_RESOLVE_IOBJ, 
                verbList[1], nounList);
                    
            if (length(nounList) > 1)
            {
                "Let's try it again: ";
                goto retry;
            }
        }
        
        /*
         *  Perform execCommand for the iobj and exit this object.
         */
        iobj = nounList[1];
        global.useCompositeDesc = true;
        execCommand(actor, verb, dobj, prep, iobj);
        global.useCompositeDesc = nil;
        exitobj;
    }
    
    buildLists( isDobjCheck, actor, verb, iobj, prep ) = {
        local verbList;
        local verifiedList;
        local nounList = [];
        local verification;
        local action;
        local highVal;
        local i;
        local len;
        
        len = length(self.componentList);
        /* no components exists. Return an empty list */
        if (len == 0)
            return [];
            
        /*
         *  Get the verification and action method pointers.
         */
        verbList    = getVerbInfo(verb, prep);
        isDobjCheck ? verification = verbList[1] 
                    : verification = verbList[2];
        isDobjCheck ? action = verbList[3] 
                    : action = verbList[4];
        
        /*
         *  Get the composite object's getBehaviorLevel value. If this
         *  value is greater than BEHAVIOR_VERIFIED then this object
         *  is capable of completing the action of the verb and we
         *  return.
         */
        highVal = self.getBehaviorLevel(verification, action);
        
        /*
         *  Build a candidates list of component objects. This list
         *  contains sublists that indicate the defines action value
         *  for each component.
         */
        for (i = 1; i <= len; ++i)
        {
            local o, val;
            
            o   = self.componentList[i];
            val = o.getBehaviorLevel(verification, action);
            
            if (val > highVal)
            {
                nounList    = [val [o]];
                highVal     = val;
            }
            else if (length(nounList) > 0
                && val == highVal)
            {
                nounList[2] += o;
            }
        }
        
        /*
         *  No components define either an action or verification method
         *  for the verb. The Composite takes precedence.
         */
        len = length(nounList);
        if (len == 0)
            return [];
            
        /*
         *  Build a verification list -- object's in this list have
         *  passed verification.
         */
       verifiedList = self.checkVerification(nounList[2],
            verification, actor, (isDobjCheck ? iobj : nil));
          
        /*
         *  if any objects are left in the verification list after
         *  verifying it then these are the ones
         *  we'll favor for action.
         */
        if (length(verifiedList) != 0)
            nounList[2] = verifiedList;
            
        return [nounList, verbList];
    }
            
    /*
     *  Used to handle cases where verb handling by multiple components
     *  is undesirable. First asks a disambiguation question and then 
     *  proceeds to parse, disambiguate, and resolve the player's input
     *  to a single object.
     */
    disambiguate( actor, verb, prep, otherobj, usageType, verProp, nounList ) = {
        local i, len, str, obj;
        local tokenList, typeList;
        local objList;
        
        str = self.askDisambigQuestion(nounList);
        
        tokenList = parserTokenize(str);

        if (tokenList == nil)
        {
            "The object name is invalid! ";
            abort;
        }
        
        typeList = parserGetTokTypes(tokenList);
        
        objList = parseNounList(tokenList, typeList, 1, true, nil,
            nil);  

        /* 
         *  the parser encountered a syntax error. parseNounList()
         *  displayed an error message.
         */
        if (objList == nil)
            abort;
            
        if (length(objList) == 1)
            self.noObjectMatch(str, car(objList), tokenList, 
                typeList);
            
        objList = self.removeNonComponents(tokenList, nounList, objList);
        
        if (length(objList) == 1)
        {
            "This doesn't appear to be one of the components of
            <<self.thedesc>>. ";
            abort;
        }
        
        objList = parserResolveObjects(actor, verb, prep, otherobj, 
            usageType, verProp, tokenList, objList, nil);
             
        if (objList[1] == PRS_SUCCESS)
        {
            /* success! return the objects, which follow the status code */
            return cdr(objList);
        }
        else if (objList[1] == PRSERR_DISAMBIG_RETRY)
        {
            /* run the new command, which is in the second element */
            parserReplaceCommand(objList[2]);
        }
        else
        {
            /* we are in non-silent mode, so the resolver displayed an error */
            abort;
        }
    }
    
    askDisambigQuestion(nounList) = {
        local i, len, str;
        
        len = length(nounList);
                    
        "Which part of <<self.thedesc>> do you mean, ";
        for (i = 1; i <= len; ++i)
        {
            nounList[i].thedesc;
            
            if (i < len)
            {
                if (i+1 < len)
                        ", ";
                else if (len == 2)
                        " or ";
                else
                    ", or ";
            }
        }
        "?\b>";
        
        str = input();
        
        return str;
    }
        
    removeNonComponents(tokenList, nounList, objList) = {
        local i, j, ln, ln2, newObjList = [], ele;
        
        newObjList += objList[1];
        
        objList = cdr(objList);
        
        ln = length(objList);
        for (i = 1; i <= ln; ++i)
        {
            local wPos, wLen, tmpList = [];
            
            ele = objList[i];
            
            wPos    = ele[1];
            wLen    = ele[2];
            
            tmpList += subList(ele, 1, 2);
            ele = subList(ele, 3);
            
            ln2 = length(ele);
            for (j = 1; j <= ln2; j = j + 2)
            {
                local o, t;
                
                o = ele[j];
                t = ele[j+1];
                                   
                if (o == nil)
                {
                    if ((t & PRSFLG_UNKNOWN) != 0)
                    {
                        "\nI don't know the word \"";
                        say(tokenList[wPos]);
                        "\". ";
                    }
                    else if ((t & PRSFLG_ALL) != 0)
                    {
                        "\nYou can't use multiple indirect 
                        objects. ";
                    }
                    else
                    {
                        "\nThis noun phrase is not valid in this 
                        context. ";
                    }
                    abort;
                }
                else if (find(nounList, o))
                {
                    tmpList += o;
                    tmpList += t;
                }
            }
            if (length(tmpList) > 2)
                newObjList += [tmpList];
        }
        
        return newObjList;
    }
    
    noObjectMatch(str, start, tokenList, typeList) = {           
        local i, len, bothChar, specChar;
        
        len = length(tokenList);
        
        /* a syntactically valid noun phrase, but no matching objects */
        if (start > len)
            abort;
            
        
        for (i = start; i <= len; ++i)
        {
            if ((typeList[i] & PRSTYP_SPEC) != 0)
            {
                switch(tokenList[i])
                {
                    case ',':
                        continue;
                    case 'B':
                        bothChar = true;
                        continue;
                    default:
                        specChar = true;
                        continue;
                }
            }
            if ((typeList[i] & PRSTYP_VERB) != 0)
                break;
        }
        
        if (i > len)
        {
            if (len == 1)
            {
                if (bothChar)
                    /* 
                     *  both is not valid for multiple indirect objects
                     */
                    "\nYou can't use multiple indirect objects. ";
                else if (specChar)
                    /* This is a special char */
                    "\nThis noun phrase is not valid in this context. ";
                else
                    /* this doesn't appear to be part of a noun phrase */
                    "\nI beg your pardon? ";
            }
            else
                /* 
                 *  there appear to be extra words after a syntactically
                 *  valid noun phrase.
                 */
                "\nThere seem to be words after the object name that 
                I can't use. ";
            abort;
        }
        else
        {
            // run the string as a command
            parserReplaceCommand(str);
        }
    }

    checkVerification( lst, verification, actor, obj) = {
        local i, len, newList = [], stat, ele;
        
        len = length(lst);
        for (i = 1; i <= len; ++i)
        {
            ele = lst[i];
            
            stat = outhide(true);
            
            if (obj)
                ele.(verification)(actor, obj);
            else
                ele.(verification)(actor);
            
            stat = outhide(nil);
            if (stat == nil)
                newList += ele;
        }
        
        return newList;
    }
;

superclassCount: function(lst, obj)
{
    local cls = firstsc(obj);
    local i, len, cnt = 0;
    
    len = length(lst);
    for (i = 1; i <= len; ++i)
        if (firstsc(lst[i]) == cls)
            cnt++;
            
    return cnt;
}

/*
 *  Returns verb information in a consistent format. The layout is
 *  as follows: 
 *      verDoPtr, verIoPtr, doActionPtr, ioActionPtr, dobjFirstFlag
 */
getVerbInfo: function(verb, prep)
{
    local tmpList, verbList = [];
    
    if (prep)
    {
        tmpList     = verbinfo(verb, prep);
        verbList    += tmpList[1];
        verbList    += tmpList[2];
        verbList    += verb.doActionPtr(prep);
        verbList    += tmpList[3];
        verbList    += tmpList[4];
    }
    else
    {
        tmpList     = verbinfo(verb);
        verbList    += tmpList[1];
        verbList    += nil;
        verbList    += tmpList[2];
        verbList    += nil;
        verbList    += nil;
    }
    
    return verbList;
}

subList: function(lst, offset, ...)
{
    local i, ln, newList = [], ele;
    local len = length(lst);
    
    if (argcount == 3) len = getarg(3);
    
    ln = length(lst);
    for (i = offset; i <= ln; ++i)
    {
        ele = lst[i];
        
        if (i > len)
            break;
        else newList += [ele];
    }
  
    return newList;
}

/*----------------------------------------------------------------------
 *  Provide doAction pointers for ADV.T's existing vocabulary.
 */
modify deepverb
    doActionPtr(prep) = {return nil;}
;
modify askVerb
    doActionPtr(prep) = {
        switch(prep)
        {
            case aboutPrep:
                return &doAskAbout;
            default:
                return nil;
        }
    }
;
modify attachVerb
    doActionPtr(prep) = {
        switch(prep)
        {
            case toPrep:
                return &doAttachTo;
            default:
                return nil;
        }
    }
;
modify attackVerb
    doActionPtr(prep) = {
        switch(prep)
        {
            case withPrep:
                return &doAttackWith;
            default:
                return nil;
        }
    }
;
modify cleanVerb
    doActionPtr(prep) = {
        switch(prep)
        {
            case withPrep:
                return &doCleanWith;
            default:
                return nil;
        }
    }
;
modify detachVerb
    doActionPtr(prep) = {
        switch(prep)
        {
            case fromPrep:
                return &doDetachFrom;
            default:
                return nil;
        }
    }
;
modify digVerb
    doActionPtr(prep) = {
        switch(prep)
        {
            case withPrep:
                return &doDigWith;
            default:
                return nil;
        }
    }
;
modify dropVerb
    doActionPtr(prep) = {
        switch(prep)
        {
            case onPrep:
                return &doPutOn;
            default:
                return nil;
        }
    }
;
modify giveVerb
    doActionPtr(prep) = {
        switch(prep)
        {
            case toPrep:
                return &doGiveTo;
            default:
                return nil;
        }
    }
;
modify inVerb
    doActionPtr(prep) = {
        switch(prep)
        {
            case inPrep:
                return &doEnterIn;
            case onPrep:
                return &doEnterOn;
            case withPrep:
                return &doEnterWith;
            default:
                return nil;
        }
    }
;
modify lockVerb
    doActionPtr(prep) = {
        switch(prep)
        {
            case withPrep:
                return &doLockWith;
            default:
                return nil;
        }
    }
;
modify moveVerb
    doActionPtr(prep) = {
        switch(prep)
        {
            case toPrep:
                return &doMoveTo;
            case withPrep:
                return &doMoveWith;
            default:
                return nil;
        }
    }
;
modify plugVerb
    doActionPtr(prep) = {
        switch(prep)
        {
            case inPrep:
                return &doPlugIn;
            default:
                return nil;
        }
    }
;
modify putVerb
    doActionPtr(prep) = {
        switch(prep)
        {
            case inPrep:
                return &doPutIn;
            case onPrep:
                return &doPutOn;
            default:
                return nil;
        }
    }
;
modify removeVerb
    doActionPtr(prep) = {
        switch(prep)
        {
            case fromPrep:
                return &doRemoveFrom;
            default:
                return nil;
        }
    }
;
modify screwVerb
    doActionPtr(prep) = {
        switch(prep)
        {
            case withPrep:
                return &doScrewWith;
            default:
                return nil;
        }
    }
;
modify showVerb
    doActionPtr(prep) = {
        switch(prep)
        {
            case toPrep:
                return &doShowTo;
            default:
                return nil;
        }
    }
;
modify takeVerb
    doActionPtr(prep) = {
        switch(prep)
        {
            case inPrep:
                return &doTakeOut;
            case fromPrep:
                return &doTakeOut;
            case offPrep:
                return &doTakeOff;
            case onPrep:
                return &doTakeOff;
            case outPrep:
                return &doTakeOut;
            default:
                return nil;
        }
    }
;
modify tellVerb
    doActionPtr(prep) = {
        switch(prep)
        {
            case aboutPrep:
                return &doTellAbout;
            default:
                return nil;
        }
    }
;
modify throwVerb
    doActionPtr(prep) = {
        switch(prep)
        {
            case atPrep:
                return &doThrowAt;
            case toPrep:
                return &doThrowTo;
            default:
                return nil;
        }
    }
;
modify turnVerb
    doActionPtr(prep) = {
        switch(prep)
        {
            case toPrep:
                return &doTurnTo;
            case withPrep:
                return &doTurnWith;
            default:
                return nil;
        }
    }
;
modify typeVerb
    doActionPtr(prep) = {
        switch(prep)
        {
            case onPrep:
                return &doTypeOn;
            default:
                return nil;
        }
    }
;
modify unlockVerb
    doActionPtr(prep) = {
        switch(prep)
        {
            case withPrep:
                return &doUnlockWith;
            default:
                return nil;
        }
    }
;
modify unplugVerb
    doActionPtr(prep) = {
        switch(prep)
        {
            case fromPrep:
                return &doUnplugFrom;
            default:
                return nil;
        }
    }
;
modify unscrewVerb
    doActionPtr(prep) = {
        switch(prep)
        {
            case withPrep:
                return &doUnscrewWith;
            default:
                return nil;
        }
    }
;

/*----------------------------------------------------------------------
 *  Setup the component list for each Composite object 
 */
initComposite: function
{
    local o;
    
    o = firstobj(Component);
    while(o != nil)
    {
        if (o.myComposite)
            o.myComposite.componentList += o;
        o = nextobj(o, Component);
    }
}

#pragma C-

#endif /* composite_module */
