/* Perdition's Flames - startup, verbs, general classes */
// copyright (c) 1993, 2002 by Michael J. Roberts.

/*
    Award points for an action.  The scoreObj argument specifies
    an object that tracks whether the particular action has been
    accomplished before.  When we first score for a particular
    action, we set scoreObj.isscored := true.  On subsequent
    performances of the action, we check scoreObj.isscored, and
    if true, we ignore the awardPoints call.  This ensures that
    performing a scoring action only awards points once.  The
    choice of the scoreObj is arbitrary, but it must be unique
    (each action must have its own scoreObj), and thus it should
    probably be somehow tightly associated with the puzzle whose
    solution awarded the points.
*/
awardPoints: function(amount, scoreObj)
{
    if (not scoreObj.isscored)
    {
        scoreObj.isscored := true;
        incscore(amount);
    }
}

/* provide exit options */
exitOptions: function
{
    "\bYou may restore a previously saved game position, start
    the game over from the beginning, undo the
    last move, or quit the game. ";

    for (;;)
    {
        local resp;

        "\nPlease enter RESTORE, RESTART, UNDO, or QUIT: >";
        resp := upper(input());     /* get input, convert to uppercase */
        if (resp = 'RESTORE')
        {
            resp := askfile('File to restore');       /* find filename */
            if (resp = nil ) "Restore failed. ";
            else if (restore(resp)) "Restore failed. ";
            else
            {
                /*
                 *   We've successfully restored a game.  Reset the status
                 *   line's score/turn counter, and resume play.
                 */
                scoreStatus(global.score, global.turnsofar);
                Me.location.lookAround(true);
                abort;
            }
        }
        else if (resp = 'RESTART')
        {
            /*
             *   We're going to start over.  Reset the status line's
             *   score/turn counter and start from the beginning.
             */
            scoreStatus(0, 0);
            restart(initRestart, global.initRestartParam);
            abort;
        }
        else if (resp = 'UNDO')
        {
            if (undo())
            {
                "(Undoing one command)\b";
                Me.location.lookAround(true);
                scoreStatus(global.score, global.turnsofar);
                abort;
            }
            else
                "Sorry, no undo information is available.";
        }
        else if (resp = 'QUIT')
        {
            /*
             *   We're quitting the game.  Do any final activity necessary,
             *   and exit.
             */
            terminate();
            quit();
            abort;
        }
    }
}

scoreRank: function
{
    local s := global.score;

    "In a total of << global.turnsofar >> turns, you have achieved
    a score of << s >> points (out of a possible << global.maxscore
    >> points).  This gives you a rank of ";

    if (s < 50) "rotting corpse";
    else if (s < 125) "lost soul";
    else if (s < 225) "wandering spirit";
    else if (s < 350) "disembodied wayfarer";
    else if (s < 500) "honored dead";
    else if (s < 666) "ethereal explorer";
    else "master adventurer (deceased)";

    ".";
}

preinit: function
{
    local o;
    
    global.lamplist := [];
    for (o := firstobj(lightsource) ; o <> nil ; o := nextobj(o, lightsource))
        if (o.islamp) global.lamplist += o;

    initSearch();
    writeGameInfo(getGameInfo());
}

terminate: function
{
}

pardon: function
{
    "I beg your pardon? ";
}

numObj: basicNumObj
    verIoTuneTo(actor) = {}
    ioTuneTo(actor, dobj) = { dobj.doTuneTo(actor, self); }
;

strObj: basicStrObj
;

goToSleep: function
{
}

darkTravel: function
{
}

global: object
    turnsofar = 0                            // no turns have transpired so far
    score = 0                            // no points have been accumulated yet
    maxscore = 666                                    // maximum possible score
    verbose = nil                             // we are currently in TERSE mode
    awakeTime = 0               // time that has elapsed since the player slept
    sleepTime = 400     // interval between sleeping times (longest time awake)
    lastMealTime = 0              // time that has elapsed since the player ate
    eatTime = 200         // interval between meals (longest time without food)
    lamplist = []              // list of all known light providers in the game
    scoreNotify = true                     // notify player of changes to score
    scoreNotifyOnce = nil                       // no notification of score yet
;

version: object
    sdesc = "\b\b\bPerdition's Flames
    \nTADS Interactive Fiction
    \nBy Michael J.\ Roberts
    \bVersion <<self.vsnnum>>/Release <<self.vsnrls>>
    \nCopyright (c) 1993, 2002 by Michael J.\ Roberts.  All Rights Reserved."

    vsnnum = '1.01'
    vsnrls = '020203'
;

Me: basicMe
    maxweight = 1000
    maxbulk = 1000
    ropeSkill = 0
    verIoClipTo(actor) = {}
    ioClipTo(actor, dobj) = { dobj.doClipTo(actor, self); }
    ioSynonym('ClipTo') = 'AttachTo'
;

/* quote - takes a list of lines of text, and an author */
quote: function(text, author)
{
    local i, tot;

    "\b\b\b\b";
    tot := length(text);
    for (i := 1 ; i <= tot ; ++i)
    {
        "\n\t\t\t<< text[i] >>";
    }
    "\n\t\t\t\ \ \ -- <<author>>\b\b\b\b";
}

// seed random number generator if "norandomize" isn't typed right away
seedRand: function(parm)
{
    if (not global.noRandomize) randomize();
}

// suppress startup seeding random number generator - for QA scripts, mostly
noRandVerb: deepverb
    verb = 'norandomize'
    action(actor) = { "Okay."; global.noRandomize := true; abort; }
;

init: function
{
    // set our own statue line mode as the very first thing
    scoreStatus(0, 0);

    // show game copyright banner
    "Perdition's Flames -- Copyright 1993, 2002 Michael J.\ Roberts\b";

    // ask a question to make sure they have a copy of the documentation
    introQuestion();
    if (not global.copyOK) quit();

    // check for a file to restore specified as a startup parameter
    if (restore(nil) = nil)
    {
        "\b[Restoring saved game]\b";
        scoreStatus(global.score, global.turnsofar);
        Me.location.lookAround(true);
        return;
    }

    "\b\b\b\b*** You have died.\ ***
    \b\tAll is dark and quiet.  There is no sensation, no time.
    Your mind floats peacefully in a void.  You perceive nothing,
    you feel nothing, you think nothing.  Sleep without dreams.
    \b\tAll is hazy and gray.  Sensation is vague and indistinct.
    Your mind is sluggish, sleepy.  You see gray shapes in a
    gray fog; you hear distant, muffled sounds.  You think, but your
    thoughts are fleeting, disconnected, momentary flashes of
    light in a dark night.  Time is still frames
    separated by eons of nothing, brief awakenings in a long
    sleep.
    \b\tAll is clear and sharp.  Sensation crystalizes from a fog.
    You see, you hear, you feel.  Your mind awakens; you become
    aware of a place, and a time.
    \b\tYou are on a boat.
    \b\b\b";

    version.sdesc;

    quote(['Hell is paved with good intentions.'], 'Fortune cookie');
    
    Me.location := boatseat;
    Me.location.lookAround(true);
    boatmain.isseen := true;

    setdaemon(turncount, nil);

    // arrange to seed random number generator if not told otherwise
    setfuse(seedRand, 3, nil);
}

theAfterlife: thing
    noun = 'afterlife' 'death'
    sdesc = "afterlife"
;

theHell: thing
    noun = 'hell' 'hades' 'inferno' 'netherworld' 'regions'
    adjective = 'infernal' 'nether'
;

theHeaven: thing
    noun = 'heaven' 'paradise'
;


// An Amulet.  Provides special protection that allows getting
// past lake of blood.  Only one amulet can be worn at once.
amuletItem: clothingItem
    verDoWear(actor) =
    {
        if (self.location <> actor)
            "You're not holding << self.thedesc >>.";
        else
            pass verDoWear;
    }
    doWear(actor) =
    {
        if (actor.currentAmulet <> nil and actor.currentAmulet.isworn
            and actor.currentAmulet.isIn(actor))
            "As you put on <<thedesc>>, you feel strangely disoriented.
            Everything grows dark and you feel very dizzy and faint.
            You realize there must be some strange interaction between
            <<thedesc>> and <<actor.currentAmulet.thedesc>>, so you
            take <<thedesc>> back off; as you do, the disorientation
            lifts.";
        else
        {
            inherited.doWear(actor);
            " A feeling of strength and safety falls over you.";
            actor.currentAmulet := self;
        }
    }
    doTake(actor) =
    {
        inherited.doTake(actor);
        awardPoints(50, self);
    }
;


compoundWord 'down' 'from' 'down_from';
compoundWord 'down' 'off' 'down_off';
compoundWord 'down' 'offof' 'down_off_of';
compoundWord 'open' 'with' 'open_with';

statusVerb: sysverb
    verb = 'status'
    action(actor) =
    {
        "You are dead.  Your flesh seems to be in fairly good
        shape (no telltale signs of rotting are evident). ";
        abort;
    }
;

notifyVerb: sysverb
    verb = 'notify'
    action(actor) =
    {
        global.scoreNotify := not global.scoreNotify;
        if (global.scoreNotify)
            "You will now be notified of score changes.";
        else
            "You will no longer be notified of score changes.
            If you change your mind and want to be notified
            again, just type NOTIFY again.";
        abort;
    }
;

timeVerb: sysverb
    verb = 'time'
    sdesc = "time"
    action(actor) =
    {
        "It is sometime during eternity.";
        abort;
    }
;

tieVerb: deepverb
    verb = 'tie'
    sdesc = "tie"
    prepDefault = toPrep
    ioAction(toPrep) = 'TieTo'
;

lassoVerb: deepverb
    verb = 'lasso'
    sdesc = "lasso"
    prepDefault = withPrep
    ioAction(withPrep) = 'LassoWith'
;

tuneVerb: deepverb
    verb = 'tune'
    sdesc = "tune"
    prepDefault = toPrep
    ioAction(toPrep) = 'TuneTo'
;

clipVerb: deepverb
    verb = 'clip'
    sdesc = "clip"
    prepDefault = toPrep
    ioAction(toPrep) = 'ClipTo'
;

buyVerb: deepverb
    verb = 'buy'
    sdesc = "buy"
    doAction = 'Buy'
;

getdownVerb: deepverb
    verb = 'get down' 'get down_from' 'get down_off' 'get down_off_of'
    sdesc = "get down from"
    doAction = 'Getdown'
    doDefault( actor, prep, io ) =
    {
        if ( actor.location and actor.location.location )
            return( [] + actor.location );
        else return( [] );
    }
;

grabVerb: deepverb
    sdesc = "grab"
    verb = 'grab' 'reach' 'reach for' 'grasp'
    doAction = 'Grab'
;

climbDownVerb: deepverb
    sdesc = "climb down"
    verb = 'climb down' 'climb out' 'climb out of'
    action(actor) = { actor.travelTo(actor.location.down); }
    doAction = 'Climbdown'
;

coverVerb: deepverb
    verb = 'cover'
    sdesc = "cover"
    prepDefault = withPrep
    ioAction(withPrep) = 'CoverWith'
;

flickVerb: deepverb
    sdesc = "flick"
    verb = 'flick'
    doAction = 'Flick'
;

lightVerb: deepverb
    sdesc = "light"
    verb = 'light'
    doAction = 'Light'
    ioAction(withPrep) = 'LightWith'
;

burnVerb: deepverb
    sdesc = "burn"
    verb = 'burn'
    doAction = 'Burn'
    ioAction(withPrep) = 'BurnWith'
;

lookOverVerb: deepverb
    sdesc = "look over"
    verb = 'look over'
    doAction = 'LookOver'
;

pryVerb: deepverb
    sdesc = "pry"
    verb = 'pry' 'pry open'
    prepDefault = withPrep
    ioAction(withPrep) = 'PryWith'
    ioAction(openwithPrep) = 'PryWith'
;

wadeVerb: deepverb
    sdesc = "wade into"
    verb = 'wade' 'wade in' 'wade into'
    doAction = 'Wade'
;

swimVerb: deepverb
    sdesc = "swim in"
    verb = 'swim in' 'swim'
    doAction = 'Swim'
;

pickVerb: deepverb
    sdesc = "pick"
    verb = 'pick'
    doAction = 'Pick'
;

fillVerb: deepverb
    sdesc = "fill"
    verb = 'fill'
    prepDefault = withPrep
    ioAction(withPrep) = 'FillWith'
;

filloutVerb: deepverb
    sdesc = "fill out"
    verb = 'fill out' 'fill in'
    doAction = 'Fillout'
;

writeVerb: deepverb
    sdesc = "write"
    verb = 'write'
    doAction = 'Write'
;

answerVerb: deepverb
    sdesc = "answer"
    verb = 'answer'
    doAction = 'Answer'
;

sharpenVerb: deepverb
    sdesc = "sharpen"
    verb = 'sharpen'
    prepDefault = withPrep
    ioAction(withPrep) = 'SharpenWith'
;

emptyVerb: deepverb
    sdesc = "empty"
    verb = 'empty'
    doAction = 'Empty'
;

pourVerb: deepverb
    sdesc = "pour"
    verb = 'pour'
    doAction = 'Pour'
    ioAction(toPrep) = 'PourTo'
    ioAction(inPrep) = 'PourIn'
    ioAction(onPrep) = 'PourOn'
;

douseVerb: deepverb
    sdesc = "douse"
    verb = 'douse'
    prepDefault = withPrep
    ioAction(withPrep) = 'DouseWith'
;

pumpVerb: deepverb
    sdesc = "pump"
    verb = 'pump'
    doAction = 'Pump'
;

creditsVerb: sysverb
    verb = 'credits'
    action(actor) =
    {
        version.sdesc;
        "\b\bGame design and implementation: Mike Roberts.
        \nTesting: Roger Dominick, John Eras, Jeff Laing, Steve McAdams,
        Bob Newell, Trevor Powell, Lon Thomas.";
        abort;
    }
;


prayVerb: deepverb
    verb = 'pray'
    action(actor) = { "Isn't it a little late for that?"; }
;

wailVerb: deepverb
    verb = 'wail' 'cry' 'howl' 'scream'
    action(actor) = { "Very impressive. Your throat is a bit sore now."; }
;

moveUpVerb: deepverb
    verb = 'move up' 'push up' 'pull up'
    sdesc = "move up"
    doAction = 'Moveup'
;

moveDownVerb: deepverb
    verb = 'move down' 'push down' 'pull down'
    sdesc = "move down"
    doAction = 'Movedown'
;

playVerb: deepverb
    verb = 'play'
    sdesc = "play"
    doAction = 'Play'
;

forPrep: Prep
    preposition = 'for'
    sdesc = "for"
;

openwithPrep: Prep
    preposition = 'open_with'
;

openPrep: Prep
    preposition = 'open'
;

downoffPrep: Prep
    preposition = 'down_off' 'down_off_of' 'down_from'
;

/* A battery - something that can go in a batteryCompartment */
class batteryItem: item
    noun = 'battery'
;

/*
   A battery compartment is a container can hold only one item at a time.
*/
class batteryCompartment: openable
    verIoPutIn(actor) =
    {
        if (self.isopen and length(self.contents) > 0)
            "You'll have to remove << self.contents[1].thedesc >> first.";
        else
            pass verIoPutIn;
    }
    ioPutIn(actor, dobj) =
    {
        if (isclass(dobj, batteryItem))
            pass ioPutIn;
        else
            "That doesn't seem to fit.";
    }
;

/*
   closedNestedRoom is similar to nestedRoom, but isn't an object
   that appears in the containing room.  It also doesn't display
   the enclosing location in the status line.
*/
class closedNestedRoom: room
    statusPrep = "in"
    outOfPrep = "out of"
    statusLine = "\^<< self.sdesc >>\n\t"
    verDoUnboard(actor) = {}
    doUnboard(actor) = { actor.travelTo(self.out); }
;

/*
   A fake item contained in a closedNestedRoom that the player can refer to.
   Set the vocabulary appropriately, and set the location to the appropriate
   closedNestedRoom.  In the closedNestedRoom, set the property myRefItem
   to point to this object.  The primary purpose of this object is for
   the command "get out".
*/
class closedRefItem: fixeditem
    sdesc = { self.location.sdesc; }
    ldesc = { self.location.lookAround(true); }
    verDoUnboard(actor) = { self.location.verDoUnboard(actor); }
    doUnboard(actor) = { self.location.doUnboard(actor); }
;

