////////////////////////////////////////////////////////////////////////
//  
//  ALT Library File: Clock 010123
//
//  Copyright (c) 2000, 2001 Kevin Forchione. All rights reserved.
//  Based on ADV.T (c) and STD.T (c) Michael Roberts.
//
//  This file is part of the ALT replacement library for ADV.T and 
//  STD.T and requires TADS 2.5.1 or later.
//
////////////////////////////////////////////////////////////////////////

#ifndef _CLOCK_H_
#define _CLOCK_H_

#pragma C+

#ifndef CLOCK_TIME_LIM_HH
#define CLOCK_TIME_LIM_HH 24
#endif
#ifndef CLOCK_TIME_LIM_MM
#define CLOCK_TIME_LIM_MM 59
#endif

#define CLOCK_TIME             1
#define CLOCK_TIME_HH          2
#define CLOCK_TIME_HH24        3
#define CLOCK_TIME_MM          4
#define CLOCK_TIME_HHMM        5
#define CLOCK_TIME_HH24MM      6
#define CLOCK_TIME_HHMMTT      7
#define CLOCK_TIME_AMPM        8

#define CLOCK_DAY              9
#define CLOCK_DAY_ABBR         10
#define CLOCK_DAY_FULL         11

#define CLOCK_DATE             12
#define CLOCK_DATE_AMERICAN    13
#define CLOCK_DATE_EUROPEAN    14  
#define CLOCK_DATE_YMD         15

#define CLOCK_ELAPSETIME       16
#define CLOCK_ELAPSETIME_MSG   17

/*
 * 	Clock: object
 *
 *	This is the Clock class which keeps track of various
 *	date and time values, advances the time when requested
 *	and converts numeric date and time values into parm values
 *	for statusLine display.
 */
class Clock: object
    /*
     *  nameOfMonth is a list of the names of the months.
     */
    nameOfMonth = ['January', 'February', 'March', 'April',
        'May', 'June', 'July', 'August', 'September', 'October',
        'November', 'December']
    /*
     *  getNameOfMonth(n)
     *
     *  Returns the name of month value for list position n.
     */
    getNameOfMonth(n) = {
        return self.nameOfMonth[n];
    }
    /*
     *  daysInMonth - the number of days in each month
     */
    daysInMonth = [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31]
    /*
     *  getDaysInMonth(n)
     *
     *  Return the number of days for the given month n (i.e. Jan = 1,
     *  Feb = 2, etc.)
     */
    getDaysInMonth(n) = {
        return self.daysInMonth[n];
    }
    /*
     * setDaysInMonth(y)
     *
     *  Set the number of days for February in daysInMonth list depending
     *  upon the year.
     */
    setDaysInMonth(y) = {
        if (y % 4 != 0)
            self.daysInMonth[2] = 28;
        else if (y % 400 == 0)
            self.daysInMonth[2] = 29;
        else if (y % 100 == 0)
            self.daysInMonth[2] = 28;
        else self.daysInMonth[2] = 29;
    }
	/*------------------------------------------------------------------
	 *	TIME
	 *
	 *	Once initialized by setClock() the time is
	 *	maintained as a cyclical value that represents the
	 *	minutes past midnight from 0 to 1439.
	 */
	time = 0 							// time representation in minutes
	getTime(...) = {
		local t, h, m, amOrPm, t24, i, parm;
		
		if (argcount > 0) parm = getarg(1);

        t = self.time;
		h = t / 60;
		t24 = h;
		m = t % 60;
		if ( h >= 12 ) {
			if ( h > 12 ) {
				h = h - 12;
			}
			amOrPm = ' pm';
		} else {
			amOrPm = ' am';
			if ( h == 0 )
			{
				h = 12;
			}
		}
		i = cvtstr( h ) + ':';
		if ( m < 10 ) {
			i = i + '0';
		}
				
		switch(parm) {
            case CLOCK_TIME_HH:
                return h;
            case CLOCK_TIME_HH24:
                return t24;
            case CLOCK_TIME_MM:
                return m;
	        case CLOCK_TIME_HHMM:
	            return i + cvtstr( m );
	        case CLOCK_TIME_HHMMTT:
	            return i + cvtstr( m ) + amOrPm;
            case CLOCK_TIME_HH24MM:
                return cvtstr( t24 ) + ':' + cvtstr( m);
		    default: 
		        return self.time;
		}
	}
	setTime(t) = {
    	if (datatype( t ) == DTY_SSTRING) {
    		local ret = self.parseTime( t );
		
    		if (ret)
    			t = ret[3];
    		else
    			t = 0;
    	}
	    self.time = t;
	}
	/*------------------------------------------------------------------
	 *	TIME RATE
	 *
	 *	The timeRate is the number of minutes per turn that time passes.
	 */
	timeRate = 1                        // time increment per turn in minutes
	setTimeRate(n) = {
	    self.timeRate = n;
	}
	getTimeRate(...) = {
	    return self.timeRate;
	}
    /*------------------------------------------------------------------
	 *	DAY
	 *		
	 *	Once initialized with setClock(), either directly
	 *	or through calculation, this maintains a 
	 *	cyclical value that represents the day of the week
	 *	with values from 0=Sun to 6=Sat.
	 */
    day = 2  
    getDay(...) = {
		local d, parm;

        if (argcount > 0) parm = getarg(1);
        
		switch( self.day ) {
			case nil:
				d = nil;
				break;
			case 0:
				d = 'Sunday';
				break;
			case 1:
				d = 'Monday';
				break;
			case 2:
				d = 'Tuesday';
				break;
			case 3:
				d = 'Wednesday';
				break;
			case 4:
				d = 'Thursday';
				break;
			case 5:
				d = 'Friday';
				break;
			case 6: 
				d = 'Saturday';
				break;
		}
		
		switch( parm ) { 
            case CLOCK_DAY_ABBR:
                return substr(d, 1, 3);
            case CLOCK_DAY_FULL:
                return d;
            default:
                return self.day;
		}
	}
    setDay(d) = {
        self.day = d;
    }
	/*
	 *	DATE ATTRIBUTES AND METHODS
	 *
	 *	Once intiliazed with setClock() the date is maintained as
	 *	a base number calculated to take into account the 
	 *	full century and year (year 2000 compliant!) as well
	 *	as leap year days.
	 */
	date = 730500
    getDate(...) = {
        local parm, b, r1, r2, t=0, t1=0, y, m, d, s;
        
        if (argcount == 1) parm = getarg(1);
    
        b = self.date;
        r1 = b/366;
        r2 = b/365;
    
        for (y = r1; y <= r2; ++y) {
            d = b - ((y-1)*365 + (y-1)/4 - (y-1)/100 + (y-1)/400);
        
            if (d < 1) break;
        }
        --y;
        d = b - ((y-1)*365 + (y-1)/4 - (y-1)/100 + (y-1)/400);

        self.setDaysInMonth(y);

        for (m = 1; m <= 12; ++m) {
            t += self.getDaysInMonth(m);
            if (t >= d)
                break;
            t1 = t;
        }
        d -= t1;
        	    	    
	    switch(parm) {
	        case CLOCK_DATE_AMERICAN:
	            s = self.nameOfMonths[m] + ' ' + cvtstr(d) + ' ' + cvtstr(y);
	            break;
            case CLOCK_DATE_EUROPEAN:
                s = cvtstr(d) + ' ' + self.nameOfMonth[m] + ' ' + cvtstr(y);
                break; 
            case CLOCK_DATE_YMD:
                s = [y, m, d];
                break;
            default: 
                s = self.date;
	    }
	    return s;
	}
	setDate(...) = {
	    local i, b, y, m, d;
	   
	    if (argcount == 1) {
	        self.date = getarg(1);
	        return;
	    } else if (argcount == 3) {
	        y = getarg(1);
	        m = getarg(2);
	        d = getarg(3);
	    }
	    else return;
	    
        self.setDaysInMonth(y);

        b = (y-1)*365 + (y-1)/4 - (y-1)/100 + (y-1)/400;        
    
        for (i = 1; i <= m - 1; ++i)
            b += self.getDaysInMonth(i);
    
        b += d;
        
        self.date = b;
        
        /* synchronises the day with the date */
        self.setDay(b % 7);
    }
	/*
	 *	ELAPSETIME ATTRIBUTES AND METHODS
	 *
	 *	This represents the total accumulated game time
	 *	in minutes. It is updated each time Clock.advance()
	 *	is called.
	 */
	elapseTime = 0
	getElapseTime(...) = {
		local parm, e, d, h, i = ' ', len;
		
		if (argcount > 0) parm = getarg(1);
		
		e = self.elapseTime;
		d = ( e / 1440 );
		e -= d*1440;
		h = ( e / 60 );
		e -= h*60;
		if ( d > 0 ) {
			i = cvtstr( d );
			if ( d == 1 )
			   i = i + ' day ';
			else
			   i = i + ' days ';
		}
		if ( h > 0 ) {
			if ( d > 0 ) {
				i = i + ' and ';
			}
			i = i + cvtstr( h );
			if ( h == 1 )
			   i = i + ' hour ';
			else
			   i = i + ' hours ';
		}
		if ( d > 0 || h > 0 ) {
			if ( e > 0 ) {
				i = i + ' and ';
				i = i + cvtstr( e );
				if ( e == 1 )
		   	   	   i = i + ' minute';
				else i = i + ' minutes';
			}
		} else {
			i = i + cvtstr( e );
			if ( e == 1 )
		  	   i = i + ' minute';
			else i = i + ' minutes';
		}

		len = length( i );

		if ( substr( i, len, 1 ) == ' ' ) {
			i = substr( i, 1, len-1 );
		}

		if (parm == CLOCK_ELAPSETIME_MSG)
		    return i;
	    else return self.elapseTime;
	}
	setElapseTime(e) = {
	    self.elapseTime = e;
	}
	/*
	 *	WAITING PROCESS ATTRIBUTES
	 *
	 *	Used by the waiting process initiated by either the 
	 *	waitingVerb class or the advanctime(). 
	 */
	timeString = nil					// last value processed by parseTime()
	waiting = nil		// waiting process active (true/nil)
	waitTime = nil		// time in minutes to wait
	waitQuery = nil		// ask player if they wish to continue waiting
	waitQblock = nil		// block any stopWaiting()
	setTimeString(str) = {
	    self.timeString = str;
	}
	getTimeString(...) = {
	    return self.timeString;
	}
	/*
	 *  wait
	 *
	 *  This is a stub method that can be used by a Clock class object
	 *  for handling a "waiting" situation. Clock objects may wish to 
	 *  implement this method in conjunction with a command.
	 */
	wait = {}
    /* 
     *	stopWaiting( parm )
     *
     *	This method allows you to halt the waiting process, and
     *	informing the player of some timely event. The parameter
     *	should be either true or nil. True will request that the waitQuery
     *	question 'Do you wish to continue waiting? [Y/N]' be issued, 
     *	giving the player a choice. Passing nil will tell the waiting process
     *	to terminate without prompting the player for a response.
     */
    stopWaiting( parm ) = {
    	self.waiting = nil;
    	self.waitQuery = parm;
    }
    /*
     *  Sets the waitQblock attribute to parm.
     */
    setWaitQblock(parm) = {
        self.waitQblock = parm;
    }
    /*
     *	parseTime( cmd )
     *
     *	This method parses cmd to determine a Clock.waitTime. If
     *  successful, it returns a list consisting of 3 elements:
     *      ret[1]: a sublist containing the original reSearch() return.
     *      ret[2]: the calculated waitTime
     *      ret[3]: for o'clock-style formats this is the calculated time in 
     *			minutes past midnight; for hours/minutes format it contains
     *          true if 24-hour restriction was enforced, nil otherwise.
     *  If unsuccessful the function returns nil.
     */
    parseTime(cmd) = {
    	local ret, grp, specRet, restr24;
    	local tot = 0, hh = 0, mm = 0, wTime = 0;
    
    	cmd = lower(cmd);
	
    	/*	
    	 *  FORMAT: MIDNIGHT/NOON
    	 *
    	 *  Convert midnight to 12:00 a.m. and noon to 12:00 p.m. 
    	 */
    	ret = reSearch('(midnight)|(noon)', cmd );
        if (ret) {
        	specRet = ret;
        	grp = reGetGroup(1);
            if (reGetGroup(1)[3] == 'midnight')
            	cmd = '12:00 a.m';
            else if (reGetGroup(2)[3] == 'noon')
                cmd = '12:00 p.m';
        }    
    
        /*
         *  FORMAT: HH:MM <AM/PM> 
         *
         *  Our ultimate goal is to determine how many minutes we need
         *  to advance from the current time (Clock.time) to reach the 
         *  desired o'clock passed in the parameter. 
         */
    	ret = reSearch('([0-9]?[0-9]):([0-9][0-9]) *([ap]m)?', 
    	    cmd );
        if (ret) {
            local amOrPmInd;
        
            grp = reGetGroup(1);
            hh = cvtnum(grp[3]);
            grp = reGetGroup(2);
            mm = cvtnum(grp[3]);        
            grp = reGetGroup(3);        
        
            /*
             *  Validate the range for hours and minutes. This code 
             *  returns nil if the hours and minutes are not within
             *  the appropriate ranges either for 24-hour or 12-hour
             *  clock-time.
             */
            if (grp && grp[2]) {
                if (!(hh > 0 && hh < 13))
                    return nil;
                else
                    amOrPmInd = grp[3];
            } else {
                if (!(hh >= 0 && hh < 24))
                    return nil;
               else if (hh > 12) {
                        amOrPmInd = 'p.m.';
                        hh -= 12;
                    }
            }
            if (!(mm >= 0 && mm < 60))
            	return nil;
        
            tot = 60*hh + mm;        	
         
            /*
             *  AM/PM ADJUSTMENT.
             *
             *  At this point total represents an unadjusted total number
             *  of minutes past midnight. Adjustments need to be made for
             *  the presence or lack of an am/pm indicator.
             */
            switch( amOrPmInd ) {
                case 'am':
                case 'a.m':
                case 'a.m.':
                    /* adjust 12:00 a.m. - 12:59 a.m. */
                    if ( tot >= 720 )
                        tot -= 720;
                    break;
                
                case 'pm':
                case 'p.m':
                case 'p.m.':
                    /* Adjust time for p.m. */
                    if ( tot < 720 )
                        tot += 720;
                    break;

       			/*
    	         * 	This handles the cases where "wait until 4:00" is requested
    		 	 *	and it's 2:00 pm. In this case we probably really want to 
    		 	 * 	"wait until 4:00 pm", not "wait until 4:00 am" which would 
    		 	 * 	happen without this adjustment.
    		 	 */
                default:
    			    /* 
    			     *  Adjusts for tot between 12:00 pm - 12:59 pm. This
    			     *  converts HH:MM to HH:MM A.M.
    			     */
    			    if (self.time == 720 && tot >= 720)		
    				    tot -=720;
				
    				/* 
    				 *  Adjusts for when it's afternoon and we've asked
    				 *  for a time later in the afternoon. This converts
    				 *  HH:MM to HH.MM P.M.
    				 */
    			    if (self.time > 720 
    			    && (tot > self.time - 720))
    				    tot += 720;
				    
    				/* 
    				 *  Adjusts for when its morning and we've asked
    				 *  for a time in the afternoon. This converts 
    				 *  HH:MM to HH:MM P.M.
    				 */
    			    if (self.time < 720 
    			    && tot <= self.time)
    				    tot += 720;
    		}
		
    		/*
    		 *  COMPUTE WAITTIME
    		 *
    		 *  At this stage it is assumed that we have derived a point
    		 *  along a 24-hour clock. It remains only to determine how
    		 *  many minutes we need to cycle forward from our current
    		 *  time (Clock.time) in order to reach the desired time.
    		 */
    		wTime = tot;
		
    		if (wTime > self.time ) {
    			wTime -= self.time;
    		} else if ( wTime < self.time) {   
    			wTime += ( 1440 - self.time);
    		} else {
    			wTime = 1440;
    		}
        		
    		/* handles parsing of 'midnight/noon' */
            if (specRet)
            	return [specRet wTime tot];
            else
            	return [ret wTime tot];
        }
    
        /*
         *	EDIT: If we have gotten this far, anything with a ':' is not a valid
         *	time-format.
         */
        ret = reSearch(':', cmd);
        if (ret) return nil;

        /*
         *  FORMAT: HH AM/PM
         *
         *  Restructure the format to HH:MM am/pm. This code recurses into the 
         *	parseTime() method, after formatting parm in HH:MM AM/PM. 
         */
        ret = reSearch('([0-9]?[0-9]) *([ap]%.?m%.?)', cmd);
        if (ret) {
            specRet = ret;
            cmd = reGetGroup(1)[3] + ':00 ' + reGetGroup(2)[3];
            ret = self.parseTime( cmd );
            if (ret) {
            	wTime = ret[2];
            	tot = ret[3];
            	return [specRet wTime tot];
            }
        }
    
        /*
         *  FORMAT HH HOUR(S) (AND) MM MINUTE(S)
         *
         *  Calculate the number of minutes to advance for the given
         *  format. 
         */
    	ret = reSearch('([0-9]*[0-9]) *hours? *(and)? *([0-9]*[0-9]) *minutes?', 
    	    cmd );
        if (ret) {
            grp = reGetGroup(1);
            hh = cvtnum(grp[3]);
            grp = reGetGroup(3);
            mm = cvtnum(grp[3]);
        
            if (hh > CLOCK_TIME_LIM_HH || mm > CLOCK_TIME_LIM_MM)
            	return nil;
        	
            wTime = 60*hh + mm;
        
            return [ret wTime];
        }
    
        /*
         *  FORMAT HH HOUR(S)
         *
         *  Calculate the number of minutes to advance for the given
         *  format.
         */
    	ret = reSearch('([0-9]*[0-9]) *hours?', cmd );
        if (ret) {
            grp = reGetGroup(1);
            hh = cvtnum(grp[3]);
        
            if (hh > CLOCK_TIME_LIM_HH)
            	return nil;
        
            wTime = 60*hh + mm;
            return [ret wTime];
        }
    
        /*
         *  FORMAT MM MINUTE(S)
         *
         *  Calculate the number of minutes to advance for the given 
         *  format.
         */
    	ret = reSearch('([0-9]*[0-9]) *minutes?', cmd );
        if (ret) {
            grp = reGetGroup(1);
            mm = cvtnum(grp[3]);
        
            if (mm > CLOCK_TIME_LIM_MM)
            	return nil;
        
            wTime = 60*hh + mm;
            return [ret wTime];
        }

        /*
         *  FORMAT: MM
         *
         *  Restructure the format to MM minutes. This code recurses into the 
         *	parseTime() method, after formatting parm in MM minutes. 
         */
    	ret = reSearch('%<([0-9]*[0-9])%>', cmd );
        if (ret) {
            specRet = ret;
            cmd = reGetGroup(1)[3] + ' minutes';
            ret = self.parseTime( cmd );
            if (ret)
            {
            	wTime = ret[2];
            	return [specRet wTime];
            }
        }
       	return nil;
    }
    /*
	 *	ADVANCE METHOD
	 *
	 *	This progresses the time, day, date, and elapsetime
	 *	and makes a call to timepasses()
 	 */
	advance( incr ) = {
		local b, d, e, t, r;

		t = self.getTime;
		r = self.getTimeRate;
		if ( self.day )
			d = self.getDay;
		else d = 0;

		if ( self.date )
			b = self.getDate;
		else b = 0;
		
		e = self.getElapseTime;

		if ( incr == nil ) {
			t += r;
			e += r;
		} else {
			t += incr;
			e += incr;
		}

		while( t >= 1440 ) { 
			t -= 1440;
			d++;
			if ( d > 6 )
				d = 0;
			b++;
		}
		
		self.setTime(t);

		if ( self.getDay != nil ) 
			self.setDay(d);

		if ( self.getDate != nil )
			self.setDate(b);
		
		self.setElapseTime(e);

		self.events;
	}
    /*
	 *	EVENTS METHOD
	 *
	 *	Allows the author to affect global time-related events, such as lighting and 
	 *	weather effects. This method is called with each increment of Clock.time.
	 *	You should use replace this method in your game source using the 'modify'
	 *	statement.
	 */
	events = {
		return nil;
	}	                                             
    getClock(...) = {
        local i, parm, s = '';
        
        for (i = 1; i <= argcount; ++i) {
            parm = getarg(i);
            
            switch(parm) {
                case CLOCK_TIME:
                case CLOCK_TIME_HH:
                case CLOCK_TIME_HH24:
                case CLOCK_TIME_MM:
                case CLOCK_TIME_HHMM:
                case CLOCK_TIME_HH24MM:
                case CLOCK_TIME_HHMMTT:
                case CLOCK_TIME_AMPM:
                    s += self.getTime(parm);
                    break;
                case CLOCK_DAY:
                case CLOCK_DAY_ABBR:
                case CLOCK_DAY_FULL:
                    s += self.getDay(parm);
                    if (i <= argcount) s += ',';
                    break;
                case CLOCK_DATE:
                case CLOCK_DATE_AMERICAN:
                case CLOCK_DATE_EUROPEAN:
                case CLOCK_DATE_YMD:
                    s += self.getDate(parm);
                    break;
                case CLOCK_ELAPSETIME:
                case CLOCK_ELAPSETIME_MSG:
                    s += self.getElapseTime(parm);
            }
            if (i < argcount) s += ' ';
        }
        return s;
    }
	/*------------------------------------------------------------------
	 *	SET Clock
	 *
	 *  This method is used to set both the time and date for Clock.
	 *  Use this method as a means of synchronising both date, day, time
	 *  and time rate.
	 */
    setClock( t, ...) = {
    	local yyyy, ddd, r, y, m, d;
	
	    if (argcount > 1) r = getarg(2);
    	if (argcount > 2) y = getarg(3);
    	if (argcount > 3) m = getarg(4);
    	if (argcount > 4) d = getarg(5);

    	if ( r == nil ) r = 1;
    	if ( y == nil ) y = 1999;
    	if ( m == nil ) m = 1;
    	if ( d == nil ) d = 1;

    	self.setTime(t);
    	self.setTimeRate(r);
    	self.setDate(y, m, d);				// sets the display values for the statusLine without 								// advancing the time.
    }
    /*
     *	advClock( incr, style )
     *	
     *	A method to advance the time through your game code
     *	and providing the option to execute the rundaemons() and
     *	runfuses() or bypass them. This function should be the last 
     *	line of code executed in a block because it will issue an 
     *	abort upon completion regardless of the style chosen.
     *
     *	If style is true then the waitingVerb.action() method is used 
     *	to advance time. This means that daemons and fuses are run, and
     *	the advance is subject to any issued stopWaiting(). 
     *
     *	if style is nil then the Clock.advance() method is called 
     *	directly. No daemons or fuses will be run and no opportunity
     *	for stopWaiting will be available. This method, however, is 
     *	much quicker than waitingVerb.action().
     */
    advClock( incr, style ) = {
    	local t;
    	if ( self.waiting ) {
    		return nil;
    	}
    	if (datatype(incr) == DTY_SSTRING) {
    		local ret = self.parseTime( incr );
	
    		if (ret)
    			t = ret[2];
    		else
    			t = 0;
    	} else t = incr;
		
    	self.waitQblock = true;	// block waitQuery

    	if ( style ) {
    		self.waitTime = t;
    		self.wait2( nil );
    	} else {
    		self.advance( t );
    		self.waitQblock = nil;
    		abort;
    	}
    }
;

/*
 * 	gameClock: Clock
 *
 *	This is the game 'time clock' which keeps track of various
 *	date and time values, advances the time when requested
 *	and converts numeric date and time values into parm values
 *	for statusLine display.
 */
gameClock: Clock
	wait = {
		if (self.waitTime == 0) {
		    local ret = self.parseTime(self.getTimeString);
		    
		    self.waitTime = ret[2];
		}
		self.wait2(true);
		self.waitTime = 0;
		abort;
	}
	wait2(parm) = {
		local t, i;
		
		t = self.waitTime;
		if ( parm ) {
			if (self.waitTime)
				"Time passes...";
			else
				"Time doesn't pass...";
		}
		i = self.getTimeRate;
		self.setTimeRate(1);
		self.waiting = true;
		while ( t > 0 ) {
			t--;
			rundaemons();
			runfuses();
			if (!self.waitQblock) {
				if (t > 0 && self.waitQuery){
					self.waitQuery = nil;
					"<P>Do you want to continue waiting? <Y/N> ";
					if ( yorn() ) { 
						self.waiting = true;
					} else {
						break;
					}
				}
				if (!self.waiting) break;
			}
 		}
		self.waiting = nil;
		self.waitQuery = nil;
		self.waitQblock = nil;
		self.setTimeRate(i);	    
	}
;

/*
 *	gameClockPPO: preparseObj
 *
 *	Preparses the wait command. This will return a list consisting of the 
 *	reSearch() return for the command, a calculated waitTime, and a field that
 *	contains the time-value for the o'clock, or a boolean which indicates
 *	whether 24-hour restriction was necessary.
 *
 *  If ppo.t is not #include'd in the source then gameClock reverts to the
 *  use of an ordinary TADS preparse() function.
 */
gameClockPPO: PreparseObj
    ppoMethod( cmd ) = {
        local ret, grp, waitRet;
        
        cmd = lower(cmd);
        cmd = self.trim(cmd);
        
        waitRet = reSearch('(.*)(%<wait%>)(.*)',cmd);
        if (waitRet == nil) {
        	waitRet = reSearch('(.*)(%<z%>)(.*)',cmd);
        	if (waitRet == nil)
        		return true;
        }
        
       	/*
       	 *	If wait has no specified time we use the default gameClock 
       	 *  timerate.
       	 */
       	if (reGetGroup(3)[3] == '' || reGetGroup(3)[3] == ' '
       	|| reGetGroup(3)[3] == '.') {
    		cmd = reGetGroup(1)[3] + reGetGroup(2)[3] + 
    		    ' ' + cvtstr(gameClock.timeRate) + reGetGroup(3)[3];
    	}
        ret = gameClock.parseTime( cmd );
        if (ret) {
            local tmpcmd = '';
            
            gameClock.waitTime = ret[2];
            
            /*
             *	Restructuring the command. The reSearch() return from parsetime is
             *	the first element of ret. We can discard the rest of the list.
             *	We save ret[3], which is are returned parsed time parm, for
             *	use with the 'Again' command.
             */
            ret = car(ret);
            
            gameClock.setTimeString(ret[3]);
            
            if (ret[1] - 1)
                tmpcmd += substr(cmd, 1, ret[1]-1);
                
            if (length(cmd) - ret[1] - ret[2] + 1) {	
             	local tmpcmd2;
             	
             	tmpcmd2 = substr(cmd, ret[1] + ret[2], 
                   length(cmd) - ret[1] - ret[2] + 1);

        		ret = reSearch('%w', tmpcmd2);
        		
        		if (ret) {
        			"[There appear to be extra words or commands after the
        			WAIT command.]\b";
        			return nil;
        		} else {
        			tmpcmd += tmpcmd2;
        		}
        	}
            return tmpcmd;
        } else {
        	if (waitRet) {
        			"[This does not appear to be a valid time-format.]\b";
        			return nil;
        	}
        }
        return true;
    }
;

#pragma C-

#endif /* _CLOCK_H_ */
