////////////////////////////////////////////////////////////////////////
//  
//  ALT Library File: Vector 010123
//
//  Copyright (c) 2000, 2001 Kevin Forchione. All rights reserved.
//  Based on ADV.T (c) and STD.T (c) Michael Roberts.
//
//  This file is part of the ALT replacement library for ADV.T and 
//  STD.T and requires TADS 2.5.1 or later.
//
////////////////////////////////////////////////////////////////////////

#ifndef _VECTOR_H_ 
#define _VECTOR_H_

#pragma C+

/*
 *  Vector: object
 *
 *  The Vector class can be used to manipulate lists.
 */
class Vector: object
    list = []
    /*
     *  Add an element to the tail of the list.
     */
    addElement(item) = {
        self.list += item;
        return self.list;
    }
    /*
     *  Add an element to the head of the list.
     */
    addElementToHead(item) = {
        local newList = [];
        
        newList += item;
        newList += self.list;
        self.list = newList;
        return self.list;
    }
    /*
     *  Add item to the list at pos n. If n > list.getSize the list
     *  is padded with nil values to a length of n-1 before adding item.
     */
    insertElementAt(item, n) = {
        local newList = [], l1 = [], l2 = [];
        
        if (n > self.getSize) self.padVector(n-1);
        
        newList += self.subList(1, n-1);
        newList += item;
        newList += self.subList(n, self.getSize);
        self.list = newList;
        return self.list;
    }
    /*
     *  Replace element at position n with item.
     */
    putElementAt(item, n) = {
        self.list[n] = item;
        return self.list;
    }
    /*
     *  Pad the list with nil elements to the length of n.
     */
    padVector(n) = {
        local i, len;
        
        len = self.getSize + 1;
        for (i = len; i <= n; ++i)
            self.list += nil;
        return self.list;
    }
    /*
     *  Left pad the list with nil elements to the length of n.
     */
    lpadVector(n) = {
        local i, len, newList = [];
       
        len = n - self.getSize;
        for (i = 1; i <= len; ++i)
            newList += nil;
            
        newList += self.list;
        self.list = newList;
        return self.list;
    }
    /*
     *  Remove the element at pos n in the list.
     */
    removeElementAt(n) = {
        local i, len, newList = [];
        
        len = self.getSize;
        for (i = 1; i <= len; ++i) {
            if (i != n)
                newList += self.list[i];
        }
        self.list = newList;
        return self.list;
    }
    /*
     *  Remove the first occurrence (from left to right) of item from
     *  the list.
     */
    removeFirstOccurrence(item) = {
        local n = self.posFirstOccurrence(item);
        
        if (0 < n && n <= self.getSize)
            self.removeElementAt(n);
        return self.list;
    }
    /*
     *  Remove the last occurrence (from left to right) of item from
     *  the list.
     */
    removeLastOccurrence(item) = {
        local n = self.posLastOccurrence(item);
        
        if (0 < n && n <= self.getSize)
            self.removeElementAt(n);
        return self.list;
    }
    /*
     *  Remove all occurrences of item from the list.
     */
    removeAllOccurrence(item) = {
        self.list -= [item];
        return self.list;
    }
    elementAt(n) = {
        if (0 < n && n <= self.getSize) {
            self.count = n;
            return self.list[n];
        } else {
            self.count = 0;
            return nil;
        }
    }
    /*
     *  Return first element in the vector. This method sets count to 1
     *  for use by nextElement().
     */
    firstElement = {
        self.count = 1;
        return self.elementAt(self.count);
    }
    /*
     *  Return the next element (after using elementAt() or
     *  firstElement().
     */
    nextElement = {
        ++self.count;
        return self.elementAt(self.count);
    }
    /*
     *  Return last element in the vector. This method returns nil if
     *  the vector is empty.
     */
    lastElement = {
        self.count = self.getSize;
        return self.elementAt(self.count);
    }
    contains(item) = {
        if (find(self.list, item))
            return true;
        else return nil;
    }
    /*
     *  Return the index n for the first occurrence (from left to right) of 
     *  item in the list starting from position f.
     */
    firstIndexOf(item, f) = {
        local i;
        self.count = f;
        for (i = f; i <= length(self.list); ++i) {
            if (self.list[i] == item)
                return i;
        }
        return -1;
    }
    nextIndexOf(item) = {
        self.firstIndexOf(item, ++self.count);
    }
    /*
     *  Return the index n for the last occurrence (from left to right) of
     *  item in the vector starting from position f.
     */
    lastIndexOf(item, f) = {
        local i;  
        self.count = f;
        for (i = length(self.list); i >= f; --i) {
            if (self.list[i] == item)
                return i;
        }
        return -1;
    }
    /*
     *  Return indexes in the vector for
     *  all occurrences of item starting from pos f.
     */
    allIndexOf(item, f) = {
        local i, len, newList = [];
        
        len = length(self.list);
        for (i = f; i <= len; ++i) {
            if (self.list[i] == item)
                newList += [i];
        }
        return newList;
    }
    /*
     *  Return a count of all occurrences of item in the list.
     */
    countOccurrenceOf(item, f) = {
        local n = self.allIndexOf(item, f);
        
        return length(n);
    }
    /*
     *  Return a subList of list, starting from pos for a length len
     *  (if provided). if len is not provided then it is the length of 
     *  the remainder of the list. If len is greater than the length 
     *  of the list then the subList contains the remainder of the list.
     *  If pos is greater than the length of the list then an empty list
     *  is returned.
     */
    subList(pos, ...) = {
        local i, len, newList = [];
        
        if (argcount > 1) len = getarg(2);
        if (len == nil) len = self.getSize - pos + 1;
        
        for (i = pos; i <= pos + len - 1; ++i) {
            if (self.getSize >= i)
                newList += self.list[i];
            else break;
        }
        return newList;
    }
    /*
     *  Returns the length of the list.
     */
    getSize = {return length(self.list);}
    /*
     *  Returns a boolean indicating whether the list is empty or not.
     */
    isEmpty = {
        if (self.getSize)
            return nil;
        else return true;
    }
    construct = {}
    destruct = {}
;  

#pragma C-

#endif  /* _VECTOR_H_ */
