#charset "us-ascii"
#include <adv3.h>

/*

Breadth-First Path Finder

Steve Breslin, 2004
steve.breslin@gmail.com

This module provides a class that implements a basic breadth-first
algorithm for finding the shortest path between two nodes in an
unweigheted graph. The basic BreadthFirst class makes no assumptions
about the nature of the underlying graph, except that it is unweighted;
subclasses must provide the concrete details about the graph being
traversed.

Everyone is granted permission to use and modify this file for any
purpose, provided that the original author is credited.

*/

class BreadthFirst: object

    /* Find the best path from 'fromNode' to 'toNode', which are nodes in
     * the underlying graph.  We'll return a vector consisting of graph
     * nodes, in order, giving the shortest path between the nodes.  Note
     * that 'fromNode' and 'toNode' are included in the returned vector.
     *
     * If the two nodes 'fromNode' and 'toNode' aren't connected in the
     * graph, we'll simply return nil.  
     */
    findPath(fromNode, toNode) {
        
        /* nodeVector is a vector whose elements alternate between nodes
         * and the indexes of their parent nodes. (An
         * even-numbered-indexed element is the index of the previous
         * element's parent.)
         *
         * We populate the vector with the fromNode, which has no
         * parent node.
         */
        local nodeVector = new Vector(32, [fromNode, nil]);

        /* until we have found the toNode, we have not succeeded in
         * finding the path to the node.
         */
        local success = nil;

        /* We wrap the main search in a try, so we can exit immediately
         * once we have found the path.
         */
        try {
            for(local i = 1 ; i < nodeVector.length() ; i+=2)  {

                /* add each adjacent item to the nodeVector */
                forEachAdjacent(nodeVector[i], new function(adj, dist) {

                    /* add the adjacent node and its parent's index
                     * only if it's not already in the node vector.
                     */
                    if(!nodeVector.indexOf(adj)) {
                        nodeVector.append(adj);
                        nodeVector.append(i);

                        /* If we just added the toNode, we're done */
                        if(adj == toNode) {
                            success = true;
                            exit;
                        }
                    }
                });
            }
        }
        finally {
            if(success)
                return calcPath(nodeVector);
            return nil;
        }
    }

    calcPath(nodeVector) {

        /* curNode is the current node, as we build the path backwards
         * from the "toNode" to the "fromNode." Initially, it is the
         * next-last element in the vector.
         */
        local curNode = nodeVector[nodeVector.length() -1];

        /* pIdx is the index of the current node's parent. Initially,
         * it is the last element in the vector.
         */
        local pIdx = nodeVector[nodeVector.length()];

        /* ret is the path vector we'll return when we're finished
         * extracting the path from the nodeVector.
         */
        local ret = new Vector(8, [curNode]);

        while(pIdx) {
            curNode = nodeVector[pIdx];
            ret.prepend(curNode);
            pIdx = nodeVector[pIdx + 1];
        }

        /* that's it - return the path */
        return ret;
    }

    /* The edges aren't weighted, so we're just counting the number of
     * nodes in the shortest path (minus one).
     */
    findDistance(fromNode, toNode) {
        return (findPath(fromNode, toNode).length() - 1);
    }

    /* 
     *   Iterate over everything adjacent to the given node in the
     *   underlying graph.  This routine must simply invoke the given
     *   callback function once for each graph node adjacent to the given
     *   graph node.
     *   
     *   The callback takes two arguments: the adjacent node, and the
     *   distance from 'node' to the adjacent node.  Note that the distance
     *   must be a positive number, as Dijkstra's algorithm depends on
     *   positive distances.  If the graph isn't weighted by distance,
     *   simply use 1 for all distances.
     *   
     *   This method isn't implemented in the base class, since we don't
     *   make any assumptions about the underlying graph.  Subclasses must
     *   provide concrete implementations of this routine to define the
     *   underlying graph.  
     */
    forEachAdjacent(node, func) { /* subclasses must override */ }
;

