#charset "us-ascii"
#include <adv3.h>

/*

"Dijkstra" and "A*"

Steve Breslin, 2004
steve.breslin@gmail.com

Drawing from "Pathfinder," copyright Mike Roberts, 2003.

This module implements the Dijkstra (and optionally the A*) algorithms
for finding the shortest path between two nodes in a graph. The base
implementation makes no assumptions about the nature of the underlying
graph; subclasses must provide the concrete details about the graph
being traversed.

Credit to Mike Roberts for "Pathfinder," the second Tads-3
implementation of Dijkstra that I am aware of (Kevin Forchione having
written the first one in 2001, as a part of his "proteus" library
extension). Although our main pathfinder algorithms -- the "A*" and
"Dijkstra" algorithms proper -- are considerably different from (and
several times (!) faster than) Mike Roberts' implementation of Dijkstra,
our overall organization draws from his very useful work; some of the
commentary is also Mike's. (Thanks Mike!)

Everyone is granted permission to use and modify this file for any
purpose, provided that the original authors are credited.

*/

/*---------------------------------------------------------------------

What is A* (A-star)?

The A* algorithm works exactly like Dijkstra, except that a heuristic is
considered when choosing the next node to expand.

 -  The next node Dijkstra expands is the unexpanded node closest to the
    starting node, i.e., the one with the lowest "bestDist"

 -  The next node A* expands is the unexpanded node with the lowest
    "bestDist + h()", where h() (the "heuristic" function) returns the
    estimated distance from that node to the target node.

If the estimate h() makes is highly accurate, the search will be highly
efficient, visiting the minimal number of nodes from the start-node to
the target. (Although the time saved by the heuristic must offset the
computation required by the heuristic, if we are to save time overall.)

A* will find the shortest path if the heuristic underestimates or
exactly estimates the distance to the target node. If it can
overestimate the distance, it will not necessarily find the very
shortest path.

====

By default, we implement Dijkstra rather than A* by setting h() to 0. If
you want to implement A*, simply modify DijkstraNode.h() to return an
approximation of the distance from self.graphNode to
libGlobal.curToNode.

====

A note on A* pathfinding:

For pathfinding in most Tads-3 situations, a simple breadth-first search
is normally much better than either Dijkstra or A*.

If you are searching a graph of "apparent room connections" (which
changes frequently and substantially as the game progresses, so
preprocessing isn't directly useful), you might use Floyd to calculate
the actual distance between rooms (which is constant), and use that as a
heuristic for an A* search of "apparent room connections" -- the "actual
distance" is normally a fairly good heuristic for the "apparent
distance". This is probably the most common case where A* is really
useful.

Otherwise, implementing a truly useful heuristic for pathfinding in a
typical Tads-3 game map is non-trivial. One way would be to preprocess
and record the best distances between all points in a matrix (if the map
is small enough -- less than 100 rooms or so), but in this case it would
be just as easy to store routing information. (See our Tads-3
implementation of Floyd-Warshall.) Other ways would be a path labeling
system, or establishing an estimate by searching a metagraph
(representing regions of the map, broken up according to some graph
analysis or so).

*/

// for use by DijkstraNode.h(), in case we are doing an A* search:
modify libGlobal
    curToNode = nil
;

class Dijkstra: object

    /* Find the best path from 'fromNode' to 'toNode', which are nodes
     * in the underlying graph. We return a vector consisting of graph
     * nodes, in order, giving the shortest path between the nodes. Note
     * that 'fromNode' and 'toNode' are included in the returned vector.
     *
     * If the two nodes 'fromNode' and 'toNode' are not connected in the
     * graph, we will simply return nil.
     */
    findPath(fromNode, toNode) {

        /* Create a DijkstraNode representation of the first node in the
         * graph. A DijkstraNode encapsulates a few pieces of
         * information about its node in the graph. The arguments are
         * (<my node>, <my predecessor>, <my distance>). Here, 'nil'
         * represents that this node has no predecessor, and '0'
         * represents that it is at zero distance from the start.
         */
        local cur = new DijkstraNode(fromNode, nil, 0);

        /* openVector is a vector of nodes visited but not yet "closed".
         * (By the terminology of search algorithms, a node is called
         * "closed" once it has added all of its connected nodes to the
         * openVector.)
         *
         * Instead of using a simple vector, we make it a binary heap,
         * which automatically so orders itself that the element with
         * the smallest bestDist property is always first. If we are
         * using A*, the vector is instead sorted by totalDist (which is
         * the known best distance from the fromNode plus our heuristic
         * guess at the shortest distance to the goal).
         *
         * We are visiting the fromNode, so add cur to the open vector.
         */
        local openVector = new BinaryHeap(&totalDist, 32, [cur]);

        /* The nodeTable keeps track of all the nodes visited, whether
         * open or closed. The keys are actual nodes on the graph, and
         * the values are their DijkstraNode representations.
         *
         * We trade a bit of memory overhead for faster computation
         * time. It is more conventional to keep a vector of closed
         * nodes, but then looking up the nodes would be considerably
         * slower.
         */
        local nodeTable = new LookupTable();

        local ret; // the return vector

        libGlobal.curToNode = toNode; // for use by h()

        /* Remember that the fromNode is represented by cur. */
        nodeTable[fromNode] = cur;

        /* Keep going while we have open nodes... */
        while (openVector.length()) {

            /* Ask the binary heap to give us the node with the lowest
             * bestDist. This also removes the element from the vector.
             */
            cur = openVector.getFirst();

            /* Mark this node as "closed," as we are about to add its
             * connected nodes.
             */
            nodeTable[cur.graphNode].closed = true;

            /* If it is the *toNode* that has the shortest distance, we
             * have found the shortest path to the target node. We can
             * break out of this 'while' loop, and return the path we
             * have found.
             */
            if(cur.graphNode == toNode)
                break;

            /* add to the openVector everything adjacent to the node
             * we just closed, creating or updating their DijkstraNode
             * representations as necessary.
             */
            forEachAdjacent(cur.graphNode, new function(adj, dist) {

                local newDist;
                local dNode;

                /* Get the DijkstraNode that represents adj, if there
                 * is one.
                 *
                 * If the node is already closed, we do not need to
                 * process it further.
                 */
                if((dNode = nodeTable[adj]) && dNode.closed)
                    return;

                /* Calculate the new distance to the adjacent node, if
                 * we were to take a path from the node we just finished
                 * -- this is simply the path distance from fromNode to
                 * the just-closed node, plus the distance from that
                 * node to the adjacent node (i.e., 'dist').
                 */
                newDist = cur.bestDist + dist;

                /* If there is not yet a dNode for the adjacent node, we
                 * we create a new DijkstraNode to represent the adj
                 * node, initialize its predNode and bestDist values,
                 * append it to the openVector, and update the
                 * nodeTable.
                 */
                if(!dNode) {
                    dNode = new DijkstraNode(adj, cur, newDist);
                    openVector.append(dNode);
                    nodeTable[adj] = dNode;
                }

                /* Otherwise, if the newDist is better than the dNode's
                 * former bestDist, we have a new "best path"
                 * predecessor and a new bestDist for this node. Update
                 * these properties, and notify the binary heap that we
                 * have changed the bestDist value, so it can re-sort
                 * itself as necessary.
                 */
                else if (newDist < dNode.bestDist) {
                    dNode.predNode = cur;
                    dNode.bestDist = newDist;
                    openVector.update(dNode);
                }
            });
        }

        /* We have either 1) broken the while loop, having found the
         * shortest path to the toNode, or 2) exhausted the open vector
         * without finding the toNode.
         *
         * If the former, cur is the DijkstraNode representing toNode.
         * In this case, we build a path, below.
         *
         * If the latter, cur is something else, and we return nil.
         */
        if(cur.graphNode != toNode)
            return nil;

        /* We have found the shortest path. We can record the bestDist
         * for use with findDistance().
         */
        bestDist_ = cur.bestDist;

        /* We are going to generate a vector of nodes that will take us
         * from fromNode to toNode. First we initialize the return
         * vector.
         */
        ret = new Vector(8, [toNode]);

        /* The information we have is sort of in reverse: for simplicity
         * and efficiency, we have been keeping track of
         * "best-predecessor" or "parent" nodes, rather than keeping
         * track of children nodes. So we prepending the shortest-path
         * parent, grandparent, and so on, until we reach the fromNode.
         *
         * Note that the predecessor of the starting element is nil, so
         * we can simply keep going until we reach a nil predecessor.  
         */

        while (cur = cur.predNode) // yes this is correct.
            ret.prepend(cur.graphNode);

        /* that's it - return the path */
        return ret;
    }

    /* In case we want to know the distance of the shortest path: */
    findDistance(fromNode, toNode) {

        /* If findPath doesn't reset bestDist_, this is because a path
         * has not been found. We'll return nil in this case.
         */
        bestDist_ = nil;
        findPath(fromNode, toNode);
        return bestDist_;
    }
    bestDist_ = 0

    /* 
     *   Iterate over everything adjacent to the given node in the
     *   underlying graph.  This routine must simply invoke the given
     *   callback function once for each graph node adjacent to the given
     *   graph node.
     *   
     *   The callback takes two arguments: the adjacent node, and the
     *   distance from 'node' to the adjacent node.  Note that the distance
     *   must be a positive number, as Dijkstra's algorithm depends on
     *   positive distances.  If the graph isn't weighted by distance,
     *   simply use 1 for all distances.
     *   
     *   This method isn't implemented in the base class, since we don't
     *   make any assumptions about the underlying graph.  Subclasses must
     *   provide concrete implementations of this routine to define the
     *   underlying graph.  
     */
    forEachAdjacent(node, func) { /* subclasses must override */ }
;

/* A DijkstraNode encapsulates some information about a node in the
 * graph.
 *
 * Note that this is NOT a node in the underlying graph; rather, this is
 * an internal data structure that we use in the path finder to keep
 * track of the underlying nodes and their status.
 */
class DijkstraNode: object

    construct(node, pred, dist) {

        /* remember the underlying node, its predecessor, and the
         * current "best distance" from the starting node.
         */
        graphNode = node;
        predNode = pred;
        bestDist = dist;
    }
    
    /* the underlying node in the graph */
    graphNode = nil

    /* the best-path predecessor for this path element */
    predNode = nil

    /* The best estimate of the shortest distance from the starting
     * point. We use "nil" to represent infinity.
     */
    bestDist = nil

    /* The totalDist is the bestDist plus the heuristic h(). This is
     * the distance from the starting point (bestDist), plus the
     * estimated distance from self.graphNode to the target node,
     * available as libGlobal.curToNode.
     */
    totalDist() { return (bestDist + h()); }

    /* modify DijkstraNode to override this: by default we use the
     * Dijkstra algorithm, which means the h() value is always zero.
     */
    h() { return 0; }

    /* have I been closed yet? */
    closed = nil
;

