#charset "us-ascii"
#include <adv3.h>

/*

Tads-3 Floyd-Warshall

Steve Breslin, 2004
steve.breslin@gmail.com

This module implements the Floyd-Warshall algorithm, for finding the
shortest distances between all points in a graph, and optionally for
finding the shortest paths between all points. The base Floyd class
makes no assumptions about the nature of the underlying graph;
subclasses must provide the concrete details about the graph being
traversed.

Everyone is granted permission to use and modify this file for any
purpose, provided that the original author is credited.

*/

class Floyd: object

    /* You can optionally record shortest-paths in addition to
     * shortest-distance between node-pairs. If you only care about
     * shortest-distance, set pathFinding to nil.
     */
    pathFinding = true

    /* We calculate a matrix-like lookup table, with keys [fromNode,
     * toNode] mapped to traversal cost between the rooms, for each
     * node combination in the graph. (We call this the dTable for
     * "distance table.")
     *
     * If pathFinding is true, we calculate another table, the pTable
     * (for path table), which records (very efficiently) the shortest
     * path between all pairs of nodes.
     *
     * You'll speed things up by calling this at compile time, but you
     * can call this function at runtime whenever the graph changes.
     */
    execute() {
        local nodeCount = 0;
        if(!nodeVector) {
            nodeVector = new Vector(32);
            populateNodeVector();
        }
        nodeCount = nodeVector.length();
        dTable = new LookupTable(nodeCount, nodeCount*nodeCount);

        if(pathFinding)
            pTable = new LookupTable(nodeCount, nodeCount*nodeCount);

        /* First, we simply populate dTable with directly connected
         * nodes.
         *
         * In the next block, we will expand dTable for indirect
         * connections.
         */
        for(local i = 1 ; i <= nodeCount ; i++) {
            for(local j = 1 ; j <= nodeCount ; j++) {

                /* we check if nodeVector[i] is directly connected to
                 * nodeVector[j]
                 */
                local dist = nodesConnected(nodeVector[i],
                                            nodeVector[j]);
                if(dist)
                    dTable[[i, j]] = dist;

                /* If there's no direct connection between i and j, the
                 * distance is infinite; we use 'nil' to represent
                 * infinity.
                 */
                else
                    dTable[[i, j]] = nil;

                /* If we're pathfinding, we also simply populate pTable
                 */
                if (pathFinding)
                    pTable[[i, j]] = i;
            }

            /* nodes connect to themselves at distance 0 */
            dTable[[i, i]] = 0;
        }

        /* We expand dTable for indirect connections. */
        for (local k = 1; k <= nodeCount ; k++)
            for (local i = 1; i <= nodeCount ; i++)
                for (local j = 1; j <= nodeCount; j++)

                    if( (dTable[[i, k]] == nil
                         || dTable[[k, j]] == nil) )
                        continue;

                    else if ( (dTable[[i,j]] == nil)
                              || (dTable[[i, k]] + dTable[[k, j]]
                                  < dTable[[i, j]]) ) {

                        dTable[[i, j]] = dTable[[i, k]]
                                         + dTable[[k, j]];
                        if(pathFinding)
                            pTable[[i, j]] = pTable[[k, j]];
                    }
    } // execute()

    /* There follows two methods that cannot be implemented in the base
     * class, since we don't make any assumptions about the underlying
     * graph. Subclasses must provide concrete implementations of these
     * routines to define the underlying graph.
     */

    /* populateNodeVector must be defined by subclasses. It adds all
     * nodes in the graph to the nodeVector. See the roomPathFloyd
     * object in RmPthFnd.t for an example.
     */
    populateNodeVector() {  /* subclasses must override */  }

    /* nodesConnected must be defined by subclasses. It should return
     * the traversal cost (the distance) if the nodes are directly
     * connected, or nil if they are not directly connected. (Indirect
     * connection will be calculated by Floyd.) Again, please see
     * roomPathFloyd in RmPthFnd.t for an example.
     */
    nodesConnected(node1, node2) {  /* subclasses must override */  }

    /* a matrix-like lookup table for distances */
    dTable = nil

    /* a matrix-like lookup table for shortest paths */
    pTable = nil

    /* a vector of all the nodes in the graph. */
    nodeVector = nil

    /* findDistance simply returns the appropriate value from dTable.
     */
    findDistance(fromNode, toNode) {

        /* If we haven't constructed a dTable, construct one now. */
        if(!dTable) {
#ifdef DEBUG__
            "\bWARNING: findDistance() called when no path has
            been constructed. Constructing table....\b";
#endif
            execute(); // recalculate dTable
        }

        /* return the appropriate distance value from the dTable */
        return dTable[[nodeVector.indexOf(fromNode),
                       nodeVector.indexOf(toNode)]];
    }

    /* findPath returns a vector of the shortest path, beginning with
     * fromNode and ending with toNode.
     *
     * On a shortest path from i to j, pTable[[i, j]] is the last node
     * before j. So, the path is: i -> ... -> pTable[[i, j]] -> j. We
     * recurse backwards through this to construct the shortest path.
     */
    findPath(fromNode, toNode) {

        local dist = findDistance(fromNode, toNode);
        local ret = new Vector((dist?dist:8));
        local i = nodeVector.indexOf(fromNode);
        local j = nodeVector.indexOf(toNode);

        if(dist == nil) { return []; }

        /* If we haven't constructed a pTable, construct one now. */
        if(!pTable) {
#ifdef DEBUG__
            "\bWARNING: findPath() called when no path has been
            constructed. Constructing table....\b";
#endif
            pathFinding = true; // set pathFinding to true
            execute(); // recalculate dTable and pTable
            dist = findDistance(fromNode, toNode); // reset dist
            i = nodeVector.indexOf(fromNode); // reset i
            j = nodeVector.indexOf(toNode); // reset j
        }

        /* If the distance is "infinity," there's no path. */
        if(dist == nil)
            return nil;

        while(i != j) {
            ret.prepend(nodeVector[ pTable[[i, j]] ]);
            j = nodeVector.indexOf(nodeVector[ pTable[[i, j]] ]);
        }
        ret += toNode;
        return ret;
    }
;
