#charset "us-ascii"

/*
 *   TADS 3 Action Report Combiner. A sample file illustrating how this
 *   extension can be used. This source file cannot be compiled on its own.
 *   The best way to test it is simply to add it to the sample game that
 *   accompanies the TADS 3 distribution, or to one of your own projects
 *   (and along with the extension itself, of course), and then compile
 *   that.
 *
 *   In the TADS 3 sample game, you can see how things work by typing the
 *   following command at the start of the game:
 *
 *   > BOB, PUT ALL THE BALLS IN THE BROWN BOX
 *
 *   The entire action (eight separate reports) should be summarized in one
 *   sentence.
 */

#include <adv3.h>
#include <en_us.h>

/* ---------------------------------------------------------------------- */
/*
 *   Define a rule class for combining the reports of first taking an
 *   object, then putting it somewhere else. We make this a class because we
 *   want to have separate rules for putting in, putting on, etc.
 *
 *   We don't have the benefit of having special report classes for the
 *   reports that we want to combine here, I.E. there's no OkayTakeReport
 *   class, so we have to look at the message properties and then figure out
 *   the involved objects based on the actions that generated the reports.
 */
class CombineTakePutRule: CombineReportsRule
    /*
     *   Our report sequence pattern. We match an okayTakeMsg or an implicit
     *   take action followed by a putting message which our instances must
     *   define.
     */
    pattern =
    [
        { x: x.messageProp_ == &okayTakeMsg
             || (x.ofKind(ImplicitActionAnnouncement)
                 && x.action_.actionOfKind(TakeAction))
        },
        { x: x.messageProp_ == putInProp
        }
    ]

    /*
     *   Combine a matching sequence of reports. We have two pattern items,
     *   one for the take report and one for the put report, so we define
     *   two parameters for these.
     */
    combineReports(take, put)
    {
        /* check that the reports come from the same actor */
        if (take.action_.actor_ != put.action_.actor_)
            return nil;

        /*
         *   Check that the reports concern the same object. In the case of
         *   the put action, there may be several direct objects, so check
         *   if the one from the take action is among them.
         */
        if (put.action_.dobjList_.indexWhich(
                { x: x.obj_ == take.action_.dobjCur_ }) == nil)
            return nil;

        /* extract the actor, the object and the new container */
        local who = take.action_.actor_;
        local obj = take.action_.dobjCur_;
        local cont = put.action_.iobjCur_;

        /*
         *   Create and return a new report of a kind defined below. Supply
         *   the actor, the object, the new container and the preposition
         *   ('in' or 'on', as defined by our instances).
         */
        return new TakePutReport(who, obj, cont, objInPrep);
    }

    /*
     *   our specific message property and preposition for the PutXxx action
     */
    putInProp = nil
    objInPrep = ''    
;

/*
 *   Define rules for combining a take followed by putting in or putting on,
 *   respectively. Rules for under and behind could be added in the same
 *   fashion.
 */
combineTakePutInRule: CombineTakePutRule
    putInProp = &okayPutInMsg
    objInPrep = 'in'
;

combineTakePutOnRule: CombineTakePutRule
    putInProp = &okayPutOnMsg
    objInPrep = 'on'
;

/*
 *   Define a new type of report for a combined take and put. By using the
 *   CustomReport class, we make it easier to match and combine these
 *   reports in other rules. (An example of this will follow.)
 */
class TakePutReport: CustomReport
    /*
     *   Get our message. The arguments are: the actor, the object which is
     *   being moved, the new container and the preposition. These will be
     *   stored automatically by the report.
     */
    getMessage(who, obj, cont, prep)
    {
        /* set up the necessary message parameters for our message */
        gMessageParams(who, obj, cont);

        /* build a message string and return it */
        return '{The who/he} {take[s]|took} {the obj/him} and put{[s who]|}
            {it obj/him} ' + prep + ' {the cont/him}. ';
    }
;

/* ---------------------------------------------------------------------- */
/*
 *   Define a rule for combining several consecutive TakePutReports of the
 *   same kind (same actor, same destination, same preposition) into a
 *   single report. Since this is a CombineCustomReportsRule, we use a
 *   different pattern format and the special combineCustomReports() method.
 */
CombineCustomReportsRule
    /*
     *   Our report sequence pattern. We match one or more TakePutReports of
     *   the same kind. Only the objects being moved are allowed to differ.
     *   The string-in-a-list syntax ensures that all of these are
     *   remembered individually.
     */
    pattern =
    [
        [TakePutReport, 'who', ['objs'], 'cont', 'prep'], multi
    ]

    /*
     *   Combine a matching sequence of custom reports. We have used four
     *   strings in our pattern, so we'll be passed four arguments. The
     *   second one will be a list.
     */
    combineCustomReports(who, objs, cont, prep)
    {
        /*
         *   Return nil if we only matched one report. Our purpose is to
         *   combine several reports into one.
         */
        if (objs.length == 1)
            return nil;

        /* set up the necessary message parameters */
        gMessageParams(who, cont);

        /*
         *   Build the message string. This will have to be done in a few
         *   separate steps, so we store it in a local variable and add to
         *   it as we go. Start with the prefix for the taking part.
         */
        local str = '{The who/he} {take[s]|took} ';

        /*
         *   Add the object list, which we generate using a custom lister.
         *   Since we need the resulting string rather than a print-out, we
         *   use output capturing while invoking the lister.
         */
        str += mainOutputStream.captureOutput(
                   {: actionItemLister.showListAll(objs, 0, 0) });

        /* add the putting part */
        str += ' and put{[s who]|} them ' + prep + ' {the cont/him}. ';

        /* return the message string */
        return str;
    }

    /*
     *   Since the reports we combine here are generated by the rules we
     *   defined earlier, we have to make sure that these are applied before
     *   this one. We can do this through the exexBeforeMe property.
     */
    execBeforeMe = [combineTakePutInRule, combineTakePutOnRule]
;

/*
 *   A custom lister for listing objects involved in a combined report for
 *   multiple take + put actions. This lister should be usable for other
 *   such lists of direct or indirect objects without modification.
 */
actionItemLister: Lister
    /*
     *   When listing an object, print its name in objective form with a
     *   definite article. This corresponds to a substitution string such as
     *   {the dobj/him}, which is normally what we use in action reports.
     */
    showListItem(obj, options, pov, infoTab)
    {
        say (obj.theNameObj);
    }

    /* for counted lists, we print a string such as "the five gold coins" */
    showListItemCounted(lst, options, pov, infoTab)
    {
        /*
         *   Arbitrarily choose the first item in the list and let it
         *   generate a counted name. Then all we have to do is to add the
         *   definite article.
         */
        "the <<lst[1].countName(lst.length)>>";
    }

    /*
     *   All objects are to be included in this kind of list. It doesn't
     *   matter if they are usually omitted from contents lists, since this
     *   is a different kind of list.
     */
    isListed(obj) { return true; }
;

