#-------------------------------------------------------------------------------
# xCopy package
# coded by Sanjit Rath
# For bugs and suggestions mail to sanjit.rath at gmail.com
# April - September 2006
#-------------------------------------------------------------------------------

package xcopy;

require v5.6.0; 
use warnings;
use strict;
use Config qw(%Config);

#version of the script 
my $VERSION = 0.1;

#check if the thread support is available
$Config{useithreads} or die "Recompile Perl with threads to run this program.";

use IO::File;
use File::Copy;
use threads;
use threads::shared;

sub BEGIN
{
	push( @INC, './tsk' );
}

#use xop package modules
use tsk::logger;
use tsk::task;
use tsk::bucket;

sub new($$@) 
{
	my ( $class, $maxThreads, $tasksPerThread, $logFile, $errorLog ) = @_;
	my @taskList     = ();
	my $logger = new tsk::logger($logFile, $errorLog);
	my $workerThreadCount : shared = 0;  
	my $self = {
		_maxThreads => $maxThreads,
		_tasksPerThread  => $tasksPerThread,
		_logger => $logger,
		_taskListRef     => \@taskList,
		_bServerStarted  => 0,                 #server is yet to be started
		_expandThreadId  => 0,
		_workerThreads => \$workerThreadCount 
	};
	
	bless $self, $class;
	return $self;
}


sub addTask
{
	my ( $this, $strSrcDir, $strDestDir, $strFlags ) = @_;
	unless ( defined($strSrcDir) || defined($strDestDir) || defined($strFlags) )
	{
		die("xcopy Error: undefined add task parameters @_ ");
		return;
	}

	my $tsk = new tsk::task( $strSrcDir, $strDestDir, $strFlags );
	push( @{ $this->{_taskListRef} }, $tsk );
}

sub __runSerialCopy
{
	my ($this) = @_;
	my @tasks = @{ $this->{'_taskListRef'} };

	sub __fileBrowser
	{
		my ( $dir, $tsk ) = @_;
		#try opening the directory
		unless ( opendir( DIRF, $dir ) )
		{
			$this->{_logger}->error("Can't open $dir\n");
			return;
		}

		my ( $dir_item, @dirs );
		foreach $dir_item ( sort readdir(DIRF) )
		{
			if ( $dir_item eq "." || $dir_item eq ".." )
			{
				next;
			}

			my $complete_path = "$dir/$dir_item";
			if ( -d $complete_path )
			{
				push( @dirs, $complete_path );
				#it is a directory
				my $dest_path = $tsk->destPath($complete_path);
				unless ($dest_path eq "" )
				{
					if ( $this->__makeDir($dest_path) )
					{
						$this->{'_logger'}->log("mkdir: $dest_path");
					}
					else
					{
						$this->{'_logger'}->error("mkdir: $dest_path"); 
						next;
					}
				}
			}
			else
			{
				#it is a file
				my $dest_path = $tsk->destPath($complete_path);
				unless ($dest_path eq "" )
				{
					#required only if file needs to be copied 
					unless ( $this->__makeDirForFile($dest_path) )
					{
						$this->error("mkdir: $dest_path");
						next;
					}
					
					if ( copy( $complete_path, $dest_path ) )
					{
						$this->{'_logger'}->log("copy: $complete_path -> $dest_path");
					}
					else
					{
						$this->{'_logger'}->error("Error: copy $complete_path -> $dest_path");
					}
				}
			}
		}

		closedir(DIRF);

		$dir_item = "";
		foreach $dir_item (@dirs)
		{
			__fileBrowser( $dir_item, $tsk );
		}
	}

	#for each tasks in task list run file browser
	foreach my $tsk (@tasks)
	{
		__fileBrowser( $tsk->__taskSourceDir(), $tsk );
	}

}

sub __runParallelCopy
{
	my ($this) = @_;
	#inputs to the thread procedure
	# array of reference
	# 1 $bucket 
	sub __threadProc
	{
		my ($bucket, $this) = @_; 
		while($bucket->has_items)
		{
			my ( $src, $dest ) = $bucket->pop;  
			#required if only files needs to be copied 
			unless ( $this->__makeDirForFile($dest) )
			{
				$this->{'_logger'}->error("Error: mkdir $dest"); 
				next;
			}
			
			if ( copy( $src, $dest ) )
			{
				$this->{'_logger'}->log("copy: $src -> $dest");
			}
			else
			{
				$this->{'_logger'}->error("Error: copy $src -> $dest");
			}
		}
		
		#decrement worker thread count 
		$this->__removeWorkerThread(); 
	}
	
	my @tasks  = @{ $this->{'_taskListRef'} };
	my $bucket = tsk::bucket->new();
	sub __fileBrowserP
	{
		my ( $dir, $tsk ) = @_;
		#try opening the directory
		unless ( opendir( DIRF, $dir ) )
		{
			$this->{'_logger'}->error("Can't open $dir");
			return;
		}

		my ( $dir_item, @dirs );
		foreach $dir_item ( sort readdir(DIRF) )
		{
			if ( $dir_item eq "." || $dir_item eq ".." )
			{
				next;
			}
			
			my $complete_path = "$dir/$dir_item";
			if ( -d $complete_path )
			{
				push( @dirs, $complete_path );
				#it is a directory dont schedule in the job queue
				my $dest_path = $tsk->destPath($complete_path);
				unless ($dest_path eq "" )
				{
					if ( $this->__makeDir($dest_path) )
					{
						$this->{'_logger'}->log("mkdir: $dest_path");
					}
					else
					{
						$this->{'_logger'}->error("Error: mkdir $dest_path"); 
						next;
					}
				}
			}
			else
			{
				#it is a file
				my $dest_path = $tsk->destPath($complete_path);
				unless ( $dest_path eq "")
				{
					$bucket->push( $complete_path, $dest_path );
					if ( $bucket->size() == $this->{'_tasksPerThread'})
					{
						$this->__waitForWorkerThreadCount ($this->{'_maxThreads'});
						if(threads->create( \&__threadProc, $bucket, $this ))
						{
							$this->__addWorkerThread(); 
						}
						else
						{
							die("Fatal: Couldn't create thred for parallel run"); 
						}
						
						#create a new bucket object for next thread
						print "creating bucket..\n";
						$bucket = tsk::bucket->new();
					}
				}
			}
		}

		closedir(DIRF);

		$dir_item = "";
		foreach $dir_item (@dirs)
		{
			__fileBrowserP( $dir_item, $tsk );
		}
	}
	
	#for each tasks in task list run file browser
	foreach my $tsk (@tasks)
	{
		__fileBrowserP( $tsk->__taskSourceDir(), $tsk);
	}
	
	#remaining copy jobs where jobs <<< bucket size
	$this->__waitForWorkerThreadCount ($this->{'_maxThreads'});
	if(threads->create( \&__threadProc, $bucket, $this ))
	{
		$this->__addWorkerThread(); 
	}
	else
	{
		die("Fatal: Couldn't create thread for parallel run"); 
	}
}

# sub to wait for threads to finish execution
# input nothing, returns after threads have finished
sub __waitForThreads
{
	my ($this) = @_;
	$this->{'_logger'}->stop();
	
	#wait for other copy task threads to end 
	foreach (threads->list())
	{
		$_->join();
	}
}

sub __workerThreadCount()
{
	my ($this) = @_; 
	my $c = $this->{'_workerThreads'};
	lock($$c); 
	return $$c;  
}
sub __addWorkerThread()
{
	my ($this) = @_; 
	my $c = $this->{'_workerThreads'}; 
	lock($$c); 
	$$c ++ and cond_signal($$c);  
}
sub __removeWorkerThread()
{
	my ($this) = @_;
	my $c = $this->{'_workerThreads'}; 
	lock($$c); 
	$$c -- and cond_signal($$c); 
}
sub __waitForWorkerThreadCount($)
{
	my ($this, $waitCount) = @_;
	my $c = $this->{'_workerThreads'}; 
	print "waiting for ..",$$c,"and",$waitCount, "\n";
	lock($$c); 
	cond_wait($$c) until ($$c <= $waitCount); 
	print "wait done $$c\n";
}

#sub to make directory tree
#inputs directory
#returns true if directory is created false otherwise
#TODO __makeDir takes directory and creates the directory structure
sub __makeDir
{
	my ( $this, $dir ) = @_;
	$dir =~ s/\\/\//g;
	#if the directory exist return
	if ( -d $dir )
	{
		return 1;
	}
	
	my @comps = split( /\//, $dir );
	my $path = "";
	for ( my $i = 0 ; $i <= $#comps ; $i++ )
	{
		if ( $path eq "" )
		{
			$path = $comps[$i];
		}
		else
		{
			$path = "$path/$comps[$i]";
		}

		#create directory component if it doesnt exist
		unless ( -d $path )
		{
			unless ( mkdir($path, 0777) )
			{
				$this->{'_logger'}->error("Error: failed to make directory $path\n");
				return 0;
			}
		}
	}

	return ( -d $dir );
}

#sub to make directory tree out of FilePath
#input: File Path
#returns true if directory is created false otherwise
sub __makeDirForFile
{
	my ( $this, $file ) = @_;
	$file =~ s/\\/\//g;
	my @comps = split( /\//, $file );
	pop(@comps);
	my $dir = join( '/', @comps );

	#if the directory exist return
	if ( -d $dir )
	{
		return 1;
	}
	
	my $path = "";
	for ( my $i = 0 ; $i <= $#comps ; $i++ )
	{
		if ( $path eq "" )
		{
			$path = $comps[$i];
		}
		else
		{
			$path = "$path/$comps[$i]";
		}

		#create directory component if it doesnt exist
		unless ( -d $path )
		{
			unless ( mkdir($path) )
			{
				$this->{'_logger'}->error("Error: failed to make directory $path\n");
				return 0;
			}
		}
	}

	return ( -d $dir );
}

sub run()
{
	my ( $this ) = @_;
	$this->{'_logger'}->start();
	if ( $this->{'_maxThreads'} == 0 )
	{
		print "running serial copy\n"; 
		$this->__runSerialCopy();
	}
	else
	{
		print "running parallel copy\n"; 
		$this->__runParallelCopy(); 
		print "out of copy";
	} 
	$this->__waitForThreads(); 
}

1;

__END__
=head1 NAME

Module xop::xcopy

This is a generic XCOPY implementation with platform independant standard 
features, in perl, with many improvements. It uses task concept, where the task is 
expanded to subtasks and each subtask is grouped as buckets, each buckets are executed
parallel or in serial as per the arguments to task. 

It is designed for very large copy of files typically used in SCM (Souce Code Management)
enviroments  like ClearCase, CVS and Oracle ADE, where time required for copy and 
accuracy is most critical. 

Features as of version 0.1  
  i.  Stable task execution
 ii.  Serial & Parallel XCOPY  
iii.  Log file generation for each task


=head1 DESCRIPTION
xop::xcopy 
Concepts: 
xopy works by thread and bucket concept, buckets represent a group of tasks, these
are executed by a single thread. This has been designed keeping in mind the following 
factors
  i. Prevent thread rush for acessing shared task: 
      there can be two approches for the problem 
       a) constant number of running thread, and variable number of file copy sub-tasks 
          if time required for the sub-tasks are small there is possibility of thread rush
          where most CPU time is consumed by the running threads
       b) constant number of running threads and constant number of file copy sub-tasks
          Here also if time required for copy task is small, there is potential thread 
          rush problem, secondly constant running threads consume CPU time
       c) Create a thread for a constant number of file copy sub-tasks (bucket) up a 
          constant number of threads (number of threads)
          This approach solves thread rush as well as most of the CPU time is given 
          for file copy sub-tasks 
          

Requirement: 
perl version 5.8 and higher although it will work with 5.6 and higher 
This is because of the improved thread in perl in higher versions. 


Usage: 

use xop::xcopy; 
my $cp = xop::xcopy->new(<number of threads>,<size of bucket>,<log>, <error log>);

 <number of threads>: if 0,1 initiates serial copy, no threads are created 
                      if >1, n, initiates parallel copy with 'n' threads running 
 <size of bucket>   : number of task to be grouped for copy, this number is relevant only if
                      running a parallel copy, ie <number of threads> is set to > 1
 < log >            : log file 
 < error log >      : error log file
 

$cp->addTask(<source dir>,<dest dir>,<xcopy flags>);
 
<source dir>: source directory content, current version doesnt support wild cards 
<dest dir>  : destination directory
<flags>     : currently being worked on, please leave this empty, ""  

possible values for flags 
set the flags
 -d:d/m/y copies files with modification time after the said date
 -s copies directories and subdirectoriesand files
 -e copies directories and subdirectories including empty ones
 -c continue copying even if error occures, default behaviour is to stop
    the script execution
 -i copyies files from directoriy tree to a destination directory. Here
   destination directory structure is not created
 -h copies system and hidden files
 -r overwrites read only files
 -t creates directory structure only
 -u copies files only if destination file exists
 -rx: simple wild card expression, possible values *, *.*, *<some>*  
 -prx: complex perl regular expression 

more tasks can be added to the xcopy objects 

$cp->run(); 

                   

=head1 README

If there is any text in this section, it will be extracted into
a separate README file.

=head1 PREREQUISITES
The module requires perl to be compiled with thread support. 

This script requires the C<strict> module.  
It also requires the following 
C<IO::File>
C<File::Copy>
C<threads>
C<threads::shared>
C<Thread::Queue>

=head1 COREQUISITES

=pod OSNAMES

any

=pod SCRIPT CATEGORIES

CPAN/Administrative
Fun/Educational

=head1 AUTHOR

Sanjit Rath (c) 2006, March - September 2006 

I am currently working on new version of scripts with almost weekly updates. 
Feel free to suggest new features, bugs @
sanjit [.] rath [@] Oracle.com 
sanjit [.] rath [@] Gmail.com 


=cut
