/*
 * Decompiled with CFR 0.152.
 */
package io.opentelemetry.exporter.internal.okhttp;

import fish.payara.shaded.okhttp3.Call;
import fish.payara.shaded.okhttp3.Callback;
import fish.payara.shaded.okhttp3.Headers;
import fish.payara.shaded.okhttp3.HttpUrl;
import fish.payara.shaded.okhttp3.MediaType;
import fish.payara.shaded.okhttp3.OkHttpClient;
import fish.payara.shaded.okhttp3.Request;
import fish.payara.shaded.okhttp3.RequestBody;
import fish.payara.shaded.okhttp3.Response;
import fish.payara.shaded.okhttp3.ResponseBody;
import fish.payara.shaded.okio.BufferedSink;
import fish.payara.shaded.okio.GzipSink;
import fish.payara.shaded.okio.Okio;
import io.opentelemetry.api.metrics.MeterProvider;
import io.opentelemetry.exporter.internal.ExporterMetrics;
import io.opentelemetry.exporter.internal.grpc.GrpcStatusUtil;
import io.opentelemetry.exporter.internal.marshal.Marshaler;
import io.opentelemetry.exporter.internal.okhttp.JsonRequestBody;
import io.opentelemetry.exporter.internal.okhttp.ProtoRequestBody;
import io.opentelemetry.exporter.internal.retry.RetryUtil;
import io.opentelemetry.sdk.common.CompletableResultCode;
import io.opentelemetry.sdk.internal.ThrottlingLogger;
import java.io.IOException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.Nullable;

public final class OkHttpExporter<T extends Marshaler> {
    private static final Logger internalLogger = Logger.getLogger(OkHttpExporter.class.getName());
    private final ThrottlingLogger logger = new ThrottlingLogger(internalLogger);
    private final AtomicBoolean isShutdown = new AtomicBoolean();
    private final String type;
    private final OkHttpClient client;
    private final HttpUrl url;
    @Nullable
    private final Headers headers;
    private final boolean compressionEnabled;
    private final Function<T, RequestBody> requestBodyCreator;
    private final ExporterMetrics exporterMetrics;

    OkHttpExporter(String exporterName, String type, OkHttpClient client, Supplier<MeterProvider> meterProviderSupplier, String endpoint, @Nullable Headers headers, boolean compressionEnabled, boolean exportAsJson) {
        this.type = type;
        this.client = client;
        this.url = HttpUrl.get(endpoint);
        this.headers = headers;
        this.compressionEnabled = compressionEnabled;
        this.requestBodyCreator = exportAsJson ? JsonRequestBody::new : ProtoRequestBody::new;
        this.exporterMetrics = exportAsJson ? ExporterMetrics.createHttpJson(exporterName, type, meterProviderSupplier) : ExporterMetrics.createHttpProtobuf(exporterName, type, meterProviderSupplier);
    }

    public CompletableResultCode export(T exportRequest, final int numItems) {
        if (this.isShutdown.get()) {
            return CompletableResultCode.ofFailure();
        }
        this.exporterMetrics.addSeen(numItems);
        Request.Builder requestBuilder = new Request.Builder().url(this.url);
        if (this.headers != null) {
            requestBuilder.headers(this.headers);
        }
        RequestBody requestBody = this.requestBodyCreator.apply(exportRequest);
        if (this.compressionEnabled) {
            requestBuilder.addHeader("Content-Encoding", "gzip");
            requestBuilder.post(OkHttpExporter.gzipRequestBody(requestBody));
        } else {
            requestBuilder.post(requestBody);
        }
        final CompletableResultCode result = new CompletableResultCode();
        this.client.newCall(requestBuilder.build()).enqueue(new Callback(){

            @Override
            public void onFailure(Call call, IOException e) {
                OkHttpExporter.this.exporterMetrics.addFailed(numItems);
                OkHttpExporter.this.logger.log(Level.SEVERE, "Failed to export " + OkHttpExporter.this.type + "s. The request could not be executed. Full error message: " + e.getMessage());
                result.fail();
            }

            @Override
            public void onResponse(Call call, Response response) {
                try (ResponseBody body = response.body();){
                    if (response.isSuccessful()) {
                        OkHttpExporter.this.exporterMetrics.addSuccess(numItems);
                        result.succeed();
                        return;
                    }
                    OkHttpExporter.this.exporterMetrics.addFailed(numItems);
                    int code = response.code();
                    String status = OkHttpExporter.extractErrorStatus(response, body);
                    OkHttpExporter.this.logger.log(Level.WARNING, "Failed to export " + OkHttpExporter.this.type + "s. Server responded with HTTP status code " + code + ". Error message: " + status);
                    result.fail();
                }
            }
        });
        return result;
    }

    public CompletableResultCode shutdown() {
        if (!this.isShutdown.compareAndSet(false, true)) {
            this.logger.log(Level.INFO, "Calling shutdown() multiple times.");
            return CompletableResultCode.ofSuccess();
        }
        this.client.dispatcher().cancelAll();
        this.client.dispatcher().executorService().shutdownNow();
        this.client.connectionPool().evictAll();
        return CompletableResultCode.ofSuccess();
    }

    static boolean isRetryable(Response response) {
        return RetryUtil.retryableHttpResponseCodes().contains(response.code());
    }

    private static RequestBody gzipRequestBody(final RequestBody requestBody) {
        return new RequestBody(){

            @Override
            public MediaType contentType() {
                return requestBody.contentType();
            }

            @Override
            public long contentLength() {
                return -1L;
            }

            @Override
            public void writeTo(BufferedSink bufferedSink) throws IOException {
                BufferedSink gzipSink = Okio.buffer(new GzipSink(bufferedSink));
                requestBody.writeTo(gzipSink);
                gzipSink.close();
            }
        };
    }

    private static String extractErrorStatus(Response response, @Nullable ResponseBody responseBody) {
        if (responseBody == null) {
            return "Response body missing, HTTP status message: " + response.message();
        }
        try {
            return GrpcStatusUtil.getStatusMessage(responseBody.bytes());
        }
        catch (IOException e) {
            return "Unable to parse response body, HTTP status message: " + response.message();
        }
    }
}

