/*
 * Decompiled with CFR 0.152.
 */
package fish.payara.microprofile.jwtauth.eesecurity;

import fish.payara.microprofile.jwtauth.eesecurity.CacheableString;
import fish.payara.microprofile.jwtauth.eesecurity.JwtKeyStoreUtils;
import fish.payara.microprofile.jwtauth.eesecurity.KeyLoadingCache;
import jakarta.enterprise.inject.spi.DeploymentException;
import jakarta.json.JsonArray;
import jakarta.json.JsonObject;
import java.math.BigInteger;
import java.security.AlgorithmParameters;
import java.security.KeyFactory;
import java.security.PublicKey;
import java.security.spec.ECGenParameterSpec;
import java.security.spec.ECParameterSpec;
import java.security.spec.ECPoint;
import java.security.spec.ECPublicKeySpec;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.RSAPublicKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.time.Duration;
import java.util.Base64;
import java.util.Optional;
import java.util.function.Supplier;
import java.util.logging.Logger;
import org.eclipse.microprofile.config.Config;
import org.eclipse.microprofile.config.ConfigProvider;

public class JwtPublicKeyStore {
    private static final Logger LOGGER = Logger.getLogger(JwtPublicKeyStore.class.getName());
    private static final String RSA_ALGORITHM = "RSA";
    private static final String EC_ALGORITHM = "EC";
    private final Config config = ConfigProvider.getConfig();
    private final Supplier<Optional<String>> cacheSupplier;
    private final Duration defaultCacheTTL;
    private String keyLocation = "/publicKey.pem";

    public JwtPublicKeyStore(Duration defaultCacheTTL) {
        this.defaultCacheTTL = defaultCacheTTL;
        this.cacheSupplier = new KeyLoadingCache(this::readRawPublicKey)::get;
    }

    public JwtPublicKeyStore(Duration defaultCacheTTL, Optional<String> keyLocation) {
        this(defaultCacheTTL);
        this.keyLocation = keyLocation.orElse(this.keyLocation);
    }

    public PublicKey getPublicKey(String keyID) {
        return this.cacheSupplier.get().map(key -> this.createPublicKey((String)key, keyID)).orElseThrow(() -> new IllegalStateException("No PublicKey found"));
    }

    private CacheableString readRawPublicKey() {
        CacheableString publicKey = JwtKeyStoreUtils.readKeyFromLocation(this.keyLocation, this.defaultCacheTTL);
        if (!publicKey.isPresent()) {
            publicKey = this.readMPEmbeddedPublicKey();
        }
        if (!publicKey.isPresent()) {
            publicKey = JwtKeyStoreUtils.readMPKeyFromLocation(this.config, "mp.jwt.verify.publickey.location", this.defaultCacheTTL);
        }
        return publicKey;
    }

    private CacheableString readMPEmbeddedPublicKey() {
        String publicKey = this.config.getOptionalValue("mp.jwt.verify.publickey", String.class).orElse(null);
        return CacheableString.from(publicKey, this.defaultCacheTTL);
    }

    private PublicKey createPublicKey(String key, String keyID) {
        try {
            return this.createPublicKeyFromPem(key);
        }
        catch (Exception pemEx) {
            try {
                return this.createPublicKeyFromJWKS(key, keyID);
            }
            catch (Exception jwksEx) {
                throw new DeploymentException((Throwable)jwksEx);
            }
        }
    }

    private PublicKey createPublicKeyFromPem(String key) throws Exception {
        key = JwtKeyStoreUtils.trimPem(key);
        byte[] keyBytes = Base64.getDecoder().decode(key);
        X509EncodedKeySpec publicKeySpec = new X509EncodedKeySpec(keyBytes);
        try {
            return KeyFactory.getInstance(RSA_ALGORITHM).generatePublic(publicKeySpec);
        }
        catch (InvalidKeySpecException invalidKeySpecException) {
            LOGGER.finer("Caught InvalidKeySpecException creating public key from PEM using RSA algorithm, attempting again using ECDSA");
            return KeyFactory.getInstance(EC_ALGORITHM).generatePublic(publicKeySpec);
        }
    }

    private PublicKey createPublicKeyFromJWKS(String jwksValue, String keyID) throws Exception {
        JsonObject jwks = JwtKeyStoreUtils.parseJwks(jwksValue);
        JsonArray keys = jwks.getJsonArray("keys");
        JsonObject jwk = keys != null ? JwtKeyStoreUtils.findJwk(keys, keyID) : jwks;
        String kty = jwk.getString("kty");
        if (kty == null) {
            throw new DeploymentException("Could not determine key type - kty field not present");
        }
        if (kty.equals(RSA_ALGORITHM)) {
            byte[] exponentBytes = Base64.getUrlDecoder().decode(jwk.getString("e"));
            BigInteger exponent = new BigInteger(1, exponentBytes);
            byte[] modulusBytes = Base64.getUrlDecoder().decode(jwk.getString("n"));
            BigInteger modulus = new BigInteger(1, modulusBytes);
            RSAPublicKeySpec publicKeySpec = new RSAPublicKeySpec(modulus, exponent);
            return KeyFactory.getInstance(RSA_ALGORITHM).generatePublic(publicKeySpec);
        }
        if (kty.equals(EC_ALGORITHM)) {
            byte[] xBytes = Base64.getUrlDecoder().decode(jwk.getString("x"));
            BigInteger x = new BigInteger(1, xBytes);
            byte[] yBytes = Base64.getUrlDecoder().decode(jwk.getString("y"));
            BigInteger y = new BigInteger(1, yBytes);
            ECPoint ecPoint = new ECPoint(x, y);
            AlgorithmParameters parameters = AlgorithmParameters.getInstance(EC_ALGORITHM);
            String crv = jwk.getString("crv");
            if (!crv.equals("P-256")) {
                throw new DeploymentException("Could not get EC key from JWKS: crv does not equal P-256");
            }
            parameters.init(new ECGenParameterSpec("secp256r1"));
            ECPublicKeySpec publicKeySpec = new ECPublicKeySpec(ecPoint, parameters.getParameterSpec(ECParameterSpec.class));
            return KeyFactory.getInstance(EC_ALGORITHM).generatePublic(publicKeySpec);
        }
        throw new DeploymentException("Could not determine key type - JWKS kty field does not equal RSA or EC");
    }
}

