/*
 * Decompiled with CFR 0.152.
 */
package org.glassfish.security.common;

import com.sun.enterprise.util.Utility;
import com.sun.enterprise.util.i18n.StringManager;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;
import org.glassfish.security.common.PrincipalImpl;
import org.glassfish.security.common.SSHA;
import org.glassfish.security.common.SharedSecureRandomImpl;

public final class FileRealmStorageManager {
    public static final String PARAM_KEYFILE = "file";
    private static final String FIELD_SEP = ";";
    private static final String GROUP_SEP = ",";
    private static final String COMMENT = "#";
    public static final String MISC_VALID_CHARS = "_-.";
    private static final int SALT_SIZE = 8;
    private static final String SSHA_TAG = "{SSHA}";
    private static final String algoSHA = "SHA";
    private static final String algoSHA256 = "SHA-256";
    private static final String RESET_KEY = "RESET";
    private static final StringManager sm = StringManager.getManager(FileRealmStorageManager.class);
    private final HashMap<String, User> userTable = new HashMap();
    private final HashMap<String, Integer> groupSizeMap = new HashMap();
    private File keyfile;

    public FileRealmStorageManager(String keyfileName) throws IOException {
        this.keyfile = new File(keyfileName);
        if (!this.keyfile.exists() && !this.keyfile.createNewFile()) {
            throw new IOException(sm.getString("filerealm.badwrite", keyfileName));
        }
        this.loadKeyFile();
    }

    public User getUser(String name) {
        return this.userTable.get(name);
    }

    public Set<String> getUserNames() {
        return this.userTable.keySet();
    }

    public Set<String> getGroupNames() {
        return this.groupSizeMap.keySet();
    }

    public String[] getGroupNames(String username) {
        User user = this.userTable.get(username);
        if (user == null) {
            return null;
        }
        return user.getGroups();
    }

    public String[] authenticate(String username, char[] password) {
        User user = this.userTable.get(username);
        if (user == null) {
            return null;
        }
        if (RESET_KEY.equals(user.getAlgo())) {
            return null;
        }
        boolean ok = false;
        try {
            ok = SSHA.verify(user.getSalt(), user.getHash(), Utility.convertCharArrayToByteArray(password, Charset.defaultCharset().displayName()), user.getAlgo());
        }
        catch (Exception e) {
            return null;
        }
        if (!ok) {
            return null;
        }
        return user.getGroups();
    }

    public boolean hasAuthenticatableUser() {
        for (User user : this.userTable.values()) {
            if (RESET_KEY.equals(user.getAlgo())) continue;
            return true;
        }
        return false;
    }

    public synchronized void addUser(String username, char[] password, String[] groupList) throws IllegalArgumentException {
        FileRealmStorageManager.validateUserName(username);
        FileRealmStorageManager.validatePassword(password);
        FileRealmStorageManager.validateGroupList(groupList);
        if (this.userTable.containsKey(username)) {
            throw new IllegalArgumentException(sm.getString("filerealm.dupuser", username));
        }
        this.addGroupNames(groupList);
        this.userTable.put(username, FileRealmStorageManager.createNewUser(username, password, groupList));
    }

    public synchronized void removeUser(String username) throws IllegalArgumentException {
        if (!this.userTable.containsKey(username)) {
            throw new IllegalArgumentException(sm.getString("filerealm.nouser", username));
        }
        User oldUser = this.userTable.get(username);
        this.userTable.remove(username);
        this.reduceGroups(oldUser.getGroups());
    }

    public synchronized void updateUser(String username, String newUsername, char[] password, String[] groups) throws IllegalArgumentException {
        FileRealmStorageManager.validateUserName(username);
        if (!this.userTable.containsKey(username)) {
            throw new IllegalArgumentException(sm.getString("filerealm.nouser", username));
        }
        FileRealmStorageManager.validateUserName(newUsername);
        FileRealmStorageManager.validateGroupList(groups);
        if (password != null) {
            FileRealmStorageManager.validatePassword(password);
        }
        if (!username.equals(newUsername) && this.userTable.containsKey(newUsername)) {
            throw new IllegalArgumentException(sm.getString("filerealm.dupuser", username));
        }
        User oldUser = this.userTable.get(username);
        assert (oldUser != null);
        User newUser = new User(newUsername);
        if (groups != null) {
            this.changeGroups(oldUser.getGroups(), groups);
            newUser.setGroups(groups);
        } else {
            newUser.setGroups(oldUser.getGroups());
        }
        if (password == null) {
            newUser.setSalt(oldUser.getSalt());
            newUser.setHash(oldUser.getHash());
            if (oldUser.getAlgo().equals(RESET_KEY)) {
                newUser.setAlgo(algoSHA256);
            } else {
                newUser.setAlgo(oldUser.getAlgo());
            }
        } else {
            FileRealmStorageManager.setPassword(newUser, password);
            newUser.setAlgo(algoSHA256);
        }
        this.userTable.remove(username);
        this.userTable.put(newUsername, newUser);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void persist() throws IOException {
        Class<FileRealmStorageManager> clazz = FileRealmStorageManager.class;
        synchronized (FileRealmStorageManager.class) {
            try (FileOutputStream out = new FileOutputStream(this.keyfile);){
                for (Map.Entry<String, User> userEntry : this.userTable.entrySet()) {
                    out.write(FileRealmStorageManager.encodeUser(userEntry.getKey(), userEntry.getValue(), userEntry.getValue().getAlgo()).getBytes(StandardCharsets.UTF_8));
                }
            }
            catch (IOException e) {
                throw e;
            }
            catch (Exception e) {
                throw new IOException(sm.getString("filerealm.badwrite", e.toString()));
            }
            // ** MonitorExit[var1_1] (shouldn't be in output)
            return;
        }
    }

    public static void validateUserName(String name) throws IllegalArgumentException {
        if (name == null || name.length() == 0) {
            throw new IllegalArgumentException(sm.getString("filerealm.noname"));
        }
        if (!FileRealmStorageManager.isValid(name, true)) {
            throw new IllegalArgumentException(sm.getString("filerealm.badname", name));
        }
        if (!name.equals(name.trim())) {
            throw new IllegalArgumentException(sm.getString("filerealm.badspaces", name));
        }
    }

    public static void validatePassword(char[] password) throws IllegalArgumentException {
        if (Arrays.equals(null, password)) {
            throw new IllegalArgumentException(sm.getString("filerealm.emptypwd"));
        }
        for (char character : password) {
            if (!Character.isSpaceChar(character)) continue;
            throw new IllegalArgumentException(sm.getString("filerealm.badspacespwd"));
        }
    }

    public static void validateGroupName(String group) throws IllegalArgumentException {
        if (group == null || group.length() == 0) {
            throw new IllegalArgumentException(sm.getString("filerealm.nogroup"));
        }
        if (!FileRealmStorageManager.isValid(group, false)) {
            throw new IllegalArgumentException(sm.getString("filerealm.badchars", group));
        }
        if (!group.equals(group.trim())) {
            throw new IllegalArgumentException(sm.getString("filerealm.badspaces", group));
        }
    }

    public static void validateGroupList(String[] groupNames) throws IllegalArgumentException {
        if (groupNames == null || groupNames.length == 0) {
            return;
        }
        for (String groupName : groupNames) {
            FileRealmStorageManager.validateGroupName(groupName);
        }
    }

    private void addGroupNames(String[] groupNames) {
        if (groupNames != null) {
            String[] stringArray = groupNames;
            int n = stringArray.length;
            for (int i = 0; i < n; ++i) {
                String groupName;
                Integer groupSize = this.groupSizeMap.get(groupName = stringArray[i]);
                this.groupSizeMap.put(groupName, groupSize != null ? groupSize + 1 : 1);
            }
        }
    }

    private void reduceGroups(String[] groupNames) {
        if (groupNames != null) {
            for (String groupName : groupNames) {
                Integer groupSize = this.groupSizeMap.get(groupName);
                if (groupSize == null) continue;
                int newGroupSize = groupSize - 1;
                if (newGroupSize > 0) {
                    this.groupSizeMap.put(groupName, newGroupSize);
                    continue;
                }
                this.groupSizeMap.remove(groupName);
            }
        }
    }

    private void changeGroups(String[] oldGroupNames, String[] newGroupNames) {
        this.addGroupNames(newGroupNames);
        this.reduceGroups(oldGroupNames);
    }

    private static boolean isValid(String s, boolean userName) {
        for (int i = 0; i < s.length(); ++i) {
            char c = s.charAt(i);
            if (Character.isLetterOrDigit(c) || Character.isWhitespace(c) || MISC_VALID_CHARS.indexOf(c) != -1 || userName && c == '@') continue;
            return false;
        }
        return true;
    }

    private void loadKeyFile() throws IOException {
        try (BufferedReader input = new BufferedReader(new FileReader(this.keyfile, StandardCharsets.UTF_8));){
            while (input.ready()) {
                String line = input.readLine();
                if (line == null || line.startsWith(COMMENT) || !line.contains(FIELD_SEP)) continue;
                User user = FileRealmStorageManager.decodeUser(line, this.groupSizeMap);
                this.userTable.put(user.getName(), user);
            }
        }
        catch (Exception e) {
            throw new IOException(e.toString(), e);
        }
    }

    private static String encodeUser(String name, User user, String algorithm) {
        StringBuilder encodedUser = new StringBuilder();
        encodedUser.append(name);
        encodedUser.append(FIELD_SEP);
        if (RESET_KEY.equals(algorithm)) {
            encodedUser.append(RESET_KEY);
        } else {
            encodedUser.append(SSHA.encode(user.getSalt(), user.getHash(), algorithm));
        }
        encodedUser.append(FIELD_SEP);
        String[] groups = user.getGroups();
        if (groups != null) {
            for (int grp = 0; grp < groups.length; ++grp) {
                if (grp > 0) {
                    encodedUser.append(GROUP_SEP);
                }
                encodedUser.append(groups[grp]);
            }
        }
        encodedUser.append("\n");
        return encodedUser.toString();
    }

    private static User decodeUser(String encodedLine, HashMap<String, Integer> newGroupSizeMap) throws IllegalArgumentException {
        StringTokenizer userTokenizer = new StringTokenizer(encodedLine, FIELD_SEP);
        String algo = algoSHA256;
        String username = null;
        String pwdInfo = null;
        String groupNames = null;
        try {
            username = userTokenizer.nextToken();
            pwdInfo = userTokenizer.nextToken();
        }
        catch (Exception e) {
            throw new IllegalArgumentException(sm.getString("filerealm.syntaxerror", encodedLine));
        }
        if (userTokenizer.hasMoreTokens()) {
            groupNames = userTokenizer.nextToken();
        }
        User user = new User(username);
        if (RESET_KEY.equals(pwdInfo)) {
            user.setAlgo(RESET_KEY);
        } else {
            if (encodedLine.contains(SSHA_TAG)) {
                algo = algoSHA;
            }
            int resultLength = 32;
            if (algoSHA.equals(algo)) {
                resultLength = 20;
            }
            byte[] hash = new byte[resultLength];
            byte[] salt = SSHA.decode(pwdInfo, hash, algo);
            user.setHash(hash);
            user.setSalt(salt);
            user.setAlgo(algo);
        }
        ArrayList<String> membership = new ArrayList<String>();
        if (groupNames != null) {
            StringTokenizer groupNameTokenizer = new StringTokenizer(groupNames, GROUP_SEP);
            while (groupNameTokenizer.hasMoreTokens()) {
                String groupName = groupNameTokenizer.nextToken();
                membership.add(groupName);
                Integer groupSize = newGroupSizeMap.get(groupName);
                newGroupSizeMap.put(groupName, groupSize != null ? groupSize + 1 : 1);
            }
        }
        user.setGroups(membership.toArray(new String[membership.size()]));
        return user;
    }

    private static User createNewUser(String username, char[] password, String[] groups) {
        User user = new User(username);
        user.setGroups(groups == null ? new String[]{} : groups);
        user.setAlgo(algoSHA256);
        FileRealmStorageManager.setPassword(user, password);
        return user;
    }

    private static void setPassword(User user, char[] password) throws IllegalArgumentException {
        assert (user != null);
        byte[] pwdBytes = null;
        try {
            pwdBytes = Utility.convertCharArrayToByteArray(password, Charset.defaultCharset().displayName());
        }
        catch (Exception ex) {
            throw new IllegalArgumentException(ex);
        }
        SecureRandom secureRandom = SharedSecureRandomImpl.get();
        byte[] salt = new byte[8];
        secureRandom.nextBytes(salt);
        user.setSalt(salt);
        String algo = user.getAlgo();
        if (algo == null) {
            algo = algoSHA256;
        }
        user.setHash(SSHA.compute(salt, pwdBytes, algo));
    }

    public static class User
    extends PrincipalImpl {
        private static final long serialVersionUID = 5310671725001301966L;
        private String[] groups;
        private String realm;
        private byte[] salt;
        private byte[] hash;
        private String algo;

        public User(String name) {
            super(name);
        }

        public User(String name, String[] groups, String realm, byte[] salt, byte[] hash, String algo) {
            super(name);
            this.groups = groups;
            this.realm = realm;
            this.hash = hash;
            this.salt = salt;
            this.algo = algo;
        }

        @Override
        public boolean equals(Object obj) {
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            User other = (User)obj;
            if (!Arrays.deepEquals(this.groups, other.groups)) {
                return false;
            }
            if (this.realm == null ? other.realm != null : !this.realm.equals(other.realm)) {
                return false;
            }
            if (!Arrays.equals(this.salt, other.salt)) {
                return false;
            }
            if (!Arrays.equals(this.hash, other.hash)) {
                return false;
            }
            if (this.algo == null ? other.algo != null : !this.algo.equals(other.algo)) {
                return false;
            }
            return super.equals(obj);
        }

        @Override
        public int hashCode() {
            int hc = 5;
            hc = 17 * hc + Arrays.deepHashCode(this.groups);
            hc = 17 * hc + (this.realm != null ? this.realm.hashCode() : 0);
            hc = 17 * hc + Arrays.hashCode(this.salt);
            hc = 17 * hc + Arrays.hashCode(this.hash);
            hc = 17 * hc + (this.algo != null ? this.algo.hashCode() : 0);
            hc = 17 * hc + super.hashCode();
            return hc;
        }

        public byte[] getSalt() {
            return this.salt;
        }

        public void setSalt(byte[] salt) {
            this.salt = salt;
        }

        public byte[] getHash() {
            return this.hash;
        }

        public void setHash(byte[] hash) {
            this.hash = hash;
        }

        public String[] getGroups() {
            return this.groups;
        }

        public void setGroups(String[] grp) {
            this.groups = grp;
        }

        public String getAlgo() {
            return this.algo;
        }

        public void setAlgo(String algo) {
            this.algo = algo;
        }
    }
}

