/*
 * Decompiled with CFR 0.152.
 */
package fish.payara.microprofile.jwtauth.jwt;

import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.JOSEObjectType;
import com.nimbusds.jose.JWEAlgorithm;
import com.nimbusds.jose.JWEDecrypter;
import com.nimbusds.jose.JWSAlgorithm;
import com.nimbusds.jose.JWSVerifier;
import com.nimbusds.jose.crypto.ECDSAVerifier;
import com.nimbusds.jose.crypto.RSADecrypter;
import com.nimbusds.jose.crypto.RSASSAVerifier;
import com.nimbusds.jwt.EncryptedJWT;
import com.nimbusds.jwt.SignedJWT;
import fish.payara.microprofile.jwtauth.eesecurity.JWTProcessingException;
import fish.payara.microprofile.jwtauth.eesecurity.JwtPrivateKeyStore;
import fish.payara.microprofile.jwtauth.eesecurity.JwtPublicKeyStore;
import fish.payara.microprofile.jwtauth.jwt.JsonWebTokenImpl;
import jakarta.json.Json;
import jakarta.json.JsonNumber;
import jakarta.json.JsonReader;
import jakarta.json.JsonString;
import jakarta.json.JsonValue;
import java.io.Reader;
import java.io.StringReader;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.interfaces.ECPublicKey;
import java.security.interfaces.RSAPublicKey;
import java.text.ParseException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import org.eclipse.microprofile.jwt.Claims;

public class JwtTokenParser {
    private static final String DEFAULT_NAMESPACE = "https://payara.fish/mp-jwt/";
    private final List<Claims> requiredClaims = Arrays.asList(Claims.iss, Claims.sub, Claims.exp, Claims.iat, Claims.jti);
    private final boolean enableNamespacedClaims;
    private final boolean disableTypeVerification;
    private final Optional<String> customNamespace;
    private String rawToken;
    private SignedJWT signedJWT;
    private EncryptedJWT encryptedJWT;

    public JwtTokenParser(Optional<Boolean> enableNamespacedClaims, Optional<String> customNamespace, Optional<Boolean> disableTypeVerification) {
        this.enableNamespacedClaims = enableNamespacedClaims.orElse(false);
        this.disableTypeVerification = disableTypeVerification.orElse(false);
        this.customNamespace = customNamespace;
    }

    public JwtTokenParser() {
        this(Optional.empty(), Optional.empty(), Optional.empty());
    }

    public JsonWebTokenImpl parse(String bearerToken, boolean encryptionRequired, JwtPublicKeyStore publicKeyStore, String acceptedIssuer, JwtPrivateKeyStore privateKeyStore, Map<String, Optional<String>> optionalConfigProps) throws JWTProcessingException {
        JsonWebTokenImpl jsonWebToken;
        try {
            String keyId;
            this.rawToken = bearerToken;
            int parts = this.rawToken.split("\\.", 5).length;
            if (parts == 3) {
                this.signedJWT = SignedJWT.parse((String)this.rawToken);
                if (!this.checkIsSignedJWT(this.signedJWT)) {
                    throw new JWTProcessingException("Not signed JWT, typ must be 'JWT'");
                }
                keyId = this.signedJWT.getHeader().getKeyID();
            } else {
                this.encryptedJWT = EncryptedJWT.parse((String)this.rawToken);
                if (!this.checkIsEncryptedJWT(this.encryptedJWT)) {
                    throw new JWTProcessingException("Not encrypted JWT, cty must be 'JWT'");
                }
                keyId = this.encryptedJWT.getHeader().getKeyID();
            }
            PublicKey publicKey = publicKeyStore.getPublicKey(keyId);
            if (this.encryptedJWT == null) {
                if (encryptionRequired) {
                    throw new JWTProcessingException("JWT expected to be encrypted, mp.jwt.decrypt.key.location was defined!");
                }
                jsonWebToken = this.verifyAndParseSignedJWT(acceptedIssuer, publicKey, optionalConfigProps.get("mp.jwt.verify.token.age").map(Long::valueOf), optionalConfigProps.get("mp.jwt.verify.clock.skew").map(Long::valueOf));
            } else {
                jsonWebToken = this.verifyAndParseEncryptedJWT(acceptedIssuer, publicKey, privateKeyStore.getPrivateKey(keyId), optionalConfigProps.get("mp.jwt.verify.token.age").map(Long::valueOf), optionalConfigProps.get("mp.jwt.verify.clock.skew").map(Long::valueOf), optionalConfigProps.get("mp.jwt.decrypt.key.algorithm"));
            }
        }
        catch (JWTProcessingException | ParseException ex) {
            throw new JWTProcessingException(ex);
        }
        return jsonWebToken;
    }

    private JsonWebTokenImpl verifyAndParseSignedJWT(String issuer, PublicKey publicKey, Optional<Long> tokenAge, Optional<Long> allowedClockSkew) throws JWTProcessingException {
        if (this.signedJWT == null) {
            throw new IllegalStateException("No parsed SignedJWT.");
        }
        JWSAlgorithm signAlgorithmName = this.signedJWT.getHeader().getAlgorithm();
        if (!signAlgorithmName.equals((Object)JWSAlgorithm.RS256) && !signAlgorithmName.equals((Object)JWSAlgorithm.ES256)) {
            throw new JWTProcessingException("Only RS256 or ES256 algorithms supported for JWT signing, used " + signAlgorithmName);
        }
        try (JsonReader reader = Json.createReader((Reader)new StringReader(this.signedJWT.getPayload().toString()));){
            Map<String, JsonValue> rawClaims = new HashMap<String, JsonValue>((Map<String, JsonValue>)reader.readObject());
            rawClaims = this.handleNamespacedClaims(rawClaims);
            String callerPrincipalName = this.getCallerPrincipalName(rawClaims);
            this.verifySignedJWT(issuer, rawClaims, signAlgorithmName, callerPrincipalName, publicKey, tokenAge, allowedClockSkew);
            rawClaims.put(Claims.raw_token.name(), (JsonValue)Json.createObjectBuilder().add("token", this.rawToken).build().get((Object)"token"));
            JsonWebTokenImpl jsonWebTokenImpl = new JsonWebTokenImpl(callerPrincipalName, rawClaims);
            return jsonWebTokenImpl;
        }
    }

    private void verifySignedJWT(String issuer, Map<String, JsonValue> rawClaims, JWSAlgorithm signAlgorithmName, String callerPrincipalName, PublicKey publicKey, Optional<Long> tokenAge, Optional<Long> allowedClockSkew) throws JWTProcessingException {
        if (!this.checkRequiredClaimsPresent(rawClaims)) {
            throw new JWTProcessingException("Not all required claims present");
        }
        if (callerPrincipalName == null) {
            throw new JWTProcessingException("One of upn, preferred_username or sub is required to be non null");
        }
        if (!this.checkIssuer(rawClaims, issuer)) {
            throw new JWTProcessingException("Bad issuer");
        }
        if (!this.checkNotExpired(rawClaims, allowedClockSkew)) {
            throw new JWTProcessingException("JWT token expired");
        }
        if (tokenAge.isPresent() && this.checkIsTokenAged(rawClaims, tokenAge.get(), allowedClockSkew)) {
            throw new JWTProcessingException(String.format("The token age has exceeded %d seconds", tokenAge.get()));
        }
        try {
            if (signAlgorithmName.equals((Object)JWSAlgorithm.RS256) ? !this.signedJWT.verify((JWSVerifier)new RSASSAVerifier((RSAPublicKey)publicKey)) : !this.signedJWT.verify((JWSVerifier)new ECDSAVerifier((ECPublicKey)publicKey))) {
                throw new JWTProcessingException("Signature of the JWT token is invalid");
            }
        }
        catch (JOSEException ex) {
            throw new JWTProcessingException("Exception during JWT signature validation", ex);
        }
    }

    private JsonWebTokenImpl verifyAndParseEncryptedJWT(String issuer, PublicKey publicKey, PrivateKey privateKey, Optional<Long> tokenAge, Optional<Long> allowedClockSkew, Optional<String> keyAlgorithm) throws JWTProcessingException {
        if (this.encryptedJWT == null) {
            throw new IllegalStateException("EncryptedJWT not parsed");
        }
        String algName = this.encryptedJWT.getHeader().getAlgorithm().getName();
        if (!JWEAlgorithm.RSA_OAEP.getName().equals(algName) && !JWEAlgorithm.RSA_OAEP_256.getName().equals(algName)) {
            throw new JWTProcessingException("Only RSA-OAEP and RSA-OAEP-256 algorithms are supported for JWT encryption, used " + algName);
        }
        if (keyAlgorithm.isPresent() && !algName.equals(keyAlgorithm.get())) {
            throw new JWTProcessingException("Key algorithm configuration specified, thus only accept " + keyAlgorithm.get());
        }
        try {
            this.encryptedJWT.decrypt((JWEDecrypter)new RSADecrypter(privateKey));
        }
        catch (JOSEException ex) {
            throw new JWTProcessingException("Exception during decrypting JWT token", ex);
        }
        this.signedJWT = this.encryptedJWT.getPayload().toSignedJWT();
        if (this.signedJWT == null) {
            throw new JWTProcessingException("Unable to parse signed JWT.");
        }
        return this.verifyAndParseSignedJWT(issuer, publicKey, tokenAge, allowedClockSkew);
    }

    private Map<String, JsonValue> handleNamespacedClaims(Map<String, JsonValue> currentClaims) {
        if (this.enableNamespacedClaims) {
            String namespace = this.customNamespace.orElse(DEFAULT_NAMESPACE);
            HashMap<String, JsonValue> processedClaims = new HashMap<String, JsonValue>(currentClaims.size());
            for (Map.Entry<String, JsonValue> entry : currentClaims.entrySet()) {
                String claimName = entry.getKey();
                JsonValue value = entry.getValue();
                if (claimName.startsWith(namespace)) {
                    claimName = claimName.substring(namespace.length());
                }
                processedClaims.put(claimName, value);
            }
            return processedClaims;
        }
        return currentClaims;
    }

    private boolean checkRequiredClaimsPresent(Map<String, JsonValue> presentedClaims) {
        for (Claims requiredClaim : this.requiredClaims) {
            if (presentedClaims.get(requiredClaim.name()) != null) continue;
            return false;
        }
        return true;
    }

    private boolean checkNotExpired(Map<String, JsonValue> presentedClaims, Optional<Long> allowedClockSkew) {
        long currentTime = System.currentTimeMillis() / 1000L;
        if (allowedClockSkew.isPresent()) {
            currentTime -= allowedClockSkew.get().longValue();
        }
        long expiredTime = ((JsonNumber)presentedClaims.get(Claims.exp.name())).longValue();
        long issueTime = ((JsonNumber)presentedClaims.get(Claims.iat.name())).longValue();
        return currentTime < expiredTime && issueTime < expiredTime;
    }

    private boolean checkIsTokenAged(Map<String, JsonValue> presentedClaims, long tokenAge, Optional<Long> allowedClockSkew) {
        long issueTime;
        long currentTime = System.currentTimeMillis() / 1000L;
        if (allowedClockSkew.isPresent()) {
            currentTime -= allowedClockSkew.get().longValue();
        }
        return currentTime - (issueTime = ((JsonNumber)presentedClaims.get(Claims.iat.name())).longValue()) > tokenAge;
    }

    private boolean checkIssuer(Map<String, JsonValue> presentedClaims, String acceptedIssuer) {
        if (!(presentedClaims.get(Claims.iss.name()) instanceof JsonString)) {
            return false;
        }
        String issuer = ((JsonString)presentedClaims.get(Claims.iss.name())).getString();
        return acceptedIssuer.equals(issuer);
    }

    private boolean checkIsSignedJWT(SignedJWT jwt) {
        return this.disableTypeVerification || Optional.ofNullable(jwt.getHeader().getType()).map(JOSEObjectType::toString).orElse("").equals("JWT");
    }

    private boolean checkIsEncryptedJWT(EncryptedJWT jwt) {
        return this.disableTypeVerification || Optional.ofNullable(jwt.getHeader().getContentType()).orElse("").equals("JWT");
    }

    private String getCallerPrincipalName(Map<String, JsonValue> rawClaims) {
        JsonString callerPrincipalClaim = (JsonString)rawClaims.get(Claims.upn.name());
        if (callerPrincipalClaim == null) {
            callerPrincipalClaim = (JsonString)rawClaims.get(Claims.preferred_username.name());
        }
        if (callerPrincipalClaim == null) {
            callerPrincipalClaim = (JsonString)rawClaims.get(Claims.sub.name());
        }
        if (callerPrincipalClaim == null) {
            return null;
        }
        return callerPrincipalClaim.getString();
    }
}

