/**
 * Declarations for the ELK Layered layout algorithm.
 */
package org.eclipse.elk.alg.layered.properties;

import java.util.EnumSet;
import org.eclipse.elk.alg.layered.LayeredLayoutProvider;
import org.eclipse.elk.alg.layered.intermediate.NodePromotionStrategy;
import org.eclipse.elk.alg.layered.intermediate.compaction.ConstraintCalculationStrategy;
import org.eclipse.elk.alg.layered.intermediate.compaction.GraphCompactionStrategy;
import org.eclipse.elk.alg.layered.p1cycles.CycleBreakingStrategy;
import org.eclipse.elk.alg.layered.p2layers.LayeringStrategy;
import org.eclipse.elk.alg.layered.p3order.CrossingMinimizationStrategy;
import org.eclipse.elk.alg.layered.p4nodes.NodePlacementStrategy;
import org.eclipse.elk.alg.layered.p4nodes.bk.EdgeStraighteningStrategy;
import org.eclipse.elk.alg.layered.properties.ContentAlignment;
import org.eclipse.elk.alg.layered.properties.EdgeLabelSideSelection;
import org.eclipse.elk.alg.layered.properties.FixedAlignment;
import org.eclipse.elk.alg.layered.properties.GreedySwitchType;
import org.eclipse.elk.alg.layered.properties.InteractiveReferencePoint;
import org.eclipse.elk.alg.layered.properties.LayerConstraint;
import org.eclipse.elk.alg.layered.properties.LayeredMetaDataProvider;
import org.eclipse.elk.alg.layered.properties.SelfLoopPlacement;
import org.eclipse.elk.alg.layered.properties.WideNodesStrategy;
import org.eclipse.elk.core.data.ILayoutMetaDataProvider;
import org.eclipse.elk.core.math.ElkMargin;
import org.eclipse.elk.core.math.ElkPadding;
import org.eclipse.elk.core.math.KVector;
import org.eclipse.elk.core.math.KVectorChain;
import org.eclipse.elk.core.options.Alignment;
import org.eclipse.elk.core.options.CoreOptions;
import org.eclipse.elk.core.options.Direction;
import org.eclipse.elk.core.options.EdgeLabelPlacement;
import org.eclipse.elk.core.options.EdgeLabelPlacementStrategy;
import org.eclipse.elk.core.options.EdgeRouting;
import org.eclipse.elk.core.options.HierarchyHandling;
import org.eclipse.elk.core.options.NodeLabelPlacement;
import org.eclipse.elk.core.options.PortAlignment;
import org.eclipse.elk.core.options.PortConstraints;
import org.eclipse.elk.core.options.PortLabelPlacement;
import org.eclipse.elk.core.options.PortSide;
import org.eclipse.elk.core.options.SizeConstraint;
import org.eclipse.elk.core.options.SizeOptions;
import org.eclipse.elk.core.util.IndividualSpacings;
import org.eclipse.elk.graph.properties.IProperty;
import org.eclipse.elk.graph.properties.Property;

@SuppressWarnings("all")
public class LayeredOptions implements ILayoutMetaDataProvider {
  /**
   * The id of the ELK Layered algorithm.
   */
  public final static String ALGORITHM_ID = "org.eclipse.elk.layered";
  
  /**
   * Spacing to be preserved between pairs of connected components.
   * This option is only relevant if 'separateConnectedComponents' is activated.
   */
  public final static IProperty<Double> SPACING_COMPONENT_COMPONENT = CoreOptions.SPACING_COMPONENT_COMPONENT;
  
  /**
   * Spacing to be preserved between any two edges. Note that while this can somewhat easily be satisfied
   * for the segments of orthogonally drawn edges, it is harder for general polylines or splines.
   */
  public final static IProperty<Double> SPACING_EDGE_EDGE = CoreOptions.SPACING_EDGE_EDGE;
  
  /**
   * The minimal distance to be preserved between a label and the edge it is associated with.
   * Note that the placement of a label is influenced by the 'edgelabels.placement' option.
   */
  public final static IProperty<Double> SPACING_EDGE_LABEL = CoreOptions.SPACING_EDGE_LABEL;
  
  /**
   * Spacing to be preserved between nodes and edges.
   */
  public final static IProperty<Double> SPACING_EDGE_NODE = CoreOptions.SPACING_EDGE_NODE;
  
  /**
   * Determines the amount of space to be left between two labels
   * of the same graph element.
   */
  public final static IProperty<Double> SPACING_LABEL_LABEL = CoreOptions.SPACING_LABEL_LABEL;
  
  /**
   * Spacing to be preserved between labels and the ports they are associated with.
   * Note that the placement of a label is influenced by the 'portlabels.placement' option.
   */
  public final static IProperty<Double> SPACING_LABEL_PORT = CoreOptions.SPACING_LABEL_PORT;
  
  /**
   * Spacing to be preserved between labels and the border of node they are associated with.
   * Note that the placement of a label is influenced by the 'nodelabels.placement' option.
   */
  public final static IProperty<Double> SPACING_LABEL_NODE = CoreOptions.SPACING_LABEL_NODE;
  
  /**
   * Default value for {@link #SPACING_NODE_NODE} with algorithm "ELK Layered".
   */
  private final static double SPACING_NODE_NODE_DEFAULT = 20;
  
  /**
   * The minimal distance to be preserved between each two nodes.
   */
  public final static IProperty<Double> SPACING_NODE_NODE = new Property<Double>(
                                CoreOptions.SPACING_NODE_NODE,
                                SPACING_NODE_NODE_DEFAULT);
  
  /**
   * Spacing between pairs of ports of the same node.
   */
  public final static IProperty<Double> SPACING_PORT_PORT = CoreOptions.SPACING_PORT_PORT;
  
  /**
   * In general spacing values apply to the children of the hierarchical node
   * (possibly the root node) for which the values are actually specified. Hereby,
   * the children include ports, edges, and labels.
   * <h3>Algorithm Specific Details</h3>
   * Most parts of the algorithm do not support this yet.
   */
  public final static IProperty<IndividualSpacings> SPACING_INDIVIDUAL_OVERRIDE = CoreOptions.SPACING_INDIVIDUAL_OVERRIDE;
  
  /**
   * Spacing to be preserved between pairs of edges that are routed between the same pair of layers.
   * Note that 'spacing.edgeEdge' is used for the spacing between pairs of edges crossing the same layer.
   */
  public final static IProperty<Double> SPACING_EDGE_EDGE_BETWEEN_LAYERS = LayeredMetaDataProvider.SPACING_EDGE_EDGE_BETWEEN_LAYERS;
  
  /**
   * The spacing to be preserved between nodes and edges that are routed next to the node's layer.
   * For the spacing between nodes and edges that cross the node's layer 'spacing.edgeNode' is used.
   */
  public final static IProperty<Double> SPACING_EDGE_NODE_BETWEEN_LAYERS = LayeredMetaDataProvider.SPACING_EDGE_NODE_BETWEEN_LAYERS;
  
  /**
   * The spacing to be preserved between any pair of nodes of two adjacent layers.
   * Note that 'spacing.nodeNode' is used for the spacing between nodes within the layer itself.
   */
  public final static IProperty<Double> SPACING_NODE_NODE_BETWEEN_LAYERS = LayeredMetaDataProvider.SPACING_NODE_NODE_BETWEEN_LAYERS;
  
  /**
   * Default value for {@link #PRIORITY} with algorithm "ELK Layered".
   */
  private final static int PRIORITY_DEFAULT = 0;
  
  /**
   * Defines the priority of an object; its meaning depends on the specific layout algorithm
   * and the context where it is used.
   * <h3>Algorithm Specific Details</h3>
   * Used by the 'simple row graph placer' to decide which connected components to place first.
   * A component's priority is the sum of the node priorities.
   */
  public final static IProperty<Integer> PRIORITY = new Property<Integer>(
                                CoreOptions.PRIORITY,
                                PRIORITY_DEFAULT);
  
  /**
   * Defines how important it is to have a certain edge point into the direction of the overall layout.
   * This option is evaluated during the cycle breaking phase.
   */
  public final static IProperty<Integer> PRIORITY_DIRECTION = LayeredMetaDataProvider.PRIORITY_DIRECTION;
  
  /**
   * Defines how important it is to keep an edge as short as possible.
   * This option is evaluated during the layering phase.
   * <h3>Algorithm Specific Details</h3>
   * Currently only supported by the network simplex layerer.
   */
  public final static IProperty<Integer> PRIORITY_SHORTNESS = LayeredMetaDataProvider.PRIORITY_SHORTNESS;
  
  /**
   * Defines how important it is to keep an edge straight, i.e. aligned with one of the two axes.
   * This option is evaluated during node placement.
   */
  public final static IProperty<Integer> PRIORITY_STRAIGHTNESS = LayeredMetaDataProvider.PRIORITY_STRAIGHTNESS;
  
  /**
   * Default value for {@link #PADDING} with algorithm "ELK Layered".
   */
  private final static ElkPadding PADDING_DEFAULT = new ElkPadding(12);
  
  /**
   * The padding to be left to a parent element's border when placing child elements. This can
   * also serve as an output option of a layout algorithm if node size calculation is setup
   * appropriately.
   */
  public final static IProperty<ElkPadding> PADDING = new Property<ElkPadding>(
                                CoreOptions.PADDING,
                                PADDING_DEFAULT);
  
  /**
   * Default value for {@link #EDGE_ROUTING} with algorithm "ELK Layered".
   */
  private final static EdgeRouting EDGE_ROUTING_DEFAULT = EdgeRouting.ORTHOGONAL;
  
  /**
   * What kind of edge routing style should be applied for the content of a parent node.
   * Algorithms may also set this option to single edges in order to mark them as splines.
   * The bend point list of edges with this option set to SPLINES must be interpreted as control
   * points for a piecewise cubic spline.
   */
  public final static IProperty<EdgeRouting> EDGE_ROUTING = new Property<EdgeRouting>(
                                CoreOptions.EDGE_ROUTING,
                                EDGE_ROUTING_DEFAULT);
  
  /**
   * Default value for {@link #PORT_BORDER_OFFSET} with algorithm "ELK Layered".
   */
  private final static double PORT_BORDER_OFFSET_DEFAULT = 0;
  
  /**
   * The offset of ports on the node border. With a positive offset the port is moved outside
   * of the node, while with a negative offset the port is moved towards the inside. An offset
   * of 0 means that the port is placed directly on the node border, i.e.
   * if the port side is north, the port's south border touches the nodes's north border;
   * if the port side is east, the port's west border touches the nodes's east border;
   * if the port side is south, the port's north border touches the node's south border;
   * if the port side is west, the port's east border touches the node's west border.
   */
  public final static IProperty<Double> PORT_BORDER_OFFSET = new Property<Double>(
                                CoreOptions.PORT_BORDER_OFFSET,
                                PORT_BORDER_OFFSET_DEFAULT);
  
  /**
   * Default value for {@link #RANDOM_SEED} with algorithm "ELK Layered".
   */
  private final static int RANDOM_SEED_DEFAULT = 1;
  
  /**
   * Seed used for pseudo-random number generators to control the layout algorithm. If the
   * value is 0, the seed shall be determined pseudo-randomly (e.g. from the system time).
   */
  public final static IProperty<Integer> RANDOM_SEED = new Property<Integer>(
                                CoreOptions.RANDOM_SEED,
                                RANDOM_SEED_DEFAULT);
  
  /**
   * Default value for {@link #ASPECT_RATIO} with algorithm "ELK Layered".
   */
  private final static double ASPECT_RATIO_DEFAULT = 1.6f;
  
  /**
   * The desired aspect ratio of the drawing, that is the quotient of width by height.
   */
  public final static IProperty<Double> ASPECT_RATIO = new Property<Double>(
                                CoreOptions.ASPECT_RATIO,
                                ASPECT_RATIO_DEFAULT);
  
  /**
   * No layout is done for the associated element. This is used to mark parts of a diagram to
   * avoid their inclusion in the layout graph, or to mark parts of the layout graph to
   * prevent layout engines from processing them. If you wish to exclude the contents of a
   * compound node from automatic layout, while the node itself is still considered on its own
   * layer, use the 'Fixed Layout' algorithm for that node.
   */
  public final static IProperty<Boolean> NO_LAYOUT = CoreOptions.NO_LAYOUT;
  
  /**
   * Defines constraints of the position of the ports of a node.
   */
  public final static IProperty<PortConstraints> PORT_CONSTRAINTS = CoreOptions.PORT_CONSTRAINTS;
  
  /**
   * The side of a node on which a port is situated. This option must be set if 'Port
   * Constraints' is set to FIXED_SIDE or FIXED_ORDER and no specific positions are given
   * for the ports.
   */
  public final static IProperty<PortSide> PORT_SIDE = CoreOptions.PORT_SIDE;
  
  /**
   * Whether additional debug information shall be generated.
   */
  public final static IProperty<Boolean> DEBUG_MODE = CoreOptions.DEBUG_MODE;
  
  /**
   * Alignment of the selected node relative to other nodes,
   * the exact meaning depends on the used algorithm.
   */
  public final static IProperty<Alignment> ALIGNMENT = CoreOptions.ALIGNMENT;
  
  /**
   * Whether the whole hierarchy shall be layouted. If this option is not set, each hierarchy
   * level of the graph is processed independently, possibly by different layout algorithms,
   * beginning with the lowest level. If it is set, the algorithm is responsible to process
   * all hierarchy levels that are contained in the associated parent node.
   */
  @Deprecated
  public final static IProperty<Boolean> LAYOUT_HIERARCHY = CoreOptions.LAYOUT_HIERARCHY;
  
  /**
   * Determines whether the descendants should be layouted separately or together with their
   * parents. If the root node is set to inherit (or not set at all), the option is assumed
   * as SEPARATE_CHILDREN.
   */
  public final static IProperty<HierarchyHandling> HIERARCHY_HANDLING = CoreOptions.HIERARCHY_HANDLING;
  
  /**
   * Default value for {@link #SEPARATE_CONNECTED_COMPONENTS} with algorithm "ELK Layered".
   */
  private final static boolean SEPARATE_CONNECTED_COMPONENTS_DEFAULT = true;
  
  /**
   * Whether each connected component should be processed separately.
   */
  public final static IProperty<Boolean> SEPARATE_CONNECTED_COMPONENTS = new Property<Boolean>(
                                CoreOptions.SEPARATE_CONNECTED_COMPONENTS,
                                SEPARATE_CONNECTED_COMPONENTS_DEFAULT);
  
  /**
   * Whether this node allows to route self loops inside of it instead of around it. If set to true,
   * this will make the node a compound node if it isn't already, and will require the layout algorithm
   * to support compound nodes with hierarchical ports.
   */
  public final static IProperty<Boolean> INSIDE_SELF_LOOPS_ACTIVATE = CoreOptions.INSIDE_SELF_LOOPS_ACTIVATE;
  
  /**
   * Whether a self loop should be routed inside a node instead of around that node.
   */
  public final static IProperty<Boolean> INSIDE_SELF_LOOPS_YO = CoreOptions.INSIDE_SELF_LOOPS_YO;
  
  /**
   * Constraints for determining node sizes. Each member of the set specifies something that
   * should be taken into account when calculating node sizes. The empty set corresponds to
   * node sizes being fixed.
   */
  public final static IProperty<EnumSet<SizeConstraint>> NODE_SIZE_CONSTRAINTS = CoreOptions.NODE_SIZE_CONSTRAINTS;
  
  /**
   * Options modifying the behavior of the size constraints set on a node. Each member of the
   * set specifies something that should be taken into account when calculating node sizes.
   * The empty set corresponds to no further modifications.
   */
  public final static IProperty<EnumSet<SizeOptions>> NODE_SIZE_OPTIONS = CoreOptions.NODE_SIZE_OPTIONS;
  
  /**
   * Default value for {@link #DIRECTION} with algorithm "ELK Layered".
   */
  private final static Direction DIRECTION_DEFAULT = Direction.UNDEFINED;
  
  /**
   * Overall direction of edges: horizontal (right / left) or
   * vertical (down / up).
   */
  public final static IProperty<Direction> DIRECTION = new Property<Direction>(
                                CoreOptions.DIRECTION,
                                DIRECTION_DEFAULT);
  
  /**
   * Hints for where node labels are to be placed; if empty, the node label's position is not
   * modified.
   */
  public final static IProperty<EnumSet<NodeLabelPlacement>> NODE_LABELS_PLACEMENT = CoreOptions.NODE_LABELS_PLACEMENT;
  
  /**
   * Decides on a placement method for port labels.
   */
  public final static IProperty<PortLabelPlacement> PORT_LABELS_PLACEMENT = CoreOptions.PORT_LABELS_PLACEMENT;
  
  /**
   * Default value for {@link #PORT_ALIGNMENT_BASIC} with algorithm "ELK Layered".
   */
  private final static PortAlignment PORT_ALIGNMENT_BASIC_DEFAULT = PortAlignment.JUSTIFIED;
  
  /**
   * Defines the default port distribution for a node. May be overridden for each side individually.
   */
  public final static IProperty<PortAlignment> PORT_ALIGNMENT_BASIC = new Property<PortAlignment>(
                                CoreOptions.PORT_ALIGNMENT_BASIC,
                                PORT_ALIGNMENT_BASIC_DEFAULT);
  
  /**
   * Defines how ports on the northern side are placed, overriding the node's general port alignment.
   */
  public final static IProperty<PortAlignment> PORT_ALIGNMENT_NORTH = CoreOptions.PORT_ALIGNMENT_NORTH;
  
  /**
   * Defines how ports on the southern side are placed, overriding the node's general port alignment.
   */
  public final static IProperty<PortAlignment> PORT_ALIGNMENT_SOUTH = CoreOptions.PORT_ALIGNMENT_SOUTH;
  
  /**
   * Defines how ports on the western side are placed, overriding the node's general port alignment.
   */
  public final static IProperty<PortAlignment> PORT_ALIGNMENT_WEST = CoreOptions.PORT_ALIGNMENT_WEST;
  
  /**
   * Defines how ports on the eastern side are placed, overriding the node's general port alignment.
   */
  public final static IProperty<PortAlignment> PORT_ALIGNMENT_EAST = CoreOptions.PORT_ALIGNMENT_EAST;
  
  /**
   * Adds bend points even if an edge does not change direction. If true, each long edge dummy
   * will contribute a bend point to its edges and hierarchy-crossing edges will always get a
   * bend point where they cross hierarchy boundaries. By default, bend points are only added
   * where an edge changes direction.
   */
  public final static IProperty<Boolean> UNNECESSARY_BENDPOINTS = LayeredMetaDataProvider.UNNECESSARY_BENDPOINTS;
  
  /**
   * Strategy for node layering.
   */
  public final static IProperty<LayeringStrategy> LAYERING_STRATEGY = LayeredMetaDataProvider.LAYERING_STRATEGY;
  
  /**
   * Reduces number of dummy nodes after layering phase (if possible).
   */
  public final static IProperty<NodePromotionStrategy> LAYERING_NODE_PROMOTION_STRATEGY = LayeredMetaDataProvider.LAYERING_NODE_PROMOTION_STRATEGY;
  
  /**
   * How much effort should be spent to produce a nice layout.
   */
  public final static IProperty<Integer> THOROUGHNESS = LayeredMetaDataProvider.THOROUGHNESS;
  
  /**
   * Determines a constraint on the placement of the node regarding the layering.
   */
  public final static IProperty<LayerConstraint> LAYERING_LAYER_CONSTRAINT = LayeredMetaDataProvider.LAYERING_LAYER_CONSTRAINT;
  
  /**
   * Strategy for cycle breaking. Cycle breaking looks for cycles in the graph and determines
   * which edges to reverse to break the cycles. Reversed edges will end up pointing to the
   * opposite direction of regular edges (that is, reversed edges will point left if edges
   * usually point right).
   */
  public final static IProperty<CycleBreakingStrategy> CYCLE_BREAKING_STRATEGY = LayeredMetaDataProvider.CYCLE_BREAKING_STRATEGY;
  
  /**
   * Strategy for crossing minimization.
   */
  public final static IProperty<CrossingMinimizationStrategy> CROSSING_MINIMIZATION_STRATEGY = LayeredMetaDataProvider.CROSSING_MINIMIZATION_STRATEGY;
  
  /**
   * By default it is decided automatically if the greedy switch is activated or not.
   * The decision is based on whether the size of the input graph (without dummy nodes)
   * is smaller than the value of this option. A '0' enforces the activation.
   */
  public final static IProperty<Integer> CROSSING_MINIMIZATION_GREEDY_SWITCH_ACTIVATION_THRESHOLD = LayeredMetaDataProvider.CROSSING_MINIMIZATION_GREEDY_SWITCH_ACTIVATION_THRESHOLD;
  
  /**
   * Greedy Switch strategy for crossing minimization. The greedy switch heuristic is executed
   * after the regular layer sweep as a post-processor.
   */
  public final static IProperty<GreedySwitchType> CROSSING_MINIMIZATION_GREEDY_SWITCH_TYPE = LayeredMetaDataProvider.CROSSING_MINIMIZATION_GREEDY_SWITCH_TYPE;
  
  /**
   * Preserves the order of nodes within a layer but still minimizes
   * crossings between edges connecting long edge dummies.
   * Requires a crossing minimization strategy that is able to
   * process 'in-layer' constraints.
   */
  public final static IProperty<Boolean> CROSSING_MINIMIZATION_SEMI_INTERACTIVE = LayeredMetaDataProvider.CROSSING_MINIMIZATION_SEMI_INTERACTIVE;
  
  /**
   * Edges that have no ports are merged so they touch the connected nodes at the same points.
   * When this option is disabled, one port is created for each edge directly connected to a
   * node. When it is enabled, all such incoming edges share an input port, and all outgoing
   * edges share an output port.
   */
  public final static IProperty<Boolean> MERGE_EDGES = LayeredMetaDataProvider.MERGE_EDGES;
  
  /**
   * If hierarchical layout is active, hierarchy-crossing edges use as few hierarchical ports
   * as possible. They are broken by the algorithm, with hierarchical ports inserted as
   * required. Usually, one such port is created for each edge at each hierarchy crossing point.
   * With this option set to true, we try to create as few hierarchical ports as possible in
   * the process. In particular, all edges that form a hyperedge can share a port.
   */
  public final static IProperty<Boolean> MERGE_HIERARCHY_EDGES = LayeredMetaDataProvider.MERGE_HIERARCHY_EDGES;
  
  /**
   * Determines which point of a node is considered by interactive layout phases.
   */
  public final static IProperty<InteractiveReferencePoint> INTERACTIVE_REFERENCE_POINT = LayeredMetaDataProvider.INTERACTIVE_REFERENCE_POINT;
  
  /**
   * Strategy for node placement.
   */
  public final static IProperty<NodePlacementStrategy> NODE_PLACEMENT_STRATEGY = LayeredMetaDataProvider.NODE_PLACEMENT_STRATEGY;
  
  /**
   * Tells the BK node placer to use a certain alignment (out of its four) instead of the
   * one producing the smallest height, or the combination of all four.
   */
  public final static IProperty<FixedAlignment> NODE_PLACEMENT_BK_FIXED_ALIGNMENT = LayeredMetaDataProvider.NODE_PLACEMENT_BK_FIXED_ALIGNMENT;
  
  /**
   * Method to decide on edge label sides.
   */
  public final static IProperty<EdgeLabelSideSelection> EDGE_LABEL_SIDE_SELECTION = LayeredMetaDataProvider.EDGE_LABEL_SIDE_SELECTION;
  
  /**
   * Whether feedback edges should be highlighted by routing around the nodes.
   */
  public final static IProperty<Boolean> FEEDBACK_EDGES = LayeredMetaDataProvider.FEEDBACK_EDGES;
  
  /**
   * Strategy to distribute wide nodes over multiple layers.
   */
  public final static IProperty<WideNodesStrategy> LAYERING_WIDE_NODES_ON_MULTIPLE_LAYERS = LayeredMetaDataProvider.LAYERING_WIDE_NODES_ON_MULTIPLE_LAYERS;
  
  /**
   * Dampens the movement of nodes to keep the diagram from getting too large.
   */
  public final static IProperty<Double> NODE_PLACEMENT_LINEAR_SEGMENTS_DEFLECTION_DAMPENING = LayeredMetaDataProvider.NODE_PLACEMENT_LINEAR_SEGMENTS_DEFLECTION_DAMPENING;
  
  /**
   * null
   */
  public final static IProperty<SelfLoopPlacement> EDGE_ROUTING_SELF_LOOP_PLACEMENT = LayeredMetaDataProvider.EDGE_ROUTING_SELF_LOOP_PLACEMENT;
  
  /**
   * Specifies how the content of compound nodes is to be aligned, e.g. top-left.
   */
  public final static IProperty<EnumSet<ContentAlignment>> CONTENT_ALIGNMENT = LayeredMetaDataProvider.CONTENT_ALIGNMENT;
  
  /**
   * Specifies whether the Brandes Koepf node placer tries to increase the number of straight edges
   * at the expense of diagram size.
   * There is a subtle difference to the 'favorStraightEdges' option, which decides whether
   * a balanced placement of the nodes is desired, or not. In bk terms this means combining the four
   * alignments into a single balanced one, or not. This option on the other hand tries to straighten
   * additional edges during the creation of each of the four alignments.
   */
  public final static IProperty<EdgeStraighteningStrategy> NODE_PLACEMENT_BK_EDGE_STRAIGHTENING = LayeredMetaDataProvider.NODE_PLACEMENT_BK_EDGE_STRAIGHTENING;
  
  /**
   * Specifies whether and how post-process compaction is applied.
   */
  public final static IProperty<GraphCompactionStrategy> COMPACTION_POST_COMPACTION_STRATEGY = LayeredMetaDataProvider.COMPACTION_POST_COMPACTION_STRATEGY;
  
  /**
   * Specifies whether and how post-process compaction is applied.
   */
  public final static IProperty<ConstraintCalculationStrategy> COMPACTION_POST_COMPACTION_CONSTRAINTS = LayeredMetaDataProvider.COMPACTION_POST_COMPACTION_CONSTRAINTS;
  
  /**
   * Tries to further compact components (disconnected sub-graphs).
   */
  public final static IProperty<Boolean> COMPACTION_CONNECTED_COMPONENTS = LayeredMetaDataProvider.COMPACTION_CONNECTED_COMPONENTS;
  
  /**
   * Makes room around high degree nodes to place leafs and trees.
   */
  public final static IProperty<Boolean> HIGH_DEGREE_NODES_TREATMENT = LayeredMetaDataProvider.HIGH_DEGREE_NODES_TREATMENT;
  
  /**
   * Whether a node is considered to have a high degree.
   */
  public final static IProperty<Integer> HIGH_DEGREE_NODES_THRESHOLD = LayeredMetaDataProvider.HIGH_DEGREE_NODES_THRESHOLD;
  
  /**
   * Maximum height of a subtree connected to a high degree node to be moved to separate layers.
   */
  public final static IProperty<Integer> HIGH_DEGREE_NODES_TREE_HEIGHT = LayeredMetaDataProvider.HIGH_DEGREE_NODES_TREE_HEIGHT;
  
  /**
   * The minimal size to which a node can be reduced.
   */
  public final static IProperty<KVector> NODE_SIZE_MINIMUM = CoreOptions.NODE_SIZE_MINIMUM;
  
  /**
   * null
   */
  @Deprecated
  public final static IProperty<Double> NODE_SIZE_MIN_WIDTH = CoreOptions.NODE_SIZE_MIN_WIDTH;
  
  /**
   * null
   */
  @Deprecated
  public final static IProperty<Double> NODE_SIZE_MIN_HEIGHT = CoreOptions.NODE_SIZE_MIN_HEIGHT;
  
  /**
   * This option is not used as option, but as output of the layout algorithms. It is
   * attached to edges and determines the points where junction symbols should be drawn in
   * order to represent hyperedges with orthogonal routing. Whether such points are computed
   * depends on the chosen layout algorithm and edge routing style. The points are put into
   * the vector chain with no specific order.
   */
  public final static IProperty<KVectorChain> JUNCTION_POINTS = CoreOptions.JUNCTION_POINTS;
  
  /**
   * The thickness of an edge. This is a hint on the line width used to draw an edge, possibly
   * requiring more space to be reserved for it.
   */
  public final static IProperty<Double> EDGE_THICKNESS = CoreOptions.EDGE_THICKNESS;
  
  /**
   * Gives a hint on where to put edge labels.
   */
  public final static IProperty<EdgeLabelPlacement> EDGE_LABELS_PLACEMENT = CoreOptions.EDGE_LABELS_PLACEMENT;
  
  /**
   * How likely it is to use cross-hierarchy (1) vs bottom-up (-1).
   */
  public final static IProperty<Double> CROSSING_MINIMIZATION_HIERARCHICAL_SWEEPINESS = LayeredMetaDataProvider.CROSSING_MINIMIZATION_HIERARCHICAL_SWEEPINESS;
  
  /**
   * The index of a port in the fixed order around a node. The order is assumed as clockwise,
   * starting with the leftmost port on the top side. This option must be set if 'Port
   * Constraints' is set to FIXED_ORDER and no specific positions are given for the ports.
   * Additionally, the option 'Port Side' must be defined in this case.
   */
  public final static IProperty<Integer> PORT_INDEX = CoreOptions.PORT_INDEX;
  
  /**
   * Whether the node should be regarded as a comment box instead of a regular node. In that
   * case its placement should be similar to how labels are handled. Any edges incident to a
   * comment box specify to which graph elements the comment is related.
   */
  public final static IProperty<Boolean> COMMENT_BOX = CoreOptions.COMMENT_BOX;
  
  /**
   * Whether the node should be handled as a hypernode.
   */
  public final static IProperty<Boolean> HYPERNODE = CoreOptions.HYPERNODE;
  
  /**
   * The offset to the port position where connections shall be attached.
   */
  public final static IProperty<KVector> PORT_ANCHOR = CoreOptions.PORT_ANCHOR;
  
  /**
   * Whether to activate partitioned layout.
   */
  public final static IProperty<Boolean> PARTITIONING_ACTIVATE = CoreOptions.PARTITIONING_ACTIVATE;
  
  /**
   * Partition to which the node belongs to. If 'layoutPartitions' is true,
   * all nodes are expected to have a partition.
   */
  public final static IProperty<Integer> PARTITIONING_PARTITION = CoreOptions.PARTITIONING_PARTITION;
  
  /**
   * Whether wide nodes should be distributed to several layers.
   */
  @Deprecated
  public final static IProperty<Boolean> LAYERING_DISTRIBUTE_NODES = LayeredMetaDataProvider.LAYERING_DISTRIBUTE_NODES;
  
  /**
   * Defines a loose upper bound on the width of the MinWidth layerer.
   * If set to '-1' multiple values are tested and the best result is selected.
   */
  public final static IProperty<Integer> LAYERING_MIN_WIDTH_UPPER_BOUND_ON_WIDTH = LayeredMetaDataProvider.LAYERING_MIN_WIDTH_UPPER_BOUND_ON_WIDTH;
  
  /**
   * Multiplied with Upper Bound On Width for defining an upper bound on the width of layers which
   * haven't been determined yet, but whose maximum width had been (roughly) estimated by the MinWidth
   * algorithm. Compensates for too high estimations.
   * If set to '-1' multiple values are tested and the best result is selected.
   */
  public final static IProperty<Integer> LAYERING_MIN_WIDTH_UPPER_LAYER_ESTIMATION_SCALING_FACTOR = LayeredMetaDataProvider.LAYERING_MIN_WIDTH_UPPER_LAYER_ESTIMATION_SCALING_FACTOR;
  
  /**
   * Whether long sausages should be folded up nice and tight.
   */
  public final static IProperty<Boolean> SAUSAGE_FOLDING = LayeredMetaDataProvider.SAUSAGE_FOLDING;
  
  /**
   * The position of a node, port, or label. This is used by the 'Fixed Layout' algorithm to
   * specify a pre-defined position.
   */
  public final static IProperty<KVector> POSITION = CoreOptions.POSITION;
  
  /**
   * Specifies that this port can either be placed on the north side of a node or on the south
   * side (if port constraints permit)
   */
  public final static IProperty<Boolean> NORTH_OR_SOUTH_PORT = LayeredMetaDataProvider.NORTH_OR_SOUTH_PORT;
  
  /**
   * Limits the number of iterations for node promotion.
   */
  public final static IProperty<Integer> LAYERING_NODE_PROMOTION_MAX_ITERATIONS = LayeredMetaDataProvider.LAYERING_NODE_PROMOTION_MAX_ITERATIONS;
  
  /**
   * Determines in which layer center labels of long edges should be placed.
   */
  public final static IProperty<EdgeLabelPlacementStrategy> EDGE_CENTER_LABEL_PLACEMENT_STRATEGY = LayeredMetaDataProvider.EDGE_CENTER_LABEL_PLACEMENT_STRATEGY;
  
  /**
   * Margins define additional space around the actual bounds of a graph element. For instance,
   * ports or labels being placed on the outside of a node's border might introduce such a
   * margin. The margin is used to guarantee non-overlap of other graph elements with those
   * ports or labels.
   */
  public final static IProperty<ElkMargin> MARGINS = CoreOptions.MARGINS;
  
  /**
   * Use less spline control points at the start and end of an edge. Might lead to crossings edge/node overlap.
   */
  public final static IProperty<Boolean> EDGE_ROUTING_SLOPPY_SPLINE_ROUTING = LayeredMetaDataProvider.EDGE_ROUTING_SLOPPY_SPLINE_ROUTING;
  
  /**
   * Spacing factor for routing area between layers when using sloppy spline routing.
   */
  public final static IProperty<Double> EDGE_ROUTING_SLOPPY_SPLINE_LAYER_SPACING = LayeredMetaDataProvider.EDGE_ROUTING_SLOPPY_SPLINE_LAYER_SPACING;
  
  /**
   * The maximum number of nodes allowed per layer.
   */
  public final static IProperty<Integer> LAYERING_COFFMAN_GRAHAM_LAYER_BOUND = LayeredMetaDataProvider.LAYERING_COFFMAN_GRAHAM_LAYER_BOUND;
  
  /**
   * Favor straight edges over a balanced node placement.
   * The default behavior is determined automatically based on the used 'edgeRouting'.
   * For an orthogonal style it is set to true, for all other styles to false.
   */
  public final static IProperty<Boolean> NODE_PLACEMENT_FAVOR_STRAIGHT_EDGES = LayeredMetaDataProvider.NODE_PLACEMENT_FAVOR_STRAIGHT_EDGES;
  
  public void apply(final org.eclipse.elk.core.data.ILayoutMetaDataProvider.Registry registry) {
    registry.register(new org.eclipse.elk.core.data.LayoutAlgorithmData(
        "org.eclipse.elk.layered",
        "ELK Layered",
        "Layer-based algorithm provided by the Eclipse Layout Kernel. Arranges as many edges as possible into one direction by placing nodes into subsequent layers. This implementation supports different routing styles (straight, orthogonal, splines); if orthogonal routing is selected, arbitrary port constraints are respected, thus enabling the layout of block diagrams such as actor-oriented models or circuit schematics. Furthermore, full layout of compound graphs with cross-hierarchy edges is supported when the respective option is activated on the top level.",
        new org.eclipse.elk.core.util.AlgorithmFactory(LayeredLayoutProvider.class, ""),
        "org.eclipse.elk.layered",
        null,
        "images/layered.png",
        EnumSet.of(org.eclipse.elk.core.options.GraphFeature.SELF_LOOPS, org.eclipse.elk.core.options.GraphFeature.INSIDE_SELF_LOOPS, org.eclipse.elk.core.options.GraphFeature.MULTI_EDGES, org.eclipse.elk.core.options.GraphFeature.EDGE_LABELS, org.eclipse.elk.core.options.GraphFeature.PORTS, org.eclipse.elk.core.options.GraphFeature.COMPOUND, org.eclipse.elk.core.options.GraphFeature.CLUSTERS)
    ));
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.spacing.componentComponent",
        SPACING_COMPONENT_COMPONENT.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.spacing.edgeEdge",
        SPACING_EDGE_EDGE.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.spacing.edgeLabel",
        SPACING_EDGE_LABEL.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.spacing.edgeNode",
        SPACING_EDGE_NODE.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.spacing.labelLabel",
        SPACING_LABEL_LABEL.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.spacing.labelPort",
        SPACING_LABEL_PORT.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.spacing.labelNode",
        SPACING_LABEL_NODE.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.spacing.nodeNode",
        SPACING_NODE_NODE_DEFAULT
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.spacing.portPort",
        SPACING_PORT_PORT.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.spacing.individualOverride",
        SPACING_INDIVIDUAL_OVERRIDE.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.spacing.edgeEdgeBetweenLayers",
        SPACING_EDGE_EDGE_BETWEEN_LAYERS.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.spacing.edgeNodeBetweenLayers",
        SPACING_EDGE_NODE_BETWEEN_LAYERS.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.spacing.nodeNodeBetweenLayers",
        SPACING_NODE_NODE_BETWEEN_LAYERS.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.priority",
        PRIORITY_DEFAULT
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.priority.direction",
        PRIORITY_DIRECTION.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.priority.shortness",
        PRIORITY_SHORTNESS.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.priority.straightness",
        PRIORITY_STRAIGHTNESS.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.padding",
        PADDING_DEFAULT
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.edgeRouting",
        EDGE_ROUTING_DEFAULT
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.port.borderOffset",
        PORT_BORDER_OFFSET_DEFAULT
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.randomSeed",
        RANDOM_SEED_DEFAULT
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.aspectRatio",
        ASPECT_RATIO_DEFAULT
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.noLayout",
        NO_LAYOUT.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.portConstraints",
        PORT_CONSTRAINTS.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.port.side",
        PORT_SIDE.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.debugMode",
        DEBUG_MODE.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.alignment",
        ALIGNMENT.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layoutHierarchy",
        LAYOUT_HIERARCHY.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.hierarchyHandling",
        HIERARCHY_HANDLING.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.separateConnectedComponents",
        SEPARATE_CONNECTED_COMPONENTS_DEFAULT
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.insideSelfLoops.activate",
        INSIDE_SELF_LOOPS_ACTIVATE.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.insideSelfLoops.yo",
        INSIDE_SELF_LOOPS_YO.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.nodeSize.constraints",
        NODE_SIZE_CONSTRAINTS.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.nodeSize.options",
        NODE_SIZE_OPTIONS.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.direction",
        DIRECTION_DEFAULT
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.nodeLabels.placement",
        NODE_LABELS_PLACEMENT.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.portLabels.placement",
        PORT_LABELS_PLACEMENT.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.portAlignment.basic",
        PORT_ALIGNMENT_BASIC_DEFAULT
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.portAlignment.north",
        PORT_ALIGNMENT_NORTH.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.portAlignment.south",
        PORT_ALIGNMENT_SOUTH.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.portAlignment.west",
        PORT_ALIGNMENT_WEST.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.portAlignment.east",
        PORT_ALIGNMENT_EAST.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.unnecessaryBendpoints",
        UNNECESSARY_BENDPOINTS.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.layering.strategy",
        LAYERING_STRATEGY.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.layering.nodePromotion.strategy",
        LAYERING_NODE_PROMOTION_STRATEGY.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.thoroughness",
        THOROUGHNESS.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.layering.layerConstraint",
        LAYERING_LAYER_CONSTRAINT.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.cycleBreaking.strategy",
        CYCLE_BREAKING_STRATEGY.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.crossingMinimization.strategy",
        CROSSING_MINIMIZATION_STRATEGY.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.crossingMinimization.greedySwitch.activationThreshold",
        CROSSING_MINIMIZATION_GREEDY_SWITCH_ACTIVATION_THRESHOLD.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.crossingMinimization.greedySwitch.type",
        CROSSING_MINIMIZATION_GREEDY_SWITCH_TYPE.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.crossingMinimization.semiInteractive",
        CROSSING_MINIMIZATION_SEMI_INTERACTIVE.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.mergeEdges",
        MERGE_EDGES.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.mergeHierarchyEdges",
        MERGE_HIERARCHY_EDGES.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.interactiveReferencePoint",
        INTERACTIVE_REFERENCE_POINT.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.nodePlacement.strategy",
        NODE_PLACEMENT_STRATEGY.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.nodePlacement.bk.fixedAlignment",
        NODE_PLACEMENT_BK_FIXED_ALIGNMENT.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.edgeLabelSideSelection",
        EDGE_LABEL_SIDE_SELECTION.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.feedbackEdges",
        FEEDBACK_EDGES.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.layering.wideNodesOnMultipleLayers",
        LAYERING_WIDE_NODES_ON_MULTIPLE_LAYERS.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.nodePlacement.linearSegments.deflectionDampening",
        NODE_PLACEMENT_LINEAR_SEGMENTS_DEFLECTION_DAMPENING.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.edgeRouting.selfLoopPlacement",
        EDGE_ROUTING_SELF_LOOP_PLACEMENT.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.contentAlignment",
        CONTENT_ALIGNMENT.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.nodePlacement.bk.edgeStraightening",
        NODE_PLACEMENT_BK_EDGE_STRAIGHTENING.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.compaction.postCompaction.strategy",
        COMPACTION_POST_COMPACTION_STRATEGY.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.compaction.postCompaction.constraints",
        COMPACTION_POST_COMPACTION_CONSTRAINTS.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.compaction.connectedComponents",
        COMPACTION_CONNECTED_COMPONENTS.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.highDegreeNodes.treatment",
        HIGH_DEGREE_NODES_TREATMENT.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.highDegreeNodes.threshold",
        HIGH_DEGREE_NODES_THRESHOLD.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.highDegreeNodes.treeHeight",
        HIGH_DEGREE_NODES_TREE_HEIGHT.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.nodeSize.minimum",
        NODE_SIZE_MINIMUM.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.nodeSize.minWidth",
        NODE_SIZE_MIN_WIDTH.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.nodeSize.minHeight",
        NODE_SIZE_MIN_HEIGHT.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.junctionPoints",
        JUNCTION_POINTS.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.edge.thickness",
        EDGE_THICKNESS.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.edgeLabels.placement",
        EDGE_LABELS_PLACEMENT.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.crossingMinimization.hierarchicalSweepiness",
        CROSSING_MINIMIZATION_HIERARCHICAL_SWEEPINESS.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.port.index",
        PORT_INDEX.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.commentBox",
        COMMENT_BOX.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.hypernode",
        HYPERNODE.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.port.anchor",
        PORT_ANCHOR.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.partitioning.activate",
        PARTITIONING_ACTIVATE.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.partitioning.partition",
        PARTITIONING_PARTITION.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.layering.distributeNodes",
        LAYERING_DISTRIBUTE_NODES.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.layering.minWidth.upperBoundOnWidth",
        LAYERING_MIN_WIDTH_UPPER_BOUND_ON_WIDTH.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.layering.minWidth.upperLayerEstimationScalingFactor",
        LAYERING_MIN_WIDTH_UPPER_LAYER_ESTIMATION_SCALING_FACTOR.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.sausageFolding",
        SAUSAGE_FOLDING.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.position",
        POSITION.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.northOrSouthPort",
        NORTH_OR_SOUTH_PORT.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.layering.nodePromotion.maxIterations",
        LAYERING_NODE_PROMOTION_MAX_ITERATIONS.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.edgeCenterLabelPlacementStrategy",
        EDGE_CENTER_LABEL_PLACEMENT_STRATEGY.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.margins",
        MARGINS.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.edgeRouting.sloppySplineRouting",
        EDGE_ROUTING_SLOPPY_SPLINE_ROUTING.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.edgeRouting.sloppySplineLayerSpacing",
        EDGE_ROUTING_SLOPPY_SPLINE_LAYER_SPACING.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.layering.coffmanGraham.layerBound",
        LAYERING_COFFMAN_GRAHAM_LAYER_BOUND.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.layered",
        "org.eclipse.elk.layered.nodePlacement.favorStraightEdges",
        NODE_PLACEMENT_FAVOR_STRAIGHT_EDGES.getDefault()
    );
  }
}
