/**
 * Copyright (c) 2018, 2020 Kiel University and others.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.elk.alg.rectpacking.options;

import java.util.EnumSet;
import org.eclipse.elk.alg.rectpacking.options.OptimizationGoal;
import org.eclipse.elk.core.data.ILayoutMetaDataProvider;
import org.eclipse.elk.core.data.LayoutOptionData;
import org.eclipse.elk.graph.properties.IProperty;
import org.eclipse.elk.graph.properties.Property;

@SuppressWarnings("all")
public class RectPackingMetaDataProvider implements ILayoutMetaDataProvider {
  /**
   * Default value for {@link #OPTIMIZATION_GOAL}.
   */
  private static final OptimizationGoal OPTIMIZATION_GOAL_DEFAULT = OptimizationGoal.MAX_SCALE_DRIVEN;
  
  /**
   * Optimization goal for approximation of the bounding box given by the first iteration. Determines whether layout is
   * sorted by the maximum scaling, aspect ratio, or area. Depending on the strategy the aspect ratio might be nearly ignored.
   */
  public static final IProperty<OptimizationGoal> OPTIMIZATION_GOAL = new Property<OptimizationGoal>(
            "org.eclipse.elk.rectpacking.optimizationGoal",
            OPTIMIZATION_GOAL_DEFAULT,
            null,
            null);
  
  /**
   * Default value for {@link #LAST_PLACE_SHIFT}.
   */
  private static final boolean LAST_PLACE_SHIFT_DEFAULT = true;
  
  /**
   * When placing a rectangle behind or below the last placed rectangle in the first iteration, it is sometimes
   * possible to shift the rectangle further to the left or right, resulting in less whitespace. True (default)
   * enables the shift and false disables it. Disabling the shift produces a greater approximated area by the first
   * iteration and a layout, when using ONLY the first iteration (default not the case), where it is sometimes
   * impossible to implement a size transformation of rectangles that will fill the bounding box and eliminate
   * empty spaces.
   */
  public static final IProperty<Boolean> LAST_PLACE_SHIFT = new Property<Boolean>(
            "org.eclipse.elk.rectpacking.lastPlaceShift",
            LAST_PLACE_SHIFT_DEFAULT,
            null,
            null);
  
  /**
   * Default value for {@link #CURRENT_POSITION}.
   */
  private static final int CURRENT_POSITION_DEFAULT = (-1);
  
  /**
   * Lower bound value for {@link #CURRENT_POSITION}.
   */
  private static final Comparable<? super Integer> CURRENT_POSITION_LOWER_BOUND = Integer.valueOf((-1));
  
  /**
   * The rectangles are ordered. Normally according to their definition the the model.
   * This option specifies the current position of a node.
   */
  public static final IProperty<Integer> CURRENT_POSITION = new Property<Integer>(
            "org.eclipse.elk.rectpacking.currentPosition",
            CURRENT_POSITION_DEFAULT,
            CURRENT_POSITION_LOWER_BOUND,
            null);
  
  /**
   * Default value for {@link #DESIRED_POSITION}.
   */
  private static final int DESIRED_POSITION_DEFAULT = (-1);
  
  /**
   * Lower bound value for {@link #DESIRED_POSITION}.
   */
  private static final Comparable<? super Integer> DESIRED_POSITION_LOWER_BOUND = Integer.valueOf((-1));
  
  /**
   * The rectangles are ordered. Normally according to their definition the the model.
   * This option allows to specify a desired position that has preference over the original position.
   */
  public static final IProperty<Integer> DESIRED_POSITION = new Property<Integer>(
            "org.eclipse.elk.rectpacking.desiredPosition",
            DESIRED_POSITION_DEFAULT,
            DESIRED_POSITION_LOWER_BOUND,
            null);
  
  /**
   * Default value for {@link #ONLY_FIRST_ITERATION}.
   */
  private static final boolean ONLY_FIRST_ITERATION_DEFAULT = false;
  
  /**
   * If enabled only the width approximation step is executed and the nodes are placed accordingly.
   * The nodes are layouted according to the packingStrategy.
   * If set to true not expansion of nodes is taking place.
   */
  public static final IProperty<Boolean> ONLY_FIRST_ITERATION = new Property<Boolean>(
            "org.eclipse.elk.rectpacking.onlyFirstIteration",
            ONLY_FIRST_ITERATION_DEFAULT,
            null,
            null);
  
  /**
   * Default value for {@link #ROW_COMPACTION}.
   */
  private static final boolean ROW_COMPACTION_DEFAULT = true;
  
  /**
   * Enables compaction. Compacts blocks if they do not use the full height of the row.
   * This option allows to have a smaller drawing.
   * If this option is disabled all nodes are placed next to each other in rows.
   */
  public static final IProperty<Boolean> ROW_COMPACTION = new Property<Boolean>(
            "org.eclipse.elk.rectpacking.rowCompaction",
            ROW_COMPACTION_DEFAULT,
            null,
            null);
  
  /**
   * Default value for {@link #EXPAND_TO_ASPECT_RATIO}.
   */
  private static final boolean EXPAND_TO_ASPECT_RATIO_DEFAULT = false;
  
  /**
   * Expands nodes if expandNodes is true to fit the aspect ratio instead of only in their bounds.
   * The option is only useful if the used packingStrategy is ASPECT_RATIO_DRIVEN, otherwise this may result
   * in unreasonable ndoe expansion.
   */
  public static final IProperty<Boolean> EXPAND_TO_ASPECT_RATIO = new Property<Boolean>(
            "org.eclipse.elk.rectpacking.expandToAspectRatio",
            EXPAND_TO_ASPECT_RATIO_DEFAULT,
            null,
            null);
  
  /**
   * Default value for {@link #TARGET_WIDTH}.
   */
  private static final double TARGET_WIDTH_DEFAULT = (-1);
  
  /**
   * Option to place the rectangles in the given target width instead of approximating the width using the desired
   * aspect ratio.
   * The padding is not included in this. Meaning a drawing will have width of targetwidth +
   * horizontal padding.
   */
  public static final IProperty<Double> TARGET_WIDTH = new Property<Double>(
            "org.eclipse.elk.rectpacking.targetWidth",
            TARGET_WIDTH_DEFAULT,
            null,
            null);
  
  public void apply(final org.eclipse.elk.core.data.ILayoutMetaDataProvider.Registry registry) {
    registry.register(new LayoutOptionData.Builder()
        .id("org.eclipse.elk.rectpacking.optimizationGoal")
        .group("")
        .name("Optimization Goal")
        .description("Optimization goal for approximation of the bounding box given by the first iteration. Determines whether layout is sorted by the maximum scaling, aspect ratio, or area. Depending on the strategy the aspect ratio might be nearly ignored.")
        .defaultValue(OPTIMIZATION_GOAL_DEFAULT)
        .type(LayoutOptionData.Type.ENUM)
        .optionClass(OptimizationGoal.class)
        .targets(EnumSet.of(LayoutOptionData.Target.NODES))
        .visibility(LayoutOptionData.Visibility.ADVANCED)
        .create()
    );
    registry.register(new LayoutOptionData.Builder()
        .id("org.eclipse.elk.rectpacking.lastPlaceShift")
        .group("")
        .name("Shift Last Placed.")
        .description("When placing a rectangle behind or below the last placed rectangle in the first iteration, it is sometimes possible to shift the rectangle further to the left or right, resulting in less whitespace. True (default) enables the shift and false disables it. Disabling the shift produces a greater approximated area by the first iteration and a layout, when using ONLY the first iteration (default not the case), where it is sometimes impossible to implement a size transformation of rectangles that will fill the bounding box and eliminate empty spaces.")
        .defaultValue(LAST_PLACE_SHIFT_DEFAULT)
        .type(LayoutOptionData.Type.BOOLEAN)
        .optionClass(Boolean.class)
        .targets(EnumSet.of(LayoutOptionData.Target.NODES))
        .visibility(LayoutOptionData.Visibility.HIDDEN)
        .create()
    );
    registry.register(new LayoutOptionData.Builder()
        .id("org.eclipse.elk.rectpacking.currentPosition")
        .group("")
        .name("Current position of a node in the order of nodes")
        .description("The rectangles are ordered. Normally according to their definition the the model. This option specifies the current position of a node.")
        .defaultValue(CURRENT_POSITION_DEFAULT)
        .lowerBound(CURRENT_POSITION_LOWER_BOUND)
        .type(LayoutOptionData.Type.INT)
        .optionClass(Integer.class)
        .targets(EnumSet.of(LayoutOptionData.Target.NODES))
        .visibility(LayoutOptionData.Visibility.HIDDEN)
        .create()
    );
    registry.register(new LayoutOptionData.Builder()
        .id("org.eclipse.elk.rectpacking.desiredPosition")
        .group("")
        .name("Desired index of node")
        .description("The rectangles are ordered. Normally according to their definition the the model. This option allows to specify a desired position that has preference over the original position.")
        .defaultValue(DESIRED_POSITION_DEFAULT)
        .lowerBound(DESIRED_POSITION_LOWER_BOUND)
        .type(LayoutOptionData.Type.INT)
        .optionClass(Integer.class)
        .targets(EnumSet.of(LayoutOptionData.Target.NODES))
        .visibility(LayoutOptionData.Visibility.ADVANCED)
        .create()
    );
    registry.register(new LayoutOptionData.Builder()
        .id("org.eclipse.elk.rectpacking.onlyFirstIteration")
        .group("")
        .name("Only Area Approximation")
        .description("If enabled only the width approximation step is executed and the nodes are placed accordingly. The nodes are layouted according to the packingStrategy. If set to true not expansion of nodes is taking place.")
        .defaultValue(ONLY_FIRST_ITERATION_DEFAULT)
        .type(LayoutOptionData.Type.BOOLEAN)
        .optionClass(Boolean.class)
        .targets(EnumSet.of(LayoutOptionData.Target.NODES))
        .visibility(LayoutOptionData.Visibility.HIDDEN)
        .create()
    );
    registry.register(new LayoutOptionData.Builder()
        .id("org.eclipse.elk.rectpacking.rowCompaction")
        .group("")
        .name("Compact Rows")
        .description("Enables compaction. Compacts blocks if they do not use the full height of the row. This option allows to have a smaller drawing. If this option is disabled all nodes are placed next to each other in rows.")
        .defaultValue(ROW_COMPACTION_DEFAULT)
        .type(LayoutOptionData.Type.BOOLEAN)
        .optionClass(Boolean.class)
        .targets(EnumSet.of(LayoutOptionData.Target.NODES))
        .visibility(LayoutOptionData.Visibility.HIDDEN)
        .create()
    );
    registry.register(new LayoutOptionData.Builder()
        .id("org.eclipse.elk.rectpacking.expandToAspectRatio")
        .group("")
        .name("Fit Aspect Ratio")
        .description("Expands nodes if expandNodes is true to fit the aspect ratio instead of only in their bounds. The option is only useful if the used packingStrategy is ASPECT_RATIO_DRIVEN, otherwise this may result in unreasonable ndoe expansion.")
        .defaultValue(EXPAND_TO_ASPECT_RATIO_DEFAULT)
        .type(LayoutOptionData.Type.BOOLEAN)
        .optionClass(Boolean.class)
        .targets(EnumSet.of(LayoutOptionData.Target.NODES))
        .visibility(LayoutOptionData.Visibility.ADVANCED)
        .create()
    );
    registry.addDependency(
        "org.eclipse.elk.rectpacking.expandToAspectRatio",
        "org.eclipse.elk.expandNodes",
        null
    );
    registry.register(new LayoutOptionData.Builder()
        .id("org.eclipse.elk.rectpacking.targetWidth")
        .group("")
        .name("Target Width")
        .description("Option to place the rectangles in the given target width instead of approximating the width using the desired aspect ratio. The padding is not included in this. Meaning a drawing will have width of targetwidth + horizontal padding.")
        .defaultValue(TARGET_WIDTH_DEFAULT)
        .type(LayoutOptionData.Type.DOUBLE)
        .optionClass(Double.class)
        .targets(EnumSet.of(LayoutOptionData.Target.NODES))
        .visibility(LayoutOptionData.Visibility.ADVANCED)
        .create()
    );
    new org.eclipse.elk.alg.rectpacking.options.RectPackingOptions().apply(registry);
  }
}
