//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2010, 2025 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.bdd.conversion;

import static org.eclipse.escet.cif.common.CifTextUtils.getAbsName;
import static org.eclipse.escet.cif.common.CifTypeUtils.normalizeType;
import static org.eclipse.escet.cif.metamodel.cif.expressions.BinaryOperator.INTEGER_DIVISION;
import static org.eclipse.escet.cif.metamodel.java.CifConstructors.newAssignment;
import static org.eclipse.escet.cif.metamodel.java.CifConstructors.newBinaryExpression;
import static org.eclipse.escet.cif.metamodel.java.CifConstructors.newBoolType;
import static org.eclipse.escet.cif.metamodel.java.CifConstructors.newEvent;
import static org.eclipse.escet.cif.metamodel.java.CifConstructors.newInputVariableExpression;
import static org.eclipse.escet.cif.metamodel.java.CifConstructors.newMonitors;
import static org.eclipse.escet.common.emf.EMFHelper.deepclone;
import static org.eclipse.escet.common.java.Lists.concat;
import static org.eclipse.escet.common.java.Lists.copy;
import static org.eclipse.escet.common.java.Lists.filter;
import static org.eclipse.escet.common.java.Lists.first;
import static org.eclipse.escet.common.java.Lists.last;
import static org.eclipse.escet.common.java.Lists.list;
import static org.eclipse.escet.common.java.Lists.listc;
import static org.eclipse.escet.common.java.Lists.reverse;
import static org.eclipse.escet.common.java.Lists.set2list;
import static org.eclipse.escet.common.java.Maps.map;
import static org.eclipse.escet.common.java.Maps.mapc;
import static org.eclipse.escet.common.java.Pair.pair;
import static org.eclipse.escet.common.java.Sets.copy;
import static org.eclipse.escet.common.java.Sets.difference;
import static org.eclipse.escet.common.java.Sets.list2set;
import static org.eclipse.escet.common.java.Sets.set;
import static org.eclipse.escet.common.java.Sets.setc;
import static org.eclipse.escet.common.java.Sets.union;
import static org.eclipse.escet.common.java.Strings.fmt;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.RoundingMode;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Random;
import java.util.Set;
import java.util.regex.Pattern;

import org.apache.commons.lang3.StringUtils;
import org.eclipse.escet.cif.bdd.conversion.bitvectors.BddBitVector;
import org.eclipse.escet.cif.bdd.conversion.bitvectors.BddBitVectorAndCarry;
import org.eclipse.escet.cif.bdd.conversion.bitvectors.SignedBddBitVector;
import org.eclipse.escet.cif.bdd.conversion.bitvectors.SignedBddBitVectorAndCarry;
import org.eclipse.escet.cif.bdd.conversion.bitvectors.UnsignedBddBitVector;
import org.eclipse.escet.cif.bdd.conversion.preconditions.CifToBddConverterPreChecker;
import org.eclipse.escet.cif.bdd.settings.AllowNonDeterminism;
import org.eclipse.escet.cif.bdd.settings.CifBddSettings;
import org.eclipse.escet.cif.bdd.settings.CifBddStatistics;
import org.eclipse.escet.cif.bdd.settings.EdgeGranularity;
import org.eclipse.escet.cif.bdd.settings.EdgeOrderDuplicateEventAllowance;
import org.eclipse.escet.cif.bdd.settings.ExplorationStrategy;
import org.eclipse.escet.cif.bdd.spec.CifBddDiscVariable;
import org.eclipse.escet.cif.bdd.spec.CifBddDomain;
import org.eclipse.escet.cif.bdd.spec.CifBddEdge;
import org.eclipse.escet.cif.bdd.spec.CifBddInputVariable;
import org.eclipse.escet.cif.bdd.spec.CifBddLocPtrVariable;
import org.eclipse.escet.cif.bdd.spec.CifBddSpec;
import org.eclipse.escet.cif.bdd.spec.CifBddTypedVariable;
import org.eclipse.escet.cif.bdd.spec.CifBddVariable;
import org.eclipse.escet.cif.bdd.utils.BddUtils;
import org.eclipse.escet.cif.bdd.varorder.helper.VarOrder;
import org.eclipse.escet.cif.bdd.varorder.helper.VarOrderHelper;
import org.eclipse.escet.cif.bdd.varorder.helper.VarOrdererData;
import org.eclipse.escet.cif.bdd.varorder.orderers.VarOrderer;
import org.eclipse.escet.cif.bdd.varorder.parser.VarOrdererParser;
import org.eclipse.escet.cif.bdd.varorder.parser.VarOrdererTypeChecker;
import org.eclipse.escet.cif.bdd.varorder.parser.ast.VarOrdererInstance;
import org.eclipse.escet.cif.cif2cif.ElimComponentDefInst;
import org.eclipse.escet.cif.cif2cif.LinearizeProduct;
import org.eclipse.escet.cif.cif2cif.LinearizeProduct.LinearizedEdgeIterator;
import org.eclipse.escet.cif.cif2cif.LocationPointerManager;
import org.eclipse.escet.cif.cif2cif.RemoveIoDecls;
import org.eclipse.escet.cif.common.CifCollectUtils;
import org.eclipse.escet.cif.common.CifEnumLiteral;
import org.eclipse.escet.cif.common.CifEquationUtils;
import org.eclipse.escet.cif.common.CifEvalException;
import org.eclipse.escet.cif.common.CifEvalUtils;
import org.eclipse.escet.cif.common.CifEventUtils;
import org.eclipse.escet.cif.common.CifEventUtils.Alphabets;
import org.eclipse.escet.cif.common.CifExecSchemeUtils;
import org.eclipse.escet.cif.common.CifGuardUtils;
import org.eclipse.escet.cif.common.CifGuardUtils.LocRefExprCreator;
import org.eclipse.escet.cif.common.CifLocationUtils;
import org.eclipse.escet.cif.common.CifReachabilityRequirementAnnotationUtils;
import org.eclipse.escet.cif.common.CifTextUtils;
import org.eclipse.escet.cif.common.CifTypeUtils;
import org.eclipse.escet.cif.common.CifUpdateUtils;
import org.eclipse.escet.cif.common.CifUpdateUtils.UnsupportedUpdateException;
import org.eclipse.escet.cif.common.CifValueUtils;
import org.eclipse.escet.cif.common.FuncLocalVarOrderer;
import org.eclipse.escet.cif.metamodel.cif.ComplexComponent;
import org.eclipse.escet.cif.metamodel.cif.Component;
import org.eclipse.escet.cif.metamodel.cif.Group;
import org.eclipse.escet.cif.metamodel.cif.InvKind;
import org.eclipse.escet.cif.metamodel.cif.Invariant;
import org.eclipse.escet.cif.metamodel.cif.Specification;
import org.eclipse.escet.cif.metamodel.cif.SupKind;
import org.eclipse.escet.cif.metamodel.cif.automata.Assignment;
import org.eclipse.escet.cif.metamodel.cif.automata.Automaton;
import org.eclipse.escet.cif.metamodel.cif.automata.Edge;
import org.eclipse.escet.cif.metamodel.cif.automata.EdgeEvent;
import org.eclipse.escet.cif.metamodel.cif.automata.Location;
import org.eclipse.escet.cif.metamodel.cif.automata.Monitors;
import org.eclipse.escet.cif.metamodel.cif.automata.Update;
import org.eclipse.escet.cif.metamodel.cif.automata.impl.EdgeEventImpl;
import org.eclipse.escet.cif.metamodel.cif.declarations.AlgVariable;
import org.eclipse.escet.cif.metamodel.cif.declarations.Constant;
import org.eclipse.escet.cif.metamodel.cif.declarations.Declaration;
import org.eclipse.escet.cif.metamodel.cif.declarations.DiscVariable;
import org.eclipse.escet.cif.metamodel.cif.declarations.EnumDecl;
import org.eclipse.escet.cif.metamodel.cif.declarations.EnumLiteral;
import org.eclipse.escet.cif.metamodel.cif.declarations.Event;
import org.eclipse.escet.cif.metamodel.cif.declarations.InputVariable;
import org.eclipse.escet.cif.metamodel.cif.expressions.AlgVariableExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.BinaryExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.BinaryOperator;
import org.eclipse.escet.cif.metamodel.cif.expressions.BoolExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.ConstantExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.DiscVariableExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.ElifExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.EventExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.Expression;
import org.eclipse.escet.cif.metamodel.cif.expressions.FunctionCallExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.FunctionExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.IfExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.InputVariableExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.IntExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.LocationExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.StdLibFunctionExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.SwitchCase;
import org.eclipse.escet.cif.metamodel.cif.expressions.SwitchExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.UnaryExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.UnaryOperator;
import org.eclipse.escet.cif.metamodel.cif.functions.AssignmentFuncStatement;
import org.eclipse.escet.cif.metamodel.cif.functions.ElifFuncStatement;
import org.eclipse.escet.cif.metamodel.cif.functions.FunctionStatement;
import org.eclipse.escet.cif.metamodel.cif.functions.IfFuncStatement;
import org.eclipse.escet.cif.metamodel.cif.functions.InternalFunction;
import org.eclipse.escet.cif.metamodel.cif.functions.ReturnFuncStatement;
import org.eclipse.escet.cif.metamodel.cif.types.BoolType;
import org.eclipse.escet.cif.metamodel.cif.types.CifType;
import org.eclipse.escet.cif.metamodel.cif.types.EnumType;
import org.eclipse.escet.cif.metamodel.cif.types.IntType;
import org.eclipse.escet.common.box.GridBox;
import org.eclipse.escet.common.box.GridBox.GridBoxLayout;
import org.eclipse.escet.common.java.Assert;
import org.eclipse.escet.common.java.Lists;
import org.eclipse.escet.common.java.Pair;
import org.eclipse.escet.common.java.Sets;
import org.eclipse.escet.common.java.Strings;
import org.eclipse.escet.common.java.Termination;
import org.eclipse.escet.common.java.exceptions.InvalidOptionException;
import org.eclipse.escet.common.java.exceptions.UnsupportedException;
import org.eclipse.escet.common.java.output.WarnOutput;
import org.eclipse.escet.common.position.metamodel.position.PositionObject;
import org.eclipse.escet.setext.runtime.DebugMode;
import org.eclipse.escet.setext.runtime.exceptions.SyntaxException;

import com.github.javabdd.BDD;
import com.github.javabdd.BDDFactory;
import com.github.javabdd.JFactory;

/**
 * Converter to convert a CIF specification to a CIF/BDD representation.
 *
 * <p>
 * To use, call the following methods, in the given order:
 * <ul>
 * <li>{@link #preprocess}</li>
 * <li>{@link #createFactory}</li>
 * <li>{@link #convert}</li>
 * <li>Optionally, any of the {@code convert*} methods, any number of times.</li>
 * <li>{@link #free}</li>
 * </ul>
 * Check their JavaDocs for further details.
 * </p>
 */
public class CifToBddConverter {
    /** The human-readable name of the application. Should start with a capital letter. */
    private final String appName;

    /**
     * Per requirement automaton, the monitors as specified in the original specification. They are replaced by monitors
     * that monitor the entire alphabet of the automaton, in order to be able to treat requirement automata as plants.
     * This mapping can be used to restore the original monitors. The mapping is {@code null} if not yet or no longer
     * available.
     */
    private Map<Automaton, Monitors> originalMonitors;

    /** Per object with a boolean type that has been converted, its cached conversion result. */
    private final Map<PositionObject, BDD> conversionBddCache = map();

    /** Per object with an integer or enum type that has been converted, its cached conversion result. */
    private final Map<PositionObject, BddBitVector<?, ?>> conversionBitVectorCache = map();

    /**
     * Constructor for the {@link CifToBddConverter} class.
     *
     * @param appName The human-readable name of the application. Should start with a capital letter.
     */
    public CifToBddConverter(String appName) {
        this.appName = appName;
    }

    /**
     * Pre-process the input model prior to {@link #convert conversion}.
     *
     * <p>
     * Performs the following pre-processing:
     * <ul>
     * <li>{@link RemoveIoDecls Removes all I/O declarations}. Warns in case any CIF/SVG declarations are present and
     * removed.</li>
     * <li>{@link ElimComponentDefInst Eliminates component definition/instantiation}.</li>
     * </ul>
     * </p>
     *
     * <p>
     * Furthermore, it checks the following preconditions:
     * <ul>
     * <li>{@link PlantsRefsReqsChecker Plants should not refer to requirement state}.</li>
     * <li><@link CifBddConversionPreChecker CIF to BDD conversion preconditions}.</li>
     * </ul>
     * </p>
     *
     * @param spec The CIF specification to pre-process. Is modified in-place.
     * @param specAbsPath The absolute local file system path to the CIF file.
     * @param warnOutput Callback for warning output.
     * @param doPlantsRefReqsWarn Whether to warn about plants that reference requirement state.
     * @param termination Cooperative termination query function.
     * @throws UnsupportedException In case the specification is unsupported.
     */
    public void preprocess(Specification spec, String specAbsPath, WarnOutput warnOutput, boolean doPlantsRefReqsWarn,
            Termination termination)
    {
        CifToBddConverterPreChecker checker = preparePreChecker(spec, warnOutput, doPlantsRefReqsWarn, termination);
        checker.reportPreconditionViolations(spec, specAbsPath, appName);
    }

    /**
     * Prepare the specification for pre-checking and return the checker.
     *
     * <p>
     * Performs the following pre-processing:
     * <ul>
     * <li>{@link RemoveIoDecls Removes all I/O declarations}. Warns in case any CIF/SVG declarations are present and
     * removed.</li>
     * <li>{@link ElimComponentDefInst Eliminates component definition/instantiation}.</li>
     * </ul>
     * </p>
     *
     * <p>
     * Furthermore, it may check the following pre-condition:
     * <ul>
     * <li>{@link PlantsRefsReqsChecker Plants should not refer to requirement state}.</li>
     * </ul>
     * </p>
     *
     * @param spec The CIF specification to pre-process. Is modified in-place.
     * @param warnOutput Callback for warning output.
     * @param doPlantsRefReqsWarn Whether to warn about plants that reference requirement state.
     * @param termination Cooperative termination query function.
     * @return The created pre-checker.
     */
    public static CifToBddConverterPreChecker preparePreChecker(Specification spec, WarnOutput warnOutput,
            boolean doPlantsRefReqsWarn, Termination termination)
    {
        // Remove/ignore I/O declarations, to increase the supported subset.
        RemoveIoDecls removeIoDecls = new RemoveIoDecls();
        removeIoDecls.transform(spec);
        removeIoDecls.warnAboutIgnoredSvgInputDecsIfRemoved(warnOutput);

        // Eliminate component definition/instantiation, to avoid having to handle them.
        new ElimComponentDefInst().transform(spec);

        // Check whether plants reference requirements.
        if (doPlantsRefReqsWarn) {
            new PlantsRefsReqsChecker(warnOutput).checkPlantRefToRequirement(spec);
        }

        // Return the checker.
        return new CifToBddConverterPreChecker(termination);
    }

    /**
     * Create a suitable BDD factory to use for the {@link #convert conversion}.
     *
     * @param settings The settings to use.
     * @param continuousOpMisses The list into which to collect continuous operation misses samples.
     * @param continuousUsedBddNodes The list into which to collect continuous used BDD nodes statistics samples.
     * @return The new BDD factory. The caller is responsible for {@link BDDFactory#done cleaning up} the factory once
     *     it is no longer needed.
     */
    public static BDDFactory createFactory(CifBddSettings settings, List<Long> continuousOpMisses,
            List<Integer> continuousUsedBddNodes)
    {
        // Determine BDD operation cache size and ratio to use.
        double bddOpCacheRatio = settings.getBddOpCacheRatio();
        Integer bddOpCacheSize = settings.getBddOpCacheSize();
        if (bddOpCacheSize == null) {
            // A non-fixed cache size should be used. Initialize BDD cache size using cache ratio.
            bddOpCacheSize = (int)(settings.getBddInitNodeTableSize() * bddOpCacheRatio);
            if (bddOpCacheSize < 2) {
                bddOpCacheSize = 2;
            }
        } else {
            // Disable cache ratio.
            bddOpCacheRatio = -1;
        }

        // Create BDD factory, and configure cache settings.
        BDDFactory factory = JFactory.init(settings.getBddInitNodeTableSize(), bddOpCacheSize);
        if (bddOpCacheRatio != -1) {
            factory.setCacheRatio(bddOpCacheRatio);
        }

        // Configure statistics.
        boolean doGcStats = settings.getCifBddStatistics().contains(CifBddStatistics.BDD_GC_COLLECT);
        boolean doResizeStats = settings.getCifBddStatistics().contains(CifBddStatistics.BDD_GC_RESIZE);
        boolean doContinuousPerformanceStats = settings.getCifBddStatistics().contains(CifBddStatistics.BDD_PERF_CONT);
        BddUtils.registerBddCallbacks(factory, doGcStats, doResizeStats, doContinuousPerformanceStats,
                settings.getNormalOutput(), continuousOpMisses, continuousUsedBddNodes);

        boolean doCacheStats = settings.getCifBddStatistics().contains(CifBddStatistics.BDD_PERF_NODE_OP);
        boolean doMaxBddNodesStats = settings.getCifBddStatistics().contains(CifBddStatistics.BDD_PERF_MAX_NODES);
        boolean doMaxMemoryStats = settings.getCifBddStatistics().contains(CifBddStatistics.MAX_MEMORY);
        if (doCacheStats || doContinuousPerformanceStats) {
            factory.getCacheStats().enableMeasurements();
        }
        if (doMaxBddNodesStats) {
            factory.getMaxUsedBddNodesStats().enableMeasurements();
        }
        if (doMaxMemoryStats) {
            factory.getMaxMemoryStats().enableMeasurements();
        }

        // Return BDD factory.
        return factory;
    }

    /**
     * Converts a CIF specification to a CIF/BDD representation. Also checks the specification for non-deterministic
     * events, as {@link CifBddSettings#getAllowNonDeterminism configured}.
     *
     * @param spec The CIF specification to convert. Must have been {@link #preprocess pre-processed} already.
     * @param settings The settings to use.
     * @param factory The BDD factory to use. A suitable factory can be created using {@link #createFactory}.
     * @return The CIF/BDD representation of the CIF specification.
     * @throws UnsupportedException In case the specification has non-determinism that is not supported.
     */
    public CifBddSpec convert(Specification spec, CifBddSettings settings, BDDFactory factory) {
        // Initialize CIF/BDD specification.
        CifBddSpec cifBddSpec = new CifBddSpec(settings);
        cifBddSpec.factory = factory;

        if (cifBddSpec.settings.getTermination().isRequested()) {
            return cifBddSpec;
        }

        // Collect event declarations. May collect more than the union of alphabets.
        List<Event> events = list();
        collectEvents(spec, events);
        if (cifBddSpec.settings.getAdhereToExecScheme()) {
            // We must consider the events in this order, to ensure that we adhere to the transition execution order
            // defined by the controller properties checker.
            CifExecSchemeUtils.orderEvents(events);
        }

        if (cifBddSpec.settings.getTermination().isRequested()) {
            return cifBddSpec;
        }

        // Collect automata.
        List<Automaton> automata = list();
        collectAutomata(spec, automata);
        if (cifBddSpec.settings.getAdhereToExecScheme()) {
            // We must consider the automata in this order, to ensure that we adhere to the transition execution order
            // defined by the controller properties checker.
            CifExecSchemeUtils.orderAutomata(automata);
        }

        if (cifBddSpec.settings.getTermination().isRequested()) {
            return cifBddSpec;
        }

        // Partition automata into plants and requirements.
        List<Automaton> plants = automata.stream().filter(a -> a.getKind() == SupKind.PLANT).toList();
        List<Automaton> requirements = automata.stream().filter(a -> a.getKind() == SupKind.REQUIREMENT).toList();
        Assert.areEqual(automata.size(), plants.size() + requirements.size());
        automata = null;

        if (cifBddSpec.settings.getTermination().isRequested()) {
            return cifBddSpec;
        }

        // Get plant and requirement alphabets.
        List<Alphabets> plantAlphabets = listc(plants.size());
        List<Alphabets> reqAlphabets = listc(requirements.size());
        Set<Event> plantAlphabet = set();
        Set<Event> reqAlphabet = set();
        for (Automaton plant: plants) {
            Alphabets alphabets = CifEventUtils.getAllAlphabets(plant, null);
            plantAlphabets.add(alphabets);
            plantAlphabet.addAll(alphabets.syncAlphabet);
            plantAlphabet.addAll(alphabets.sendAlphabet);
            plantAlphabet.addAll(alphabets.recvAlphabet);
        }
        for (Automaton req: requirements) {
            Alphabets alphabets = CifEventUtils.getAllAlphabets(req, null);
            reqAlphabets.add(alphabets);
            reqAlphabet.addAll(alphabets.syncAlphabet);
            reqAlphabet.addAll(alphabets.sendAlphabet);
            reqAlphabet.addAll(alphabets.recvAlphabet);
        }

        if (cifBddSpec.settings.getTermination().isRequested()) {
            return cifBddSpec;
        }

        // Get alphabet for (un)controlled system. We allow events that are only in the alphabet of the requirements, by
        // considering them always enabled in the plant. Since the requirement automata are used as fully monitored
        // plants (full observers) during linearization, the combined linearized edges for such an event always allow
        // that event, in the uncontrolled system.
        cifBddSpec.alphabet = union(plantAlphabet, reqAlphabet);
        if (cifBddSpec.settings.getAdhereToExecScheme()) {
            // We must consider the events in this order, to ensure that we adhere to the transition execution order
            // defined by the controller properties checker.
            List<Event> orderedAlphabet = Lists.set2list(cifBddSpec.alphabet);
            CifExecSchemeUtils.orderEvents(orderedAlphabet);
            cifBddSpec.alphabet = Sets.list2set(orderedAlphabet);
        }

        if (cifBddSpec.settings.getTermination().isRequested()) {
            return cifBddSpec;
        }

        // Get controllable events subset of the alphabet. If we configured the settings to ensure that we adhere to
        // the execution scheme of the controller properties checker, these events will be in the order that they should
        // be considered according to the execution scheme.
        cifBddSpec.controllables = cifBddSpec.alphabet.stream().filter(Event::getControllable).collect(Sets.toSet());

        if (cifBddSpec.settings.getTermination().isRequested()) {
            return cifBddSpec;
        }

        // Collect variables, and automata for which we need to create location pointer variables, i.e. the automata
        // with more than one location.
        List<PositionObject> cifVarObjs = list();
        collectVariableObjects(spec, cifVarObjs);

        if (cifBddSpec.settings.getTermination().isRequested()) {
            return cifBddSpec;
        }

        // Create location pointer manager.
        List<Automaton> lpAuts = filter(cifVarObjs, Automaton.class);
        CifBddLocationPointerManager locPtrManager = new CifBddLocationPointerManager(lpAuts);

        if (cifBddSpec.settings.getTermination().isRequested()) {
            return cifBddSpec;
        }

        // Convert variables. Create location pointers.
        cifBddSpec.variables = new CifBddVariable[cifVarObjs.size()];
        for (int i = 0; i < cifBddSpec.variables.length; i++) {
            PositionObject cifVarObj = cifVarObjs.get(i);
            if (cifVarObj instanceof Automaton) {
                Automaton lpAut = (Automaton)cifVarObj;
                DiscVariable lpVar = locPtrManager.getLocationPointer(lpAut);
                cifBddSpec.variables[i] = createLpVar(lpAut, lpVar);
            } else {
                cifBddSpec.variables[i] = convertTypedVar((Declaration)cifVarObj);
            }
        }

        if (cifBddSpec.settings.getTermination().isRequested()) {
            return cifBddSpec;
        }

        // Order variables and create domains.
        orderVars(cifBddSpec, spec);
        if (cifBddSpec.settings.getTermination().isRequested()) {
            return cifBddSpec;
        }

        createVarDomains(cifBddSpec);
        if (cifBddSpec.settings.getTermination().isRequested()) {
            return cifBddSpec;
        }

        // Create auxiliary data for updates.
        createUpdateAuxiliaries(cifBddSpec);
        if (cifBddSpec.settings.getTermination().isRequested()) {
            return cifBddSpec;
        }

        // Convert initialization predicates.
        cifBddSpec.initialsVars = listc(cifBddSpec.variables.length);
        for (int i = 0; i < cifBddSpec.variables.length; i++) {
            cifBddSpec.initialsVars.add(null);
        }
        cifBddSpec.initialsComps = list();
        cifBddSpec.initialsLocs = list();
        cifBddSpec.initialVars = cifBddSpec.factory.one();
        cifBddSpec.initialComps = cifBddSpec.factory.one();
        cifBddSpec.initialLocs = cifBddSpec.factory.one();
        convertInit(spec, cifBddSpec, locPtrManager);
        BDD initialCompsAndLocs = cifBddSpec.initialComps.and(cifBddSpec.initialLocs);
        cifBddSpec.initial = cifBddSpec.initialVars.and(initialCompsAndLocs);
        initialCompsAndLocs.free();

        if (cifBddSpec.settings.getTermination().isRequested()) {
            return cifBddSpec;
        }

        // Convert marker predicates.
        cifBddSpec.markedsComps = list();
        cifBddSpec.markedsLocs = list();
        cifBddSpec.markedComps = cifBddSpec.factory.one();
        cifBddSpec.markedLocs = cifBddSpec.factory.one();
        convertMarked(spec, cifBddSpec, locPtrManager);
        cifBddSpec.marked = cifBddSpec.markedComps.and(cifBddSpec.markedLocs);

        if (cifBddSpec.settings.getTermination().isRequested()) {
            return cifBddSpec;
        }

        // Initialize state plant invariants (predicates).
        cifBddSpec.plantInvsComps = list();
        cifBddSpec.plantInvsLocs = list();
        cifBddSpec.plantInvComps = cifBddSpec.factory.one();
        cifBddSpec.plantInvLocs = cifBddSpec.factory.one();

        // Initialize state requirement invariants (predicates).
        cifBddSpec.reqInvsComps = list();
        cifBddSpec.reqInvsLocs = list();
        cifBddSpec.reqInvComps = cifBddSpec.factory.one();
        cifBddSpec.reqInvLocs = cifBddSpec.factory.one();

        // Convert state invariants.
        convertStateInvs(spec, cifBddSpec, locPtrManager);

        // Determine state plant invariant for the system, combination of the state plant invariant for the components
        // and the state plant invariant for the locations of the automata.
        cifBddSpec.plantInv = cifBddSpec.plantInvComps.and(cifBddSpec.plantInvLocs);

        // Determine state requirement invariant for the system, combination of the state requirement invariant for the
        // components and the state requirement invariant for the locations of automata.
        cifBddSpec.reqInv = cifBddSpec.reqInvComps.and(cifBddSpec.reqInvLocs);

        if (cifBddSpec.settings.getTermination().isRequested()) {
            return cifBddSpec;
        }

        // Set combined predicate for initialization with state invariants.
        cifBddSpec.initialPlantInv = cifBddSpec.initial.and(cifBddSpec.plantInv);
        cifBddSpec.initialInv = cifBddSpec.initialPlantInv.and(cifBddSpec.reqInv);

        // Set combined predicate for marking with state invariants.
        cifBddSpec.markedPlantInv = cifBddSpec.marked.and(cifBddSpec.plantInv);
        cifBddSpec.markedInv = cifBddSpec.markedPlantInv.and(cifBddSpec.reqInv);

        if (cifBddSpec.settings.getTermination().isRequested()) {
            return cifBddSpec;
        }

        // Initialize state/event plant exclusion conditions for events.
        cifBddSpec.stateEvtExclPlants = mapc(cifBddSpec.alphabet.size());
        cifBddSpec.stateEvtExclPlantLists = mapc(cifBddSpec.alphabet.size());
        for (Event event: cifBddSpec.alphabet) {
            cifBddSpec.stateEvtExclPlants.put(event, cifBddSpec.factory.one());
            cifBddSpec.stateEvtExclPlantLists.put(event, list());
        }

        // Initialize state/event requirement exclusion conditions for controllable events.
        cifBddSpec.stateEvtExclsReqAuts = mapc(cifBddSpec.controllables.size());
        cifBddSpec.stateEvtExclsReqInvs = mapc(cifBddSpec.controllables.size());
        for (Event event: cifBddSpec.controllables) {
            cifBddSpec.stateEvtExclsReqAuts.put(event, cifBddSpec.factory.one());
            cifBddSpec.stateEvtExclsReqInvs.put(event, cifBddSpec.factory.one());
        }

        // Initialize state/event requirement exclusion conditions for events.
        cifBddSpec.stateEvtExclReqs = mapc(cifBddSpec.alphabet.size());
        cifBddSpec.stateEvtExclReqLists = mapc(cifBddSpec.alphabet.size());
        for (Event event: cifBddSpec.alphabet) {
            cifBddSpec.stateEvtExclReqs.put(event, cifBddSpec.factory.one());
            cifBddSpec.stateEvtExclReqLists.put(event, list());
        }

        if (cifBddSpec.settings.getTermination().isRequested()) {
            return cifBddSpec;
        }

        // Convert state/event exclusion invariants.
        convertStateEvtExclInvs(spec, cifBddSpec, locPtrManager);
        if (cifBddSpec.settings.getTermination().isRequested()) {
            return cifBddSpec;
        }

        // Preconvert requirement automata, to enable treating them as plants from here on.
        preconvertReqAuts(requirements, reqAlphabets, cifBddSpec);
        if (cifBddSpec.settings.getTermination().isRequested()) {
            return cifBddSpec;
        }

        // Convert plant and requirement automata.
        convertPlantReqAuts(plants, requirements, plantAlphabets, reqAlphabets, locPtrManager, cifBddSpec);
        if (cifBddSpec.settings.getTermination().isRequested()) {
            return cifBddSpec;
        }

        // Convert reachability requirement annotation predicates.
        convertReachReqPreds(spec, cifBddSpec);
        if (cifBddSpec.settings.getTermination().isRequested()) {
            return cifBddSpec;
        }

        // Restore original monitors of the requirement automata.
        for (Entry<Automaton, Monitors> entry: originalMonitors.entrySet()) {
            entry.getKey().setMonitors(entry.getValue());
        }
        originalMonitors = null;

        if (cifBddSpec.settings.getTermination().isRequested()) {
            return cifBddSpec;
        }

        // Add events and edges for input variables.
        addInputVariableEdges(cifBddSpec);
        if (cifBddSpec.settings.getTermination().isRequested()) {
            return cifBddSpec;
        }

        // Merge edges to the desired granularity.
        mergeEdges(cifBddSpec);
        if (cifBddSpec.settings.getTermination().isRequested()) {
            return cifBddSpec;
        }

        // Order the CIF/BDD edges.
        orderEdges(cifBddSpec);
        if (cifBddSpec.settings.getTermination().isRequested()) {
            return cifBddSpec;
        }

        // Check edge workset algorithm settings.
        checkEdgeWorksetAlgorithmSettings(cifBddSpec.settings);
        if (cifBddSpec.settings.getTermination().isRequested()) {
            return cifBddSpec;
        }

        // Check saturation settings.
        checkSaturationSettings(cifBddSpec.settings);
        if (cifBddSpec.settings.getTermination().isRequested()) {
            return cifBddSpec;
        }

        // Return the conversion result, the CIF/BDD specification.
        return cifBddSpec;
    }

    /**
     * Converts a CIF variable to a CIF/BDD variable.
     *
     * @param var The CIF variable. Must be a {@link DiscVariable} or a {@link InputVariable}.
     * @return The CIF/BDD variable.
     */
    private CifBddVariable convertTypedVar(Declaration var) {
        // Get normalized type of the variable.
        CifType type = switch (var) {
            case DiscVariable discVar -> discVar.getType();
            case InputVariable inputVar -> inputVar.getType();
            default -> throw new RuntimeException("Unexpected variable: " + var);
        };
        type = normalizeType(type);

        // Get variable information, based on type.
        int lower;
        int upper;
        if (type instanceof BoolType) {
            // Represent as 'int[0..1]', with '0' and '1' for 'false' and 'true', respectively.
            lower = 0;
            upper = 1;
        } else if (type instanceof IntType) {
            // Get integer type range.
            IntType intType = (IntType)type;
            Assert.check(!CifTypeUtils.isRangeless(intType));
            lower = intType.getLower();
            upper = intType.getUpper();
        } else if (type instanceof EnumType) {
            // Represent as 'int[0..n-1]' for an enumeration with 'n' literals.
            EnumDecl enumDecl = ((EnumType)type).getEnum();
            int count = enumDecl.getLiterals().size();
            lower = 0;
            upper = count - 1;
        } else {
            throw new RuntimeException("Unexpected type: " + type);
        }

        // Construct CIF/BDD variable.
        return switch (var) {
            case DiscVariable discVar -> new CifBddDiscVariable(discVar, type, lower, upper);
            case InputVariable inputVar -> new CifBddInputVariable(inputVar, type, lower, upper);
            default -> throw new RuntimeException("Unexpected variable: " + var);
        };
    }

    /**
     * Creates a CIF/BDD location pointer variable for a CIF automaton.
     *
     * @param aut The CIF automaton.
     * @param var The dummy, internally-created location pointer variable for the CIF automaton. Does not have a type.
     * @return The CIF/BDD variable.
     */
    private CifBddVariable createLpVar(Automaton aut, DiscVariable var) {
        int count = aut.getLocations().size();
        Assert.check(count > 1);
        return new CifBddLocPtrVariable(aut, var);
    }

    /**
     * Orders the CIF/BDD variables. Also sets the {@link CifBddVariable#group group} of each CIF/BDD variable.
     *
     * @param cifBddSpec The CIF/BDD specification.
     * @param spec The CIF specification.
     */
    private void orderVars(CifBddSpec cifBddSpec, Specification spec) {
        // Configure variable orderer.
        String varOrderTxt = cifBddSpec.settings.getBddVarOrderAdvanced();
        List<VarOrdererInstance> parseResult;
        try {
            parseResult = new VarOrdererParser().parseString(varOrderTxt, "/in-memory.varorder", null, DebugMode.NONE);
        } catch (SyntaxException ex) {
            throw new InvalidOptionException("Invalid BDD variable ordering configuration.", ex);
        }

        VarOrdererTypeChecker typeChecker = new VarOrdererTypeChecker(Arrays.asList(cifBddSpec.variables),
                cifBddSpec.settings);
        VarOrderer varOrderer = typeChecker.typeCheck(parseResult);
        Assert.check(!typeChecker.hasWarning());
        if (varOrderer == null) {
            Assert.check(typeChecker.hasError());
            Assert.check(typeChecker.getProblems().size() == 1);
            InvalidOptionException ex = new InvalidOptionException(typeChecker.getProblems().get(0).toString());
            throw new InvalidOptionException("Invalid BDD variable ordering configuration.", ex);
        }

        // Skip ordering, including debug output printing, if no variables are present.
        int varCnt = cifBddSpec.variables.length;
        if (varCnt == 0) {
            cifBddSpec.settings.getDebugOutput().line("The specification has no BDD variables.");
            return;
        }

        // Initialize to model variable order without interleaving.
        for (int i = 0; i < cifBddSpec.variables.length; i++) {
            cifBddSpec.variables[i].group = i;
        }

        // Print variable debugging information, before ordering.
        boolean dbgEnabled = cifBddSpec.settings.getDebugOutput().isEnabled();
        if (dbgEnabled) {
            debugCifVars(cifBddSpec);
        }

        // Only apply variable ordering if there are at least two variables (to order).
        if (cifBddSpec.variables.length < 2) {
            if (dbgEnabled) {
                cifBddSpec.settings.getDebugOutput().line();
                cifBddSpec.settings.getDebugOutput().line("Skipping variable ordering: only one variable present.");
            }
            return;
        }

        // Create variable order helper, based on model order.
        List<CifBddVariable> varsInModelOrder = Collections.unmodifiableList(Arrays.asList(cifBddSpec.variables));
        VarOrderHelper helper = new VarOrderHelper(spec, varsInModelOrder, cifBddSpec.settings.getDebugOutput(),
                cifBddSpec.settings.getIndentAmount());

        // Get current variable order, which is model order.
        VarOrder curOrder = VarOrder.createFromOrderedVars(varsInModelOrder);
        List<CifBddVariable> varsInCurOrder = curOrder.getOrderedVars();

        // Create variable order data for input to first orderer.
        VarOrdererData data = new VarOrdererData(varsInModelOrder, curOrder, helper);

        // Get new variable order.
        if (dbgEnabled) {
            cifBddSpec.settings.getDebugOutput().line();
            cifBddSpec.settings.getDebugOutput().line("Applying variable ordering:");
        }
        VarOrdererData orderingResult = varOrderer.order(data, dbgEnabled, 1);
        VarOrder newOrder = orderingResult.varOrder;
        List<CifBddVariable> varsInNewOrder = newOrder.getOrderedVars();

        // Check sanity of current and new variable orders.
        Assert.check(curOrder.getVarOrder().stream().allMatch(grp -> !grp.isEmpty())); // No empty groups.
        Assert.check(newOrder.getVarOrder().stream().allMatch(grp -> !grp.isEmpty())); // No empty groups.
        Assert.areEqual(varsInCurOrder.size(), list2set(varsInCurOrder).size()); // No duplicates.
        Assert.areEqual(varsInNewOrder.size(), list2set(varsInNewOrder).size()); // No duplicates.
        Assert.areEqual(varsInCurOrder.size(), varsInNewOrder.size()); // Same number of variables.
        Assert.areEqual(list2set(varsInCurOrder), list2set(varsInNewOrder)); // Same variables.

        // Update the variable order of the CIF/BDD specification.
        cifBddSpec.variables = varsInNewOrder.toArray(n -> new CifBddVariable[n]);
        for (int i = 0; i < newOrder.getVarOrder().size(); i++) {
            List<CifBddVariable> group = newOrder.getVarOrder().get(i);
            for (CifBddVariable var: group) {
                var.group = i;
            }
        }

        // If the new order differs from the current order, print updated variable debugging information.
        if (dbgEnabled) {
            boolean orderChanged = !curOrder.equals(newOrder);
            cifBddSpec.settings.getDebugOutput().line();
            cifBddSpec.settings.getDebugOutput().line("Variable order %schanged.", orderChanged ? "" : "un");
            if (orderChanged) {
                cifBddSpec.settings.getDebugOutput().line();
                debugCifVars(cifBddSpec);
            }
        }
    }

    /**
     * Prints CIF variable information. Should only be invoked if debug output is enabled.
     *
     * @param cifBddSpec The CIF/BDD specification.
     */
    private static void debugCifVars(CifBddSpec cifBddSpec) {
        // Get variable and domain counts.
        int cifVarCnt = cifBddSpec.variables.length;
        int nrOfExtraDomains = cifBddSpec.settings.getBddExtraVarDomainNames().size();
        int nrOfDomainsPerVar = 2 + nrOfExtraDomains;
        String nrOfDomainsPerVarTxt = fmt("%,d", nrOfDomainsPerVar);

        // Initialize grid and header.
        GridBox grid = new GridBox(cifVarCnt + 4, 10, 0, 2);
        grid.set(0, 0, "Nr");
        grid.set(0, 1, "Kind");
        grid.set(0, 2, "Type");
        grid.set(0, 3, "Name");
        grid.set(0, 4, "Group");
        grid.set(0, 5, "BDD vars");
        grid.set(0, 6, "CIF values");
        grid.set(0, 7, "BDD values");
        grid.set(0, 8, "Values used");
        grid.set(0, 9, "BDD var indices");

        // Fill grid with variable information.
        Set<Integer> groups = set();
        int totalBddVarCnt = 0;
        BigInteger totalUsedCnt = BigInteger.ZERO;
        BigInteger totalReprCnt = BigInteger.ZERO;
        for (int i = 0; i < cifVarCnt; i++) {
            // Get CIF/BDD variable.
            CifBddVariable var = cifBddSpec.variables[i];

            // Get type text.
            String typeTxt = var.getTypeText();
            if (typeTxt == null) {
                typeTxt = "n/a";
            }

            // Get some counts.
            int bddCnt = var.getBddVarCount();
            int usedCnt = var.count;
            BigInteger reprCnt = var.getBddDomainSize();
            int currentBddVarCnt = bddCnt * nrOfDomainsPerVar;
            int previousBddVarCnt = totalBddVarCnt;
            totalBddVarCnt += currentBddVarCnt;

            // Fill grid row.
            grid.set(i + 2, 0, fmt("%,d", i + 1));
            grid.set(i + 2, 1, var.getKindText());
            grid.set(i + 2, 2, typeTxt);
            grid.set(i + 2, 3, var.name);
            grid.set(i + 2, 4, fmt("%,d", var.group));
            grid.set(i + 2, 5, fmt("%,d", bddCnt) + " * " + nrOfDomainsPerVarTxt);
            grid.set(i + 2, 6, fmt("%,d", usedCnt) + " * " + nrOfDomainsPerVarTxt);
            grid.set(i + 2, 7, fmt("%,d", reprCnt) + " * " + nrOfDomainsPerVarTxt);
            grid.set(i + 2, 8, getPercentageText(BigInteger.valueOf(usedCnt), reprCnt));
            grid.set(i + 2, 9, fmt("%,d", previousBddVarCnt) + ".." + fmt("%,d", (totalBddVarCnt - 1)));

            // Update totals.
            groups.add(var.group);
            totalUsedCnt = totalUsedCnt.add(BigInteger.valueOf(usedCnt)
                    .multiply(BigInteger.valueOf(nrOfDomainsPerVar)));
            totalReprCnt = totalReprCnt.add(reprCnt.multiply(BigInteger.valueOf(nrOfDomainsPerVar)));

            Assert.check(totalBddVarCnt >= 0, totalBddVarCnt);
        }

        // Fill grid with totals.
        grid.set(cifVarCnt + 3, 0, "Total");
        grid.set(cifVarCnt + 3, 1, "");
        grid.set(cifVarCnt + 3, 2, "");
        grid.set(cifVarCnt + 3, 3, "");
        grid.set(cifVarCnt + 3, 4, fmt("%,d", groups.size()));
        grid.set(cifVarCnt + 3, 5, fmt("%,d", totalBddVarCnt));
        grid.set(cifVarCnt + 3, 6, fmt("%,d", totalUsedCnt));
        grid.set(cifVarCnt + 3, 7, fmt("%,d", totalReprCnt));
        grid.set(cifVarCnt + 3, 8, getPercentageText(totalUsedCnt, totalReprCnt));
        grid.set(cifVarCnt + 3, 9, "0.." + (totalBddVarCnt - 1));

        // Fill separation rows.
        GridBoxLayout layout = grid.computeLayout();
        for (int i = 0; i < layout.numCols; i++) {
            String bar = Strings.duplicate("-", layout.widths[i]);
            grid.set(1, i, bar);
            grid.set(cifVarCnt + 2, i, bar);
        }

        // Print the variable information, for debugging.
        cifBddSpec.settings.getDebugOutput().line("CIF variables and location pointers:");
        cifBddSpec.settings.getDebugOutput().inc();
        for (String line: grid.getLines()) {
            cifBddSpec.settings.getDebugOutput().line(line);
        }
        cifBddSpec.settings.getDebugOutput().dec();
    }

    /**
     * Returns the percentage of a part with respect to a total, as end-user readable text.
     *
     * @param part The part value.
     * @param total The total value.
     * @return The percentage as integer number in the range [0..100], followed by a {@code "%"} sign, and preceded by a
     *     {@code "~"} if the integer percentage is rounded from a non-integer percentage. If the percentage can not be
     *     computed because {@code total} is zero, the result is {@code "n/a"}.
     */
    private static String getPercentageText(BigInteger part, BigInteger total) {
        // If total is zero, there is no percentage.
        if (total.compareTo(BigInteger.ZERO) == 0) {
            return "n/a";
        }

        // Compute percentage: (100 * part) / total. Also compute the remainder.
        BigDecimal left = BigDecimal.valueOf(100.0).multiply(new BigDecimal(part));
        BigInteger percentage = left.divide(new BigDecimal(total), 0, RoundingMode.HALF_UP).toBigIntegerExact();
        BigInteger remainder = left.remainder(new BigDecimal(total)).toBigIntegerExact();

        // Format the percentage as a whole integer number.
        String txt = fmt("%d%%", percentage);

        // If the percentage is not a whole number, indicate that it is approximate.
        if (remainder.compareTo(BigInteger.ZERO) != 0) {
            txt = "~" + txt;
        }

        // Return the text.
        return txt;
    }

    /**
     * Creates and sets BDD {@link CifBddVariable#domain domains} for all the CIF/BDD variables.
     *
     * @param cifBddSpec The CIF/BDD specification.
     */
    private void createVarDomains(CifBddSpec cifBddSpec) {
        // Skip if no variables.
        int varCnt = cifBddSpec.variables.length;
        if (varCnt == 0) {
            return;
        }

        // Make sure the CIF/BDD variable domain interleaving groups are set, ascending and contiguous.
        int cur = 0;
        for (int i = 0; i < varCnt; i++) {
            int group = cifBddSpec.variables[i].group;
            Assert.check(group >= 0);
            if (group == cur) {
                continue;
            }
            if (group == cur + 1) {
                cur = group;
                continue;
            }
            Assert.fail(fmt("Invalid cur/group: %d/%d.", cur, group));
        }

        // Count number of CIF/BDD variables per group of interleaving domains.
        CifBddVariable lastVar = cifBddSpec.variables[varCnt - 1];
        int[] counts = new int[lastVar.group + 1];
        for (int i = 0; i < varCnt; i++) {
            counts[cifBddSpec.variables[i].group]++;
        }

        // Create and set domains, per group of interleaving domains.
        int nrOfExtraDomains = cifBddSpec.settings.getBddExtraVarDomainNames().size();
        int nrOfDomainsPerVar = 2 + nrOfExtraDomains;
        int offset = 0;
        for (int grpIdx = 0; grpIdx < counts.length; grpIdx++) {
            // Get domain lengths, for variables in the group.
            int grpVarCnt = counts[grpIdx];
            int[] lengths = new int[grpVarCnt * nrOfDomainsPerVar];
            for (int varIdx = 0; varIdx < grpVarCnt; varIdx++) {
                int length = cifBddSpec.variables[offset + varIdx].getBddVarCount();
                for (int domainIdx = 0; domainIdx < nrOfDomainsPerVar; domainIdx++) {
                    lengths[(nrOfDomainsPerVar * varIdx) + domainIdx] = length;
                }
            }

            // Create domains.
            CifBddDomain[] domains = cifBddSpec.createDomains(lengths);

            // Set domains.
            for (int varIdx = 0; varIdx < grpVarCnt; varIdx++) {
                CifBddVariable cifBddVar = cifBddSpec.variables[offset + varIdx];
                cifBddVar.domain = domains[(nrOfDomainsPerVar * varIdx) + 0];
                cifBddVar.domainNew = domains[(nrOfDomainsPerVar * varIdx) + 1];
                cifBddVar.domainsExtra = new CifBddDomain[nrOfExtraDomains];
                for (int extraDomainIdx = 0; extraDomainIdx < nrOfExtraDomains; extraDomainIdx++) {
                    cifBddVar.domainsExtra[extraDomainIdx] = domains[(nrOfDomainsPerVar * varIdx) + 2 + extraDomainIdx];
                }
            }

            // Proceed with next group of interleaving domains.
            offset += grpVarCnt;
        }
    }

    /**
     * Create auxiliary data for updates.
     *
     * @param cifBddSpec The CIF/BDD specification. Is modified in-place.
     * @see CifBddSpec#varSetOld
     * @see CifBddSpec#varSetNew
     * @see CifBddSpec#varSetsExtra
     */
    private void createUpdateAuxiliaries(CifBddSpec cifBddSpec) {
        // Sanity check.
        int domainCnt = cifBddSpec.getDomainCount();
        int cifVarCnt = cifBddSpec.variables.length;
        int nrOfExtraDomains = cifBddSpec.settings.getBddExtraVarDomainNames().size();
        int nrOfDomainsPerVar = 2 + nrOfExtraDomains; // '2' for 'old' + 'new'.
        Assert.areEqual(cifVarCnt * nrOfDomainsPerVar, domainCnt);

        // Get old/new domains.
        CifBddDomain[] oldDomains = new CifBddDomain[cifVarCnt];
        CifBddDomain[] newDomains = new CifBddDomain[cifVarCnt];
        for (int i = 0; i < cifVarCnt; i++) {
            oldDomains[i] = cifBddSpec.variables[i].domain;
            newDomains[i] = cifBddSpec.variables[i].domainNew;
        }

        if (cifBddSpec.settings.getTermination().isRequested()) {
            return;
        }

        // varSetOld = {x, y, z, ...}
        // varSetNew = {x+, y+, z+, ...}
        // varSetsExtra = {x0, y0, z0, ...}, {x1, y1, x1, ...}, ...
        int bddVarCnt = cifBddSpec.factory.varNum();
        Assert.check(bddVarCnt % nrOfDomainsPerVar == 0);
        int bddVarCntPerDomain = bddVarCnt / nrOfDomainsPerVar;
        int[] varIdxsOld = new int[bddVarCntPerDomain];
        int[] varIdxsNew = new int[bddVarCntPerDomain];
        int[][] varIdxsExtra = new int[nrOfExtraDomains][bddVarCntPerDomain];
        int varIdx = 0;
        for (int i = 0; i < cifVarCnt; i++) {
            CifBddDomain oldDomain = oldDomains[i];
            CifBddDomain newDomain = newDomains[i];
            int[] domainVarIdxsOld = oldDomain.getVarIndices();
            int[] domainVarIdxsNew = newDomain.getVarIndices();
            System.arraycopy(domainVarIdxsOld, 0, varIdxsOld, varIdx, domainVarIdxsOld.length);
            System.arraycopy(domainVarIdxsNew, 0, varIdxsNew, varIdx, domainVarIdxsNew.length);

            CifBddDomain[] extraDomains = cifBddSpec.variables[i].domainsExtra;
            for (int j = 0; j < extraDomains.length; j++) {
                CifBddDomain extraDomain = extraDomains[j];
                int[] domainVarIdxsExtra = extraDomain.getVarIndices();
                System.arraycopy(domainVarIdxsExtra, 0, varIdxsExtra[j], varIdx, domainVarIdxsExtra.length);
            }

            varIdx += domainVarIdxsOld.length;
        }
        cifBddSpec.varSetOld = cifBddSpec.factory.makeSet(varIdxsOld);
        cifBddSpec.varSetNew = cifBddSpec.factory.makeSet(varIdxsNew);
        cifBddSpec.varSetsExtra = Collections.unmodifiableList(
                Arrays.stream(varIdxsExtra).map(idxs -> cifBddSpec.factory.makeSet(idxs)).collect(Lists.toList()));

        if (cifBddSpec.settings.getTermination().isRequested()) {
            return;
        }
    }

    /**
     * Converts initialization predicates from the components, initialization predicates from the locations of automata,
     * and the initial values of the variables.
     *
     * @param comp The component for which to convert initialization, recursively.
     * @param cifBddSpec The CIF/BDD specification to be updated with initialization information.
     * @param locPtrManager Location pointer manager.
     */
    private void convertInit(ComplexComponent comp, CifBddSpec cifBddSpec, LocationPointerManager locPtrManager) {
        // Initialization predicates of the component.
        for (Expression pred: comp.getInitials()) {
            // Convert.
            BDD initial = convertPred(pred, true, cifBddSpec, null);

            // Store.
            cifBddSpec.initialsComps.add(initial);
            cifBddSpec.initialComps = cifBddSpec.initialComps.andWith(initial.id());
        }

        // Initial values of discrete variables (automata only).
        if (comp instanceof Automaton) {
            for (Declaration cifDecl: comp.getDeclarations()) {
                // Skip all but discrete variables.
                if (!(cifDecl instanceof DiscVariable)) {
                    continue;
                }
                DiscVariable cifVar = (DiscVariable)cifDecl;

                // Get CIF/BDD variable.
                int varIdx = getDiscVarIdx(cifBddSpec.variables, cifVar);
                Assert.check(varIdx >= 0);
                CifBddVariable cifBddVar = cifBddSpec.variables[varIdx];
                Assert.check(cifBddVar instanceof CifBddDiscVariable);
                CifBddDiscVariable var = (CifBddDiscVariable)cifBddVar;

                // Get initial value expressions. Use 'null' to indicate any value in the CIF domain.
                List<Expression> values;
                if (cifVar.getValue() == null) {
                    // Default initial value.
                    CifType type = cifVar.getType();
                    values = list(CifValueUtils.getDefaultValue(type, null));
                } else if (cifVar.getValue().getValues().isEmpty()) {
                    // Any value in its domain.
                    values = null;
                } else {
                    // One or more specific initialization values.
                    values = cifVar.getValue().getValues();
                }

                // Create initialization predicate for the discrete variable.
                BDD pred;
                if (values == null) {
                    // Any value in its domain.
                    pred = BddUtils.getCifVarDomainBdd(var, false, cifBddSpec.factory);
                } else {
                    // Specific values.
                    pred = cifBddSpec.factory.zero();
                    for (Expression value: values) {
                        // Case distinction on types of values.
                        if (var.type instanceof BoolType) {
                            // Convert right hand side (value to assign).
                            BDD valueBdd = convertPred(value, true, cifBddSpec, null);

                            // Create BDD for the left hand side (variable to get a new value).
                            Assert.check(var.domain.getVarCount() == 1);
                            int varVar = var.domain.getVarIndices()[0];
                            BDD varBdd = cifBddSpec.factory.ithVar(varVar);

                            // Construct 'var = value' relation.
                            BDD relation = varBdd.biimpWith(valueBdd);

                            // Update initialization predicate for the variable.
                            pred = pred.orWith(relation);
                        } else {
                            // Get bit vectors for the variable and its initial value.
                            BddBitVector<?, ?> varVector = var.createBitVector(false);
                            BddBitVector<?, ?> valueVector = convertExpr(value, true, cifBddSpec, null);

                            // The CIF type checker ensures that the type of the value is contained in the type of the
                            // discrete variable, and thus always fits in the bit vector representation. There are thus
                            // no runtime errors.

                            // The representations of the variable and value bit vectors can be different. For instance,
                            // a signed variable may be assigned an non-negative (unsigned) value. And for an unsigned
                            // variable, the value may be computed in such a way that we get a signed bit vector.
                            // Therefore, we ensure the representations are compatible.
                            Pair<BddBitVector<?, ?>, BddBitVector<?, ?>> vectors = BddBitVector
                                    .ensureCompatible(varVector, valueVector);
                            varVector = vectors.left;
                            valueVector = vectors.right;

                            // Resize the variable and value vectors to have the same length, such that an equality
                            // operation can be performed.
                            BddBitVector.ensureSameLength(varVector, valueVector);

                            // Construct 'var = value' relation.
                            BDD relation = varVector.equalToAny(valueVector);

                            // Cleanup.
                            varVector.free();
                            valueVector.free();

                            // Update initialization predicate for the variable.
                            pred = pred.orWith(relation);
                        }
                    }
                }

                // Store initialization.
                cifBddSpec.initialsVars.set(varIdx, pred);
                cifBddSpec.initialVars = cifBddSpec.initialVars.andWith(pred.id());
            }
        }

        // Initialization predicates of locations (automata only).
        if (comp instanceof Automaton aut) {
            // Combine initialization predicates from the locations.
            BDD autInit = cifBddSpec.factory.zero();
            for (Location loc: aut.getLocations()) {
                // Skip location without initialization predicates (implicit 'false').
                if (loc.getInitials().isEmpty()) {
                    continue;
                }

                // Convert initialization predicates of the location.
                List<Expression> locInits = loc.getInitials();
                BDD locInit = convertPreds(locInits, true, cifBddSpec, null);

                // Add location predicate.
                Expression srcLocRef = locPtrManager.createLocRef(loc);
                BDD srcLocPred = convertPred(srcLocRef, true, cifBddSpec, null);
                locInit = locInit.and(srcLocPred);

                // Combine with initialization predicates of other locations.
                autInit = autInit.or(locInit);
            }

            // Store.
            cifBddSpec.initialsLocs.add(autInit);
            cifBddSpec.initialLocs = cifBddSpec.initialLocs.andWith(autInit.id());
        }

        // Proceed recursively (groups only).
        if (comp instanceof Group group) {
            for (Component child: group.getComponents()) {
                convertInit((ComplexComponent)child, cifBddSpec, locPtrManager);
            }
        }
    }

    /**
     * Converts marker predicates from the components and marker predicates from the locations of automata.
     *
     * @param comp The component for which to convert marking, recursively.
     * @param cifBddSpec The CIF/BDD specification to be updated with marking information.
     * @param locPtrManager Location pointer manager.
     */
    private void convertMarked(ComplexComponent comp, CifBddSpec cifBddSpec, LocationPointerManager locPtrManager) {
        // Marker predicates of the component.
        for (Expression pred: comp.getMarkeds()) {
            // Convert.
            BDD marked = convertPred(pred, false, cifBddSpec, null);

            // Store.
            cifBddSpec.markedsComps.add(marked);
            cifBddSpec.markedComps = cifBddSpec.markedComps.andWith(marked.id());
        }

        // Marker predicates of locations (automata only).
        if (comp instanceof Automaton aut) {
            // Combine marker predicates from the locations.
            BDD autMarked = cifBddSpec.factory.zero();
            for (Location loc: aut.getLocations()) {
                // Skip location without marker predicates (implicit 'false').
                if (loc.getMarkeds().isEmpty()) {
                    continue;
                }

                // Convert marker predicates of the location.
                List<Expression> locMarkeds = loc.getMarkeds();
                BDD locMarked = convertPreds(locMarkeds, false, cifBddSpec, null);

                // Add location predicate.
                Expression srcLocRef = locPtrManager.createLocRef(loc);
                BDD srcLocPred = convertPred(srcLocRef, false, cifBddSpec, null);
                locMarked = locMarked.andWith(srcLocPred);

                // Combine with marker predicates of other locations.
                autMarked = autMarked.orWith(locMarked);
            }

            // Store.
            cifBddSpec.markedsLocs.add(autMarked);
            cifBddSpec.markedLocs = cifBddSpec.markedLocs.andWith(autMarked.id());
        }

        // Proceed recursively (groups only).
        if (comp instanceof Group group) {
            for (Component child: group.getComponents()) {
                convertMarked((ComplexComponent)child, cifBddSpec, locPtrManager);
            }
        }
    }

    /**
     * Converts state invariants (predicates) from the components and the locations of automata.
     *
     * @param comp The component for which to convert state invariants (predicates), recursively.
     * @param cifBddSpec The CIF/BDD specification to be updated with state invariants (predicates) information.
     * @param locPtrManager Location pointer manager.
     */
    private void convertStateInvs(ComplexComponent comp, CifBddSpec cifBddSpec, LocationPointerManager locPtrManager) {
        // State invariants (predicates) of the component.
        for (Invariant inv: comp.getInvariants()) {
            // Skip non-state invariants.
            if (inv.getInvKind() != InvKind.STATE) {
                continue;
            }

            // Convert.
            Expression pred = inv.getPredicate();
            BDD invComp = convertPred(pred, false, cifBddSpec, null);

            // Store.
            switch (inv.getSupKind()) {
                case PLANT:
                    cifBddSpec.plantInvsComps.add(invComp);
                    cifBddSpec.plantInvComps = cifBddSpec.plantInvComps.andWith(invComp.id());
                    break;
                case REQUIREMENT:
                    cifBddSpec.reqInvsComps.add(invComp);
                    cifBddSpec.reqInvComps = cifBddSpec.reqInvComps.andWith(invComp.id());
                    break;
                default:
                    throw new RuntimeException("Unexpected kind: " + inv.getSupKind());
            }
        }

        // State invariants (predicates) of locations (automata only).
        if (comp instanceof Automaton aut) {
            // Add state invariants (predicates) from the locations.
            for (Location loc: aut.getLocations()) {
                for (Invariant inv: loc.getInvariants()) {
                    // Skip non-state invariants.
                    if (inv.getInvKind() != InvKind.STATE) {
                        continue;
                    }

                    // Convert.
                    Expression pred = inv.getPredicate();
                    BDD invLoc = convertPred(pred, false, cifBddSpec, null);

                    // Add location predicate (srcLocPred => locInv).
                    Expression srcLocRef = locPtrManager.createLocRef(loc);
                    BDD srcLocPred = convertPred(srcLocRef, false, cifBddSpec, null);
                    invLoc = srcLocPred.not().orWith(invLoc);
                    srcLocPred.free();

                    // Store.
                    switch (inv.getSupKind()) {
                        case PLANT:
                            cifBddSpec.plantInvsLocs.add(invLoc);
                            cifBddSpec.plantInvLocs = cifBddSpec.plantInvLocs.andWith(invLoc.id());
                            break;
                        case REQUIREMENT:
                            cifBddSpec.reqInvsLocs.add(invLoc);
                            cifBddSpec.reqInvLocs = cifBddSpec.reqInvLocs.andWith(invLoc.id());
                            break;
                        default:
                            throw new RuntimeException("Unexpected kind: " + inv.getSupKind());
                    }
                }
            }
        }

        // Proceed recursively (groups only).
        if (comp instanceof Group group) {
            for (Component child: group.getComponents()) {
                convertStateInvs((ComplexComponent)child, cifBddSpec, locPtrManager);
            }
        }
    }

    /**
     * Converts state/event exclusion invariants (both plant and requirement) from the components and the locations of
     * automata.
     *
     * @param comp The component for which to convert state/event exclusion invariants, recursively.
     * @param cifBddSpec The CIF/BDD specification to be updated with state/event exclusion invariants information.
     * @param locPtrManager Location pointer manager.
     */
    private void convertStateEvtExclInvs(ComplexComponent comp, CifBddSpec cifBddSpec,
            LocationPointerManager locPtrManager)
    {
        // State/event exclusion invariants of the component.
        for (Invariant inv: comp.getInvariants()) {
            // Skip state invariants.
            if (inv.getInvKind() == InvKind.STATE) {
                continue;
            }

            // Check that event is in the alphabet.
            Event event = ((EventExpression)inv.getEvent()).getEvent();
            if (!cifBddSpec.alphabet.contains(event)) {
                // Skip the rest as we won't use this invariant any further.
                continue;
            }

            // Convert predicate.
            Expression pred = inv.getPredicate();
            BDD compInv = convertPred(pred, false, cifBddSpec, null);

            // Adapt predicate for the kind of invariant.
            if (inv.getInvKind() == InvKind.EVENT_DISABLES) {
                BDD compInvNot = compInv.not();
                compInv.free();
                compInv = compInvNot;
            }

            // Store copies of the BDD.
            switch (inv.getSupKind()) {
                case PLANT:
                    storeStateEvtExclInv(cifBddSpec.stateEvtExclPlantLists, event, compInv.id());
                    conjunctAndStoreStateEvtExclInv(cifBddSpec.stateEvtExclPlants, event, compInv.id());
                    break;
                case REQUIREMENT:
                    storeStateEvtExclInv(cifBddSpec.stateEvtExclReqLists, event, compInv.id());
                    conjunctAndStoreStateEvtExclInv(cifBddSpec.stateEvtExclReqs, event, compInv.id());
                    if (event.getControllable()) {
                        conjunctAndStoreStateEvtExclInv(cifBddSpec.stateEvtExclsReqInvs, event, compInv.id());
                    }
                    break;
                default:
                    throw new RuntimeException("Unexpected kind: " + inv.getSupKind());
            }

            // Free the original BDD.
            compInv.free();
        }

        // State/event exclusion requirement invariants of locations (automata only).
        if (comp instanceof Automaton aut) {
            // Add state/event exclusion invariants from the locations.
            for (Location loc: aut.getLocations()) {
                for (Invariant inv: loc.getInvariants()) {
                    // Skip state invariants.
                    if (inv.getInvKind() == InvKind.STATE) {
                        continue;
                    }

                    // Check that event is in the alphabet.
                    Event event = ((EventExpression)inv.getEvent()).getEvent();
                    if (!cifBddSpec.alphabet.contains(event)) {
                        // Skip the rest as we won't use this invariant any further.
                        continue;
                    }

                    // Convert predicate.
                    Expression pred = inv.getPredicate();
                    BDD locInv = convertPred(pred, false, cifBddSpec, null);

                    // Get location predicate (srcLocPred => locInv).
                    Expression srcLocRef = locPtrManager.createLocRef(loc);
                    BDD srcLocPred = convertPred(srcLocRef, false, cifBddSpec, null);
                    locInv = srcLocPred.not().orWith(locInv);
                    srcLocPred.free();

                    // Adapt predicate for the kind of invariant.
                    if (inv.getInvKind() == InvKind.EVENT_DISABLES) {
                        BDD locInvNot = locInv.not();
                        locInv.free();
                        locInv = locInvNot;
                    }

                    // Store copies of the BDD.
                    switch (inv.getSupKind()) {
                        case PLANT:
                            storeStateEvtExclInv(cifBddSpec.stateEvtExclPlantLists, event, locInv.id());
                            conjunctAndStoreStateEvtExclInv(cifBddSpec.stateEvtExclPlants, event, locInv.id());
                            break;
                        case REQUIREMENT:
                            storeStateEvtExclInv(cifBddSpec.stateEvtExclReqLists, event, locInv.id());
                            conjunctAndStoreStateEvtExclInv(cifBddSpec.stateEvtExclReqs, event, locInv.id());
                            if (event.getControllable()) {
                                conjunctAndStoreStateEvtExclInv(cifBddSpec.stateEvtExclsReqInvs, event, locInv.id());
                            }
                            break;
                        default:
                            throw new RuntimeException("Unexpected kind: " + inv.getSupKind());
                    }

                    // Free the original BDD.
                    locInv.free();
                }
            }
        }

        // Proceed recursively (groups only).
        if (comp instanceof Group group) {
            for (Component child: group.getComponents()) {
                convertStateEvtExclInvs((ComplexComponent)child, cifBddSpec, locPtrManager);
            }
        }
    }

    /**
     * Adds the given state/event exclusion invariant to the state/event exclusion invariants collected so far for the
     * supplied event. The invariants are retrieved from the supplied mapping.
     *
     * @param eventInvs Mapping from events to their corresponding invariants.
     * @param event The event to use as a key.
     * @param inv The invariant to add.
     */
    private void storeStateEvtExclInv(Map<Event, List<BDD>> eventInvs, Event event, BDD inv) {
        List<BDD> invs = eventInvs.get(event);
        invs.add(inv);
    }

    /**
     * Combines the given state/event exclusion invariant with the state/event exclusion invariants collected so far for
     * the supplied event. The invariants are combined, using conjunction. The invariants (as a BDD) are retrieved from
     * the supplied mapping.
     *
     * @param eventInvs Mapping from events to their corresponding invariants.
     * @param event The event to use as a key.
     * @param inv The invariant to combine.
     */
    private void conjunctAndStoreStateEvtExclInv(Map<Event, BDD> eventInvs, Event event, BDD inv) {
        BDD invs = eventInvs.get(event);
        invs = invs.andWith(inv);
        eventInvs.put(event, invs);
    }

    /**
     * Preconvert requirement automata, to enable treating them as monitor plants in the CIF/BDD representation.
     *
     * @param requirements The requirement automata.
     * @param alphabets Per requirement automaton, all the alphabets.
     * @param cifBddSpec The CIF/BDD specification.
     */
    private void preconvertReqAuts(List<Automaton> requirements, List<Alphabets> alphabets, CifBddSpec cifBddSpec) {
        // Initialization.
        originalMonitors = mapc(requirements.size());

        // For CIF/BDD, requirement automata are treated as plants that monitor the entire alphabet. They thus don't
        // restrict anything guard-wise. We add additional state/event exclusion requirements to restrict the behavior
        // to what the original requirement automaton allowed.
        for (int i = 0; i < requirements.size(); i++) {
            // Get requirement automaton and alphabets.
            Automaton requirement = requirements.get(i);
            Alphabets reqAlphabets = alphabets.get(i);

            // Add state/event exclusion requirements for non-monitored events.
            for (Event event: reqAlphabets.syncAlphabet) {
                // Skip events that are already monitored.
                if (reqAlphabets.moniAlphabet.contains(event)) {
                    continue;
                }

                // Get combined guard.
                Expression cifGuard = CifGuardUtils.mergeGuards(requirement, event, EdgeEventImpl.class,
                        LocRefExprCreator.DEFAULT);

                // Convert guard.
                BDD cifBddGuard = convertPred(cifGuard, false, cifBddSpec, null);

                // Add guard as state/event exclusion requirement for the event.
                storeStateEvtExclInv(cifBddSpec.stateEvtExclReqLists, event, cifBddGuard.id());
                conjunctAndStoreStateEvtExclInv(cifBddSpec.stateEvtExclReqs, event, cifBddGuard.id());

                if (event.getControllable()) {
                    conjunctAndStoreStateEvtExclInv(cifBddSpec.stateEvtExclsReqAuts, event, cifBddGuard.id());
                }

                cifBddGuard.free();
            }

            // Change requirement automaton to monitor all events. Skip this if the alphabet is empty, as we then get a
            // warning that we monitor an empty alphabet, when the input specification is converted to the output
            // specification, saved on disk, and used with other tools.
            if (reqAlphabets.syncAlphabet.isEmpty()) {
                // No alphabet, so shouldn't monitor anything. It may however already monitor the entire (empty)
                // alphabet. If so, just leave that as is.
            } else {
                // Store the original monitors, to be able to restore them later on.
                originalMonitors.put(requirement, requirement.getMonitors());

                // Monitor all events in the alphabet.
                requirement.setMonitors(newMonitors());
                reqAlphabets.moniAlphabet = copy(reqAlphabets.syncAlphabet);
            }
        }
    }

    /**
     * Converts the plant and requirement automata, to a single linearized CIF/BDD specification.
     *
     * @param plants The plant automata of the specification.
     * @param requirements The requirement automata.
     * @param plantAlphabets Per plant automaton, all the alphabets.
     * @param reqAlphabets Per requirement automaton, all the alphabets.
     * @param locPtrManager Location pointer manager.
     * @param cifBddSpec The CIF/BDD specification to be updated.
     * @throws UnsupportedException In case the specification has non-determinism that is not supported.
     */
    private void convertPlantReqAuts(List<Automaton> plants, List<Automaton> requirements,
            List<Alphabets> plantAlphabets, List<Alphabets> reqAlphabets, CifBddLocationPointerManager locPtrManager,
            CifBddSpec cifBddSpec)
    {
        // Get the automata (with their alphabet) and events to consider.
        List<Automaton> automata = concat(plants, requirements);
        List<Alphabets> alphabets = concat(plantAlphabets, reqAlphabets);
        List<Event> events = set2list(cifBddSpec.alphabet);

        // If adherence to the execution scheme prescribed by the controller properties checker is requested, order the
        // automata, alphabets and events.
        if (cifBddSpec.settings.getAdhereToExecScheme()) {
            Assert.areEqual(automata.size(), alphabets.size());
            Map<Automaton, Alphabets> alphabetsPerAut = mapc(alphabets.size());
            for (int i = 0; i < automata.size(); i++) {
                alphabetsPerAut.put(automata.get(i), alphabets.get(i));
            }

            CifExecSchemeUtils.orderAutomata(automata);
            alphabets = automata.stream().map(aut -> alphabetsPerAut.get(aut)).toList();
            CifExecSchemeUtils.orderEvents(events);
        }

        // Linearize edges for all events in the alphabet.
        // Must match a similar call to linearize edges in 'LinearizedHyperEdgeCreator'.
        // We use linearize product (with merge), which adheres to the execution scheme prescribed by the controller
        // properties checker.
        List<LinearizedEdgeIterator> linearizedEdgeIters = LinearizeProduct.linearizeEdges(automata, alphabets, events,
                locPtrManager, false, true, true);

        // Create and add CIF/BDD edges.
        long expectedSizeLong = linearizedEdgeIters.stream().map(iter -> iter.getResultSize())
                .filter(size -> size.isEmpty()).map(size -> size.get()).reduce(0L, Long::sum);
        int expectedSizeInt = (int)Math.min(Lists.MAX_SAFE_LIST_SIZE, Math.max(0, expectedSizeLong));
        cifBddSpec.edges = listc(expectedSizeInt);
        cifBddSpec.eventEdges = mapc(cifBddSpec.alphabet.size());
        for (LinearizedEdgeIterator linearizedEdgeIter: linearizedEdgeIters) {
            while (linearizedEdgeIter.hasNext()) {
                // Check for termination.
                if (cifBddSpec.settings.getTermination().isRequested()) {
                    break;
                }

                // Get next linearized CIF edge.
                Edge cifEdge = linearizedEdgeIter.next();

                // Create CIF/BDD edge.
                CifBddEdge cifBddEdge = new CifBddEdge(cifBddSpec);
                cifBddEdge.edges = list(cifEdge);

                // Set event.
                Assert.check(cifEdge.getEvents().size() == 1);
                EdgeEvent edgeEvent = first(cifEdge.getEvents());
                Event event = CifEventUtils.getEventFromEdgeEvent(edgeEvent);
                cifBddEdge.event = event;

                // Add edge.
                cifBddSpec.edges.add(cifBddEdge);
                List<CifBddEdge> cifBddEdges = cifBddSpec.eventEdges.get(event);
                if (cifBddEdges == null) {
                    cifBddEdges = list();
                    cifBddSpec.eventEdges.put(event, cifBddEdges);
                }
                cifBddEdges.add(cifBddEdge);

                // Convert and set guards.
                BDD guard = convertPreds(cifEdge.getGuards(), false, cifBddSpec, null);
                cifBddEdge.guard = guard;
                cifBddEdge.origGuard = guard.id();

                // Convert and set assignments.
                List<Update> updates = cifEdge.getUpdates();
                convertUpdates(updates, cifBddEdge, locPtrManager, cifBddSpec);
                if (cifBddSpec.settings.getTermination().isRequested()) {
                    return;
                }

                // Strengthen the guard to prevent runtime errors.
                cifBddEdge.guard = cifBddEdge.guard.andWith(cifBddEdge.error.not());
            }
        }

        if (cifBddSpec.settings.getTermination().isRequested()) {
            return;
        }

        // Check for non-determinism.
        checkNonDeterminism(cifBddSpec.edges, cifBddSpec.settings.getAllowNonDeterminism());
    }

    /**
     * Check CIF/BDD edges to make sure there is no non-determinism, i.e., non-determinism by means of multiple outgoing
     * edges for the same event, with overlapping guards.
     *
     * @param edges The CIF/BDD edges (self loops). May include edges for both controllable and uncontrollable events.
     * @param allowNonDeterminism Events for which to allow non-determinism.
     * @throws UnsupportedException In case the specification has non-determinism that is not supported.
     */
    private void checkNonDeterminism(List<CifBddEdge> edges, AllowNonDeterminism allowNonDeterminism) {
        // Initialize conflict information.
        Map<Event, BDD> eventGuards = map();
        Set<Event> conflicts = setc(0);

        // Check edges for conflicts (non-determinism).
        for (CifBddEdge edge: edges) {
            // Skip events for which non-determinism is allowed.
            Event evt = edge.event;
            boolean controllable = evt.getControllable();
            if (allowNonDeterminism.allowFor(controllable)) {
                continue;
            }

            // Skip events already found to have conflicting edges.
            if (conflicts.contains(evt)) {
                continue;
            }

            // Get guards so far and new guard of the edge.
            BDD curGuard = eventGuards.get(evt);
            BDD newGuard = edge.guard;

            // Check for overlap.
            if (curGuard == null) {
                // First edge for this event.
                eventGuards.put(evt, newGuard.id());
            } else {
                BDD overlap = curGuard.and(newGuard);
                if (overlap.isZero()) {
                    // No overlap, update guard so far.
                    eventGuards.put(evt, curGuard.orWith(newGuard.id()));
                } else {
                    // Overlap. Conflict found.
                    conflicts.add(evt);
                }
                overlap.free();
            }
        }

        // Clean up.
        for (BDD guard: eventGuards.values()) {
            guard.free();
        }

        // Report conflicts.
        Set<String> problems = setc(conflicts.size());
        for (Event conflict: conflicts) {
            // Get edges for the event.
            List<CifBddEdge> eventEdges = list();
            for (CifBddEdge edge: edges) {
                if (edge.event != conflict) {
                    continue;
                }
                eventEdges.add(edge);
            }

            // Partition guards into non-overlapping groups.
            List<List<CifBddEdge>> groups = groupOnGuardOverlap(eventEdges);

            // Get guard texts per group with overlap.
            List<String> guardsTxts = list();
            for (List<CifBddEdge> group: groups) {
                // Only overlap in the group if at least two edges.
                if (group.size() < 2) {
                    continue;
                }

                // Add guards text for this group.
                List<String> guardTxts = list();
                for (CifBddEdge edge: group) {
                    Assert.check(edge.edges.size() == 1); // No CIF/BDD edges have been merged yet.
                    List<Expression> guards = first(edge.edges).getGuards();
                    String guardsTxt;
                    if (guards.isEmpty()) {
                        guardsTxt = "true";
                    } else {
                        guardsTxt = CifTextUtils.exprsToStr(guards);
                    }
                    guardTxts.add("\"" + guardsTxt + "\"");
                }
                Assert.check(!guardTxts.isEmpty());
                guardsTxts.add(String.join(", ", guardTxts));
            }
            Assert.check(!guardsTxts.isEmpty());

            // Get groups text.
            String groupsTxt;
            if (guardsTxts.size() == 1) {
                groupsTxt = " " + guardsTxts.get(0) + ".";
            } else {
                for (int i = 0; i < guardsTxts.size(); i++) {
                    String txt = guardsTxts.get(i);
                    txt = fmt("\n    Group %d: %s", i + 1, txt);
                    guardsTxts.set(i, txt);
                }
                groupsTxt = String.join("", guardsTxts);
            }

            // Add problem.
            String eventKind = switch (allowNonDeterminism) {
                case ALL -> throw new AssertionError("Should not get here, as non-determinism is allowed.");
                case NONE -> "";
                case CONTROLLABLE -> "uncontrollable ";
                case UNCONTROLLABLE -> "controllable ";
            };
            String msg = fmt("Unsupported linearized edges with non-determinism detected for edges of "
                    + "%sevent \"%s\" with overlapping guards:%s", eventKind, getAbsName(conflict), groupsTxt);
            problems.add(msg);
        }

        // Report problems.
        if (problems.isEmpty()) {
            return;
        }

        String msg = fmt("%s failed due to unsatisfied preconditions:\n - ", appName)
                + String.join("\n - ", Sets.sortedstrings(problems));
        throw new UnsupportedException(msg);
    }

    /**
     * Group the edges with overlapping guards. That is, partition the edges into groups with non-overlapping guards.
     *
     * @param edges The edges.
     * @return The groups of edges.
     */
    private static List<List<CifBddEdge>> groupOnGuardOverlap(List<CifBddEdge> edges) {
        // Initialize to one edge per group.
        List<List<CifBddEdge>> groups = listc(edges.size());
        for (int i = 0; i < edges.size(); i++) {
            groups.add(list(edges.get(i)));
        }

        // Merge groups with overlapping guards. For each group, we merge with all overlapping groups that come after
        // it.
        for (int i = 0; i < groups.size(); i++) {
            // All groups start with exactly one edge, so get that guard.
            Assert.check(groups.get(i).size() == 1);
            BDD curGuard = groups.get(i).get(0).guard.id();

            // Process all groups that come after the current one.
            boolean changed = true;
            while (changed) {
                changed = false;

                for (int j = i + 1; j < groups.size(); j++) {
                    // All groups start with exactly one edge, so get that guard.
                    Assert.check(groups.get(j).size() == 1);
                    BDD newGuard = groups.get(j).get(0).guard;

                    // If disjoint (no overlap), groups don't need to be merged.
                    BDD overlapPred = curGuard.and(newGuard);
                    boolean disjoint = overlapPred.isZero();
                    overlapPred.free();
                    if (disjoint) {
                        continue;
                    }

                    // Overlap detected. Merge groups.
                    changed = true;
                    groups.get(i).add(groups.get(j).get(0));
                    groups.remove(j);

                    curGuard = curGuard.andWith(newGuard.id());
                }
            }

            // Cleanup.
            curGuard.free();
        }

        // Return the disjoint groups.
        return groups;
    }

    /**
     * Converts CIF updates to CIF/BDD updates.
     *
     * @param updates The CIF updates.
     * @param cifBddEdge The CIF/BDD edge in which to store the CIF/BDD updates. Is modified in-place.
     * @param locPtrManager Location pointer manager.
     * @param cifBddSpec The CIF/BDD specification.
     */
    public void convertUpdates(List<Update> updates, CifBddEdge cifBddEdge, CifBddLocationPointerManager locPtrManager,
            CifBddSpec cifBddSpec)
    {
        // Initialization.
        boolean[] assigned = new boolean[cifBddSpec.variables.length];

        // Convert separate updates, and merge to form the edge's update relation and runtime error predicate.
        BDD relation = cifBddSpec.factory.one();
        BDD error = cifBddSpec.factory.zero();
        for (Update update: updates) {
            // Convert update.
            List<Pair<BDD, BDD>> conversionResults = convertUpdate(update, assigned, locPtrManager, cifBddSpec);
            if (cifBddSpec.settings.getTermination().isRequested()) {
                return;
            }

            // Merge for each assigned variable.
            for (Pair<BDD, BDD> conversionResult: conversionResults) {
                // Add the variable's update relation to the edge's update relation.
                BDD updateRelation = conversionResult.left;
                relation = relation.andWith(updateRelation);

                if (cifBddSpec.settings.getTermination().isRequested()) {
                    return;
                }

                // Add error to edge error predicate.
                BDD updateError = conversionResult.right;
                error = error.orWith(updateError);

                if (cifBddSpec.settings.getTermination().isRequested()) {
                    return;
                }
            }
        }

        // Collect all BDD variables that are being assigned on this edge.
        for (int i = 0; i < assigned.length; i++) {
            if (assigned[i]) {
                cifBddEdge.assignedVariables.add(cifBddSpec.variables[i]);
            }
        }

        // Store data for the updates.
        cifBddEdge.updates = list(copy(updates));
        cifBddEdge.update = relation;
        cifBddEdge.error = error;
    }

    /**
     * Converts a CIF update to an update predicate and runtime error predicate per (possibly-)assigned variable.
     *
     * @param update The CIF update to convert.
     * @param assigned Per CIF/BDD variable, whether it has been assigned on this edge. Is modified in-place.
     * @param locPtrManager Location pointer manager.
     * @param cifBddSpec The CIF/BDD specification.
     * @return Per (possibly-)assigned variable, its update relation and runtime error predicate.
     */
    public List<Pair<BDD, BDD>> convertUpdate(Update update, boolean[] assigned,
            CifBddLocationPointerManager locPtrManager, CifBddSpec cifBddSpec)
    {
        // Get per (possibly-)assigned variable, its new value expression.
        Map<Declaration, Expression> newValueExprPerVar;
        try {
            newValueExprPerVar = CifUpdateUtils.updateToNewValueExprPerVar(update);
        } catch (UnsupportedUpdateException e) {
            // Preconditions should have prevented these cases.
            throw new AssertionError("Unexpected unsupported update.", e);
        }

        // Convert the 'assignment' for each variable.
        List<Pair<BDD, BDD>> result = listc(newValueExprPerVar.size());
        for (Entry<Declaration, Expression> newValueExprForVar: newValueExprPerVar.entrySet()) {
            DiscVariable discVar = (DiscVariable)newValueExprForVar.getKey();
            Expression newValueExpr = newValueExprForVar.getValue();
            Pair<BDD, BDD> relationAndError = convertAssignment(discVar, newValueExpr, assigned, locPtrManager,
                    cifBddSpec);
            result.add(relationAndError);

            if (cifBddSpec.settings.getTermination().isRequested()) {
                return result;
            }
        }
        return result;
    }

    /**
     * Converts a CIF assignment to an update predicate and runtime error predicate.
     *
     * @param cifVar The CIF variable being assigned.
     * @param newValue The expression indicating the variable's new value.
     * @param assigned Per CIF/BDD variable, whether it has been assigned on this edge. Is modified in-place.
     * @param locPtrManager Location pointer manager.
     * @param cifBddSpec The CIF/BDD specification.
     * @return The update relation and runtime error predicate.
     */
    public Pair<BDD, BDD> convertAssignment(DiscVariable cifVar, Expression newValue, boolean[] assigned,
            CifBddLocationPointerManager locPtrManager, CifBddSpec cifBddSpec)
    {
        // Special case for location pointer variable assignments created during linearization. Note that location
        // pointers are only created for automata with more than one location, and updates are only created for non self
        // loop edges. Since automata with one location have only self loops, automata for which location pointer
        // updates are created also have location pointer variables.
        Automaton cifAut = locPtrManager.getAutomaton(cifVar);
        if (cifAut != null) {
            // Get CIF/BDD variable.
            int varIdx = getLpVarIdx(cifBddSpec.variables, cifAut);
            Assert.check(varIdx >= 0);
            CifBddVariable var = cifBddSpec.variables[varIdx];
            Assert.check(var instanceof CifBddLocPtrVariable);

            // Mark variable as assigned.
            Assert.check(!assigned[varIdx]);
            assigned[varIdx] = true;

            // Convert the location pointer variable (lp) and its new value expression (v) to an update relation
            // (lp+ = v).
            BDD relation = convertLocPtrAssignment((CifBddLocPtrVariable)var, newValue, cifBddSpec);

            // Return the update relation and runtime error predicate. The location always fits within the domain of
            // the location pointer variable, so there is no error predicate.
            return pair(relation, cifBddSpec.factory.zero());
        }

        // Normal case: assignment originating from original CIF model.
        int varIdx = getTypedVarIdx(cifBddSpec.variables, cifVar);
        Assert.check(varIdx >= 0);
        CifBddVariable cifBddVar = cifBddSpec.variables[varIdx];
        Assert.check(cifBddVar instanceof CifBddTypedVariable);
        CifBddTypedVariable var = (CifBddTypedVariable)cifBddVar;

        // Mark variable as assigned.
        Assert.check(!assigned[varIdx]);
        assigned[varIdx] = true;

        // Case distinction on types of values.
        if (var.type instanceof BoolType) {
            // Convert right hand side (value to assign).
            BDD rhsBdd = convertPred(newValue, false, cifBddSpec, null);

            // Create BDD for the left hand side (variable to get a new value).
            Assert.check(var.domainNew.getVarCount() == 1);
            int lhsVar = var.domainNew.getVarIndices()[0];
            BDD lhsBdd = cifBddSpec.factory.ithVar(lhsVar);

            // Construct 'lhs+ = rhs' relation.
            BDD relation = lhsBdd.biimpWith(rhsBdd);

            // Return the full relation.
            return pair(relation, cifBddSpec.factory.zero());
        } else {
            // Get lhs (variable) and rhs (value to assign) bit vectors.
            BddBitVector<?, ?> varVector = var.createBitVector(true);
            BddBitVector<?, ?> valueVector = convertExpr(newValue, false, cifBddSpec, null);

            // Get information on kinds of vectors and their lengths, before making them compatible and equal size.
            boolean varVectorIsUnsigned = varVector instanceof UnsignedBddBitVector;
            boolean valueVectorIsUnsigned = valueVector instanceof UnsignedBddBitVector;
            int varVectorLength = varVector.length();
            int valueVectorLength = valueVector.length();

            // Ensure the bit vectors have compatible representations. Converting an unsigned bit vector to a signed
            // bit vector only adds an extra 'false' bit, and thus the existing bits are not changed.
            Pair<BddBitVector<?, ?>, BddBitVector<?, ?>> vectors = BddBitVector.ensureCompatible(
                    varVector, valueVector);
            varVector = vectors.left;
            valueVector = vectors.right;

            // Resize the variable and value vectors to have the same length, such that an equality operation can be
            // performed. Resizing a bit vector only adds extra bits, and thus the existing bits are not changed.
            BddBitVector.ensureSameLength(varVector, valueVector);

            // Construct 'var+ = value' relation.
            BDD relation = varVector.equalToAny(valueVector);

            // Cleanup.
            varVector.free();

            // Prevent out of bounds assignment.
            //
            // We only need to prevent values that can be represented by the rhs/value vector, but cannot be represented
            // by the lhs/variable vector. The values that the rhs/value vector can have that the lhs/variable vector
            // cannot have, but that can be represented by the lhs/variable vector, are already prevented elsewhere by
            // means of the range invariants.
            //
            // Note that we can still access all bits of the original rhs/value bit vector, as they have not been
            // changed above by making these bit vectors compatible and of equal length.
            BDD error = cifBddSpec.factory.zero();
            if (varVectorIsUnsigned) {
                if (valueVectorIsUnsigned) {
                    // Both vectors are unsigned. Too large values of the value vector cannot be represented in the
                    // variable vector. Hence, all extra bits of the value vector are error bits.
                    for (int bitIdx = varVectorLength; bitIdx < valueVectorLength; bitIdx++) {
                        error = error.orWith(valueVector.getBit(bitIdx).id());
                    }
                } else {
                    // The variable vector is unsigned, and the value vector is signed. Signed values cannot be
                    // represented in the variable vector, and neither can too large unsigned values. Hence, the sign
                    // bit is an error bit, as are all extra non-sign bits of the value vector.
                    int minBitIdx = Math.min(varVectorLength, valueVectorLength - 1);
                    for (int bitIdx = minBitIdx; bitIdx < valueVectorLength; bitIdx++) {
                        error = error.orWith(valueVector.getBit(bitIdx).id());
                    }
                }
            } else {
                if (valueVectorIsUnsigned) {
                    // The variable vector is signed, and the value vector is unsigned. Too large unsigned values of the
                    // value vector cannot be represented in the variable vector. Hence, all extra unsigned bits of the
                    // value vector, so those of the variable vector's sign bit and beyond, are error bits.
                    for (int bitIdx = varVectorLength - 1; bitIdx < valueVectorLength; bitIdx++) {
                        error = error.orWith(valueVector.getBit(bitIdx).id());
                    }
                } else {
                    // Both vectors are signed. Signed values in two's complement representation can be sign-extended
                    // while preserving the value of the vector. Hence, if the value vector has extra bits compared to
                    // the variable vector, and such an extra bit is different from the value vector's sign bit, the
                    // extra bit is an error bit.
                    BDD signBit = valueVector.getBit(valueVectorLength - 1);
                    for (int bitIdx = varVectorLength - 1; bitIdx < valueVectorLength - 1; bitIdx++) {
                        BDD signBitEquivValueBit = signBit.biimp(valueVector.getBit(bitIdx));
                        BDD signBitUnequivValueBit = signBitEquivValueBit.not();
                        signBitEquivValueBit.free();
                        error = error.orWith(signBitUnequivValueBit);
                    }
                }
            }

            // Cleanup.
            valueVector.free();

            // Return the full relation.
            return pair(relation, error);
        }
    }

    /**
     * Convert a CIF location pointer variable assignment, consisting of a CIF location pointer variable ({@code lp})
     * and its new value expression ({@code v}), to an update predicate ({@code lp+ = v}).
     *
     * @param locPtr The location pointer CIF/BDD variable.
     * @param newValue The expression indicating the variable's new value.
     * @param cifBddSpec The CIF/BDD specification.
     * @return The update relation.
     */
    private BDD convertLocPtrAssignment(CifBddLocPtrVariable locPtr, Expression newValue, CifBddSpec cifBddSpec) {
        if (newValue instanceof IntExpression intNewValue) {
            // Get 0-based location index, which is also the bit index.
            int locIdx = intNewValue.getValue();
            Assert.check(locIdx >= 0);

            // Create and return assignment relation 'lp+ = locIdx'.
            UnsignedBddBitVector varVector = (UnsignedBddBitVector)locPtr.createBitVector(true);
            UnsignedBddBitVector locVector = UnsignedBddBitVector.createFromInt(cifBddSpec.factory, locIdx);
            Assert.check(locVector.length() <= varVector.length());
            locVector.resize(varVector.length());
            BDD relation = varVector.equalTo(locVector);
            varVector.free();
            locVector.free();
            return relation;
        } else if (newValue instanceof DiscVariableExpression discVarNewValue) {
            // Must be a self-assignment (lp := lp).
            DiscVariable discVar = discVarNewValue.getVariable();
            Assert.areEqual(locPtr.var, discVar);

            // Create and return 'lp+ = lp' relation.
            BddBitVector<?, ?> vectorOld = locPtr.createBitVector(false);
            BddBitVector<?, ?> vectorNew = locPtr.createBitVector(true);
            BDD relation = vectorNew.equalToAny(vectorOld);
            vectorOld.free();
            vectorNew.free();
            return relation;
        } else if (newValue instanceof IfExpression ifNewValue) {
            // Convert else.
            BDD result = convertLocPtrAssignment(locPtr, ifNewValue.getElse(), cifBddSpec);

            // Convert elifs/thens.
            for (int i = ifNewValue.getElifs().size() - 1; i >= 0; i--) {
                ElifExpression elifExpr = ifNewValue.getElifs().get(i);
                BDD elifGuards = convertPreds(elifExpr.getGuards(), false, cifBddSpec, null);
                BDD elifThen = convertLocPtrAssignment(locPtr, elifExpr.getThen(), cifBddSpec);
                BDD elifResult = elifGuards.ite(elifThen, result);
                elifGuards.free();
                elifThen.free();
                result.free();
                result = elifResult;
            }

            // Convert if/then.
            BDD ifGuards = convertPreds(ifNewValue.getGuards(), false, cifBddSpec, null);
            BDD ifThen = convertLocPtrAssignment(locPtr, ifNewValue.getThen(), cifBddSpec);
            BDD ifResult = ifGuards.ite(ifThen, result);
            ifGuards.free();
            ifThen.free();
            result.free();
            result = ifResult;

            // Return the result.
            return result;
        } else {
            throw new AssertionError("Unexpected new value for location poitner CIF/BDD variable: " + newValue);
        }
    }

    /**
     * Convert reachability requirement annotation predicates.
     *
     * @param spec The CIF specification.
     * @param cifBddSpec The CIF/BDD specification. Is modified in-place.
     */
    private void convertReachReqPreds(Specification spec, CifBddSpec cifBddSpec) {
        // Get reachability requirement annotation predicates.
        List<Expression> reachReqPreds = CifCollectUtils.getComplexComponentsStream(spec)
                .flatMap(c -> CifReachabilityRequirementAnnotationUtils.getPredicates(c).stream()).toList();

        // Convert the predicates to BDDs and store them in the CIF/BDD specification.
        cifBddSpec.reachReqPreds = listc(reachReqPreds.size());
        for (Expression reachReqPred: reachReqPreds) {
            cifBddSpec.reachReqPreds.add(convertPred(reachReqPred, false, cifBddSpec, null));

            if (cifBddSpec.settings.getTermination().isRequested()) {
                return;
            }
        }
    }

    /**
     * Adds an event and edge for each input variable.
     *
     * @param cifBddSpec The CIF/BDD specification. Is modified in-place.
     */
    private void addInputVariableEdges(CifBddSpec cifBddSpec) {
        // Initialization.
        cifBddSpec.inputVarEvents = set();

        // Add for each input variable.
        for (CifBddVariable var: cifBddSpec.variables) {
            // Handle only input variables.
            if (!(var instanceof CifBddInputVariable)) {
                continue;
            }
            CifBddInputVariable cifBddInputVar = (CifBddInputVariable)var;

            // Create uncontrollable event for the input variable.
            Event event = newEvent();
            event.setControllable(false);
            event.setName(cifBddInputVar.var.getName());

            // Add new event to the alphabet.
            cifBddSpec.alphabet.add(event);

            // Add new event to the original specification, for proper absolute naming. Also store it in the CIF/BDD
            // specification, to allow for later removal of the temporary event, for instance after synthesis.
            ComplexComponent comp = (ComplexComponent)cifBddInputVar.var.eContainer();
            comp.getDeclarations().add(event);
            cifBddSpec.inputVarEvents.add(event);

            // Add edge that allows input variable to change to any other value.
            CifBddEdge edge = new CifBddEdge(cifBddSpec);
            edge.edges = list((Edge)null);
            edge.event = event;
            edge.origGuard = cifBddSpec.factory.one();
            edge.guard = cifBddSpec.factory.one();
            edge.error = cifBddSpec.factory.zero();
            cifBddSpec.edges.add(edge);
            cifBddSpec.eventEdges.put(event, list(edge));

            // Add CIF assignment to edge. Right hand side not filled, as it is not a 'normal' assignment. Also,
            // technically in CIF an input variable can not be assigned.
            InputVariableExpression addr = newInputVariableExpression();
            addr.setVariable(cifBddInputVar.var);
            Assignment asgn = newAssignment();
            asgn.setAddressable(addr);
            edge.updates = list(list(asgn));

            // Add the update relation, which is defined to be the predicate 'input+ != input' to allow the input
            // variable to change to any other value, thereby keeping the new value in the CIF variable domain.
            BddBitVector<?, ?> vectorOld = var.createBitVector(false);
            BddBitVector<?, ?> vectorNew = var.createBitVector(true);
            edge.update = vectorOld.unequalToAny(vectorNew);
            edge.update = edge.update.andWith(BddUtils.getCifVarDomainBdd(var, true, cifBddSpec.factory));
            vectorOld.free();
            vectorNew.free();

            // Indicate that the current input variable is assigned on this edge.
            edge.assignedVariables.add(var);
        }
    }

    /**
     * Merges CIF/BDD edges to the desired granularity.
     *
     * @param cifBddSpec The CIF/BDD specification. Is modified in-place.
     */
    private void mergeEdges(CifBddSpec cifBddSpec) {
        Assert.notNull(cifBddSpec.eventEdges);

        // Sanity check.
        EdgeGranularity granularity = cifBddSpec.settings.getEdgeGranularity();
        boolean adhereToExecScheme = cifBddSpec.settings.getAdhereToExecScheme();
        Assert.implies(adhereToExecScheme, granularity == EdgeGranularity.PER_EDGE);

        // Merge the edges, if needed.
        switch (granularity) {
            case PER_EDGE:
                // Nothing to do, as already at per-edge granularity.
                return;
            case PER_EVENT: {
                // Merge edges per event.
                int eventCount = cifBddSpec.eventEdges.size();
                cifBddSpec.edges = listc(eventCount);
                for (Entry<Event, List<CifBddEdge>> entry: cifBddSpec.eventEdges.entrySet()) {
                    CifBddEdge mergedEdge = entry.getValue().stream().reduce(CifBddEdge::mergeEdges).get();
                    cifBddSpec.edges.add(mergedEdge);
                    entry.setValue(list(mergedEdge));
                }
                return;
            }
        }
        throw new RuntimeException("Unknown granularity: " + granularity);
    }

    /**
     * Orders the CIF/BDD edges, for reachability computations.
     *
     * @param cifBddSpec The CIF/BDD specification. Is modified in-place.
     */
    private void orderEdges(CifBddSpec cifBddSpec) {
        cifBddSpec.orderedEdgesBackward = orderEdgesForDirection(cifBddSpec.edges,
                cifBddSpec.settings.getEdgeOrderBackward(), cifBddSpec.settings.getEdgeOrderAllowDuplicateEvents(),
                false);
        cifBddSpec.orderedEdgesForward = orderEdgesForDirection(cifBddSpec.edges,
                cifBddSpec.settings.getEdgeOrderForward(), cifBddSpec.settings.getEdgeOrderAllowDuplicateEvents(),
                true);
    }

    /**
     * Orders the CIF/BDD edges, for a single direction, i.e., for forward or backward reachability computations.
     *
     * @param edges The edges in linearized model order.
     * @param orderTxt The order as textual value from the settings, for the given direction.
     * @param edgeOrderAllowDuplicateEvents Whether duplicate events are allowed for custom edge orders.
     * @param forForwardReachability Order for forward reachability ({@code true}) or backward reachability
     *     ({@code false}).
     * @return The ordered edges.
     */
    private static List<CifBddEdge> orderEdgesForDirection(List<CifBddEdge> edges, String orderTxt,
            EdgeOrderDuplicateEventAllowance edgeOrderAllowDuplicateEvents, boolean forForwardReachability)
    {
        if (orderTxt.toLowerCase(Locale.US).equals("model")) {
            // No reordering. Keep linearized model order.
            return edges;
        } else if (orderTxt.toLowerCase(Locale.US).equals("reverse-model")) {
            // Reorder to the reverse of the linearized model order.
            return reverse(edges);
        } else if (orderTxt.toLowerCase(Locale.US).equals("sorted")) {
            // Sort based on absolute event names. This edge order is, for forward application of edges, the only order
            // that adheres to the execution scheme prescribed by the controller properties checker.
            return edges.stream()
                    .sorted((v, w) -> Strings.SORTER.compare(getAbsName(v.event, false), getAbsName(w.event, false)))
                    .toList();
        } else if (orderTxt.toLowerCase(Locale.US).equals("reverse-sorted")) {
            // Sort based on absolute event names, and reverse.
            return reverse(edges.stream()
                    .sorted((v, w) -> Strings.SORTER.compare(getAbsName(v.event, false), getAbsName(w.event, false)))
                    .toList());
        } else if (orderTxt.toLowerCase(Locale.US).equals("random")
                || orderTxt.toLowerCase(Locale.US).startsWith("random:"))
        {
            // Get seed, if specified.
            Long seed = null;
            if (orderTxt.contains(":")) {
                int idx = orderTxt.indexOf(":");
                String seedTxt = orderTxt.substring(idx + 1);
                try {
                    seed = Long.parseUnsignedLong(seedTxt);
                } catch (NumberFormatException ex) {
                    String msg = fmt("Invalid random %s edge order seed number: \"%s\".",
                            forForwardReachability ? "forward" : "backward", orderTxt);
                    throw new InvalidOptionException(msg, ex);
                }
            }

            // Shuffle to random order.
            List<CifBddEdge> orderedEdges = copy(edges);
            if (seed == null) {
                Collections.shuffle(orderedEdges);
            } else {
                Collections.shuffle(orderedEdges, new Random(seed));
            }
            return orderedEdges;
        } else {
            // Sort based on supplied custom event order.
            List<CifBddEdge> orderedEdges = listc(edges.size());
            Set<CifBddEdge> processedEdges = set();

            // Process elements.
            for (String elemTxt: StringUtils.split(orderTxt, ",")) {
                // Skip empty.
                elemTxt = elemTxt.trim();
                if (elemTxt.isEmpty()) {
                    continue;
                }

                // Create regular expression for matching.
                Pattern pattern;
                try {
                    pattern = CifTextUtils.getRegExForCifNamePattern(elemTxt);
                } catch (IllegalArgumentException ex) {
                    throw new InvalidOptionException(
                            fmt("Invalid custom %s edge order: invalid name pattern \"%s\".",
                                    forForwardReachability ? "forward" : "backward", elemTxt),
                            new InvalidOptionException(ex.getMessage()));
                }

                // Found actual element. Look up matching CIF/BDD edges.
                List<CifBddEdge> matches = list();
                for (CifBddEdge edge: edges) {
                    String name = getAbsName(edge.event, false);
                    if (pattern.matcher(name).matches()) {
                        matches.add(edge);
                    }
                }

                // Need a least one match.
                if (matches.isEmpty()) {
                    String msg = fmt(
                            "Invalid custom %s edge order: can't find a match for \"%s\". There is no supported event "
                                    + "or input variable in the specification that matches the given name pattern.",
                            forForwardReachability ? "forward" : "backward", elemTxt);
                    throw new InvalidOptionException(msg);
                }

                // Sort matches.
                Collections.sort(matches,
                        (v, w) -> Strings.SORTER.compare(getAbsName(v.event, false), getAbsName(w.event, false)));

                // Check for duplicate events, if duplicates are disallowed.
                if (edgeOrderAllowDuplicateEvents == EdgeOrderDuplicateEventAllowance.DISALLOWED) {
                    for (CifBddEdge edge: matches) {
                        if (processedEdges.contains(edge)) {
                            String msg = fmt(
                                    "Invalid custom %s edge order: event \"%s\" is included more than once. "
                                            + "If the duplicate event is intentional, enable allowing duplicate events "
                                            + "in the custom event order.",
                                    forForwardReachability ? "forward" : "backward", getAbsName(edge.event, false));
                            throw new InvalidOptionException(msg);
                        }
                    }
                }

                // Update for matched edges.
                processedEdges.addAll(matches);
                orderedEdges.addAll(matches);
            }

            // Check completeness.
            Set<CifBddEdge> missingEdges = difference(edges, processedEdges);
            if (!missingEdges.isEmpty()) {
                Set<String> names = set();
                for (CifBddEdge edge: missingEdges) {
                    names.add("\"" + getAbsName(edge.event, false) + "\"");
                }
                List<String> sortedNames = Sets.sortedgeneric(names, Strings.SORTER);
                String msg = fmt(
                        "Invalid custom %s edge order: "
                                + "the following event(s) are missing from the specified order: %s.",
                        forForwardReachability ? "forward" : "backward", String.join(", ", sortedNames));
                throw new InvalidOptionException(msg);
            }

            // Return the custom edge order.
            return orderedEdges;
        }
    }

    /**
     * Check edge workset algorithm settings.
     *
     * @param settings The settings.
     */
    private void checkEdgeWorksetAlgorithmSettings(CifBddSettings settings) {
        // Skip if workset algorithm is disabled.
        if (settings.getExplorationStrategy() != ExplorationStrategy.CHAINING_WORKSET) {
            return;
        }

        // Edge workset algorithm requires per-event edge granularity, and no duplicate edges in the edge order.
        if (settings.getEdgeGranularity() != EdgeGranularity.PER_EVENT) {
            throw new InvalidOptionException(
                    "The edge workset algorithm can only be used with per-event edge granularity. "
                            + "Either disable the edge workset algorithm, or configure per-event edge granularity.");
        }
        if (settings.getEdgeOrderAllowDuplicateEvents() == EdgeOrderDuplicateEventAllowance.ALLOWED) {
            throw new InvalidOptionException(
                    "The edge workset algorithm can not be used with duplicate events in the edge order. "
                            + "Either disable the edge workset algorithm, or disable duplicates for custom edge orders.");
        }
    }

    /**
     * Check saturation settings.
     *
     * @param settings The settings.
     */
    private void checkSaturationSettings(CifBddSettings settings) {
        // Skip if saturation is disabled.
        if (settings.getExplorationStrategy() != ExplorationStrategy.SATURATION) {
            return;
        }

        // Saturation requires no duplicate edges in the edge order.
        if (settings.getEdgeOrderAllowDuplicateEvents() == EdgeOrderDuplicateEventAllowance.ALLOWED) {
            throw new InvalidOptionException(
                    "Saturation can not be used with duplicate events in the edge order. "
                            + "Either disable saturation, or disable duplicates for custom edge orders.");
        }
    }

    /**
     * Converts CIF predicates to a BDD predicate, assuming conjunction between the CIF predicates.
     *
     * @param preds The CIF predicates.
     * @param initial Whether the predicates apply only to the initial state ({@code true}) or any state ({@code false},
     *     includes the initial state).
     * @param cifBddSpec The CIF/BDD specification.
     * @param funcValuation The valuation of function parameters and local variables to their values (each as a
     *     {@link BDD} or {@link BddBitVector}). Must be {@code null} if the predicate is not part of a function body.
     * @return The BDD predicate.
     */
    public BDD convertPreds(List<Expression> preds, boolean initial, CifBddSpec cifBddSpec,
            FunctionValuation funcValuation)
    {
        BDD rslt = cifBddSpec.factory.one();
        for (Expression pred: preds) {
            rslt = rslt.andWith(convertPred(pred, initial, cifBddSpec, funcValuation));
        }
        return rslt;
    }

    /**
     * Converts a CIF predicate to a BDD predicate.
     *
     * @param pred The CIF predicate.
     * @param initial Whether the predicate applies only to the initial state ({@code true}) or any state
     *     ({@code false}, includes the initial state).
     * @param cifBddSpec The CIF/BDD specification.
     * @param funcValuation The valuation of function parameters and local variables to their values (each as a
     *     {@link BDD} or {@link BddBitVector}). Must be {@code null} if the predicate is not part of a function body.
     * @return The BDD predicate.
     */
    public BDD convertPred(Expression pred, boolean initial, CifBddSpec cifBddSpec, FunctionValuation funcValuation) {
        if (pred instanceof BoolExpression) {
            // Boolean literal.
            boolean value = ((BoolExpression)pred).isValue();
            return value ? cifBddSpec.factory.one() : cifBddSpec.factory.zero();
        } else if (pred instanceof DiscVariableExpression) {
            // Check function valuation, if present.
            DiscVariable cifVar = ((DiscVariableExpression)pred).getVariable();
            if (funcValuation != null) {
                Object result = funcValuation.get(cifVar);
                if (result != null) {
                    Assert.check(result instanceof BDD);
                    return ((BDD)result).id(); // Use a copy to keep the valuation intact.
                }
            }

            // Boolean variable reference.
            Assert.check(normalizeType(cifVar.getType()) instanceof BoolType);
            int varIdx = getDiscVarIdx(cifBddSpec.variables, cifVar);
            Assert.check(varIdx >= 0);

            // Create BDD predicate for 'x' or 'x = true'.
            CifBddVariable var = cifBddSpec.variables[varIdx];
            int[] bddVarIndices = var.domain.getVarIndices();
            Assert.areEqual(1, bddVarIndices.length);
            return cifBddSpec.factory.ithVar(bddVarIndices[0]);
        } else if (pred instanceof InputVariableExpression) {
            // Boolean variable reference.
            InputVariable cifVar = ((InputVariableExpression)pred).getVariable();
            Assert.check(normalizeType(cifVar.getType()) instanceof BoolType);
            int varIdx = getInputVarIdx(cifBddSpec.variables, cifVar);
            Assert.check(varIdx >= 0);

            // Create BDD predicate for 'x' or 'x = true'.
            CifBddVariable var = cifBddSpec.variables[varIdx];
            int[] bddVarIndices = var.domain.getVarIndices();
            Assert.areEqual(1, bddVarIndices.length);
            return cifBddSpec.factory.ithVar(bddVarIndices[0]);
        } else if (pred instanceof AlgVariableExpression algExpr) {
            // Algebraic variable reference. If there is no cached result, convert the algebraic variable to a BDD and
            // store it in the cache.
            AlgVariable var = algExpr.getVariable();
            BDD result = conversionBddCache.get(var);
            if (result == null) {
                // Get the single defining value expression, representing the value of the variable. It is in an 'if'
                // expression if an equation is provided per location of an automaton with more than one location.
                Assert.check(normalizeType(var.getType()) instanceof BoolType);
                Expression value = CifEquationUtils.getSingleValueForAlgVar(var);

                // Convert the defining value expression and store the result in the cache.
                result = convertPred(value, initial, cifBddSpec, funcValuation);
                conversionBddCache.put(algExpr.getVariable(), result);
            }

            // Return a copy of the cached BDD, to ensure the cache remains valid.
            return result.id();
        } else if (pred instanceof LocationExpression) {
            // Location reference.
            Location loc = ((LocationExpression)pred).getLocation();
            Automaton aut = CifLocationUtils.getAutomaton(loc);
            int varIdx = getLpVarIdx(cifBddSpec.variables, aut);
            if (varIdx == -1) {
                // Automata with only one location have no location pointer, but are always the active location. So,
                // referring to them is as using a 'true' predicate.
                Assert.areEqual(aut.getLocations().size(), 1);
                return cifBddSpec.factory.one();
            }
            Assert.check(varIdx >= 0);
            CifBddVariable var = cifBddSpec.variables[varIdx];

            // Create BDD predicate for location pointer being equal to value that represents the location.
            int locIdx = aut.getLocations().indexOf(loc);
            Assert.check(locIdx >= 0);
            return BddUtils.getVarEqualToValueBdd(var, locIdx, false, cifBddSpec.factory);
        } else if (pred instanceof ConstantExpression) {
            // Boolean constant reference.
            Constant constant = ((ConstantExpression)pred).getConstant();
            Assert.check(normalizeType(constant.getType()) instanceof BoolType);

            // Evaluate the constant's value.
            Object valueObj;
            try {
                valueObj = CifEvalUtils.eval(constant.getValue(), initial);
            } catch (CifEvalException ex) {
                throw new AssertionError("Precondition violation.", ex);
            }

            return (boolean)valueObj ? cifBddSpec.factory.one() : cifBddSpec.factory.zero();
        } else if (pred instanceof UnaryExpression) {
            // Unary expression.
            UnaryExpression upred = (UnaryExpression)pred;
            UnaryOperator op = upred.getOperator();

            switch (op) {
                case INVERSE: {
                    BDD child = convertPred(upred.getChild(), initial, cifBddSpec, funcValuation);
                    BDD rslt = child.not();
                    child.free();
                    return rslt;
                }

                default:
                    break; // Try static evaluation.
            }
        } else if (pred instanceof BinaryExpression) {
            // Binary expression.
            BinaryExpression bpred = (BinaryExpression)pred;
            BinaryOperator op = (((BinaryExpression)pred).getOperator());
            Expression lhs = bpred.getLeft();
            Expression rhs = bpred.getRight();

            switch (op) {
                case CONJUNCTION: {
                    CifType ltype = normalizeType(lhs.getType());
                    CifType rtype = normalizeType(rhs.getType());
                    Assert.check(ltype instanceof BoolType);
                    Assert.check(rtype instanceof BoolType);

                    BDD left = convertPred(lhs, initial, cifBddSpec, funcValuation);
                    BDD right = convertPred(rhs, initial, cifBddSpec, funcValuation);
                    return left.andWith(right);
                }

                case DISJUNCTION: {
                    CifType ltype = normalizeType(lhs.getType());
                    CifType rtype = normalizeType(rhs.getType());
                    Assert.check(ltype instanceof BoolType);
                    Assert.check(rtype instanceof BoolType);

                    BDD left = convertPred(lhs, initial, cifBddSpec, funcValuation);
                    BDD right = convertPred(rhs, initial, cifBddSpec, funcValuation);
                    return left.orWith(right);
                }

                case IMPLICATION: {
                    BDD left = convertPred(lhs, initial, cifBddSpec, funcValuation);
                    BDD right = convertPred(rhs, initial, cifBddSpec, funcValuation);
                    return left.impWith(right);
                }

                case BI_CONDITIONAL: {
                    BDD left = convertPred(lhs, initial, cifBddSpec, funcValuation);
                    BDD right = convertPred(rhs, initial, cifBddSpec, funcValuation);
                    return left.biimpWith(right);
                }

                case EQUAL:
                case GREATER_EQUAL:
                case GREATER_THAN:
                case LESS_EQUAL:
                case LESS_THAN:
                case UNEQUAL:
                    return convertCmpPred(lhs, rhs, op, initial, cifBddSpec, funcValuation);

                default:
                    break; // Try static evaluation.
            }
        } else if (pred instanceof FunctionCallExpression fcExpr
                && fcExpr.getFunction() instanceof FunctionExpression funcRef
                && funcRef.getFunction() instanceof InternalFunction internalFunc)
        {
            // Direct calls to internal user-defined functions, so not on function-typed variables, functions returned
            // by calls to other functions, etc.
            Object returnValue = convertFuncCall(internalFunc, fcExpr.getArguments(), initial, cifBddSpec,
                    funcValuation);
            Assert.check(returnValue instanceof BDD);
            return (BDD)returnValue;
        } else if (pred instanceof IfExpression) {
            // Condition expression with boolean result values.
            IfExpression ifPred = (IfExpression)pred;

            // Convert else.
            BDD rslt = convertPred(ifPred.getElse(), initial, cifBddSpec, funcValuation);

            // Convert elifs/thens.
            for (int i = ifPred.getElifs().size() - 1; i >= 0; i--) {
                ElifExpression elifPred = ifPred.getElifs().get(i);
                BDD elifGuards = convertPreds(elifPred.getGuards(), initial, cifBddSpec, funcValuation);
                BDD elifThen = convertPred(elifPred.getThen(), initial, cifBddSpec, funcValuation);
                BDD elifRslt = elifGuards.ite(elifThen, rslt);
                elifGuards.free();
                elifThen.free();
                rslt.free();
                rslt = elifRslt;
            }

            // Convert if/then.
            BDD ifGuards = convertPreds(ifPred.getGuards(), initial, cifBddSpec, funcValuation);
            BDD ifThen = convertPred(ifPred.getThen(), initial, cifBddSpec, funcValuation);
            BDD ifRslt = ifGuards.ite(ifThen, rslt);
            ifGuards.free();
            ifThen.free();
            rslt.free();
            rslt = ifRslt;

            // Return converted conditional expression.
            return rslt;
        } else if (pred instanceof SwitchExpression) {
            // Switch expression with boolean result values.
            SwitchExpression switchPred = (SwitchExpression)pred;
            Expression value = switchPred.getValue();
            List<SwitchCase> cases = switchPred.getCases();

            // Convert else.
            BDD rslt = convertPred(last(cases).getValue(), initial, cifBddSpec, funcValuation);

            // Convert cases.
            for (int i = cases.size() - 2; i >= 0; i--) {
                SwitchCase cse = cases.get(i);
                Expression caseGuardExpr = CifTypeUtils.isAutRefExpr(value) ? cse.getKey() : newBinaryExpression(
                        deepclone(value), BinaryOperator.EQUAL, null, deepclone(cse.getKey()), newBoolType());
                BDD caseGuard = convertPred(caseGuardExpr, initial, cifBddSpec, funcValuation);
                BDD caseThen = convertPred(cse.getValue(), initial, cifBddSpec, funcValuation);
                BDD caseRslt = caseGuard.ite(caseThen, rslt);
                caseGuard.free();
                caseThen.free();
                rslt.free();
                rslt = caseRslt;
            }

            return rslt;
        }

        // Evaluate statically-evaluable predicate.
        Object valueObj;
        try {
            valueObj = CifEvalUtils.eval(pred, initial);
        } catch (CifEvalException ex) {
            throw new AssertionError("Precondition violation.", ex);
        }

        // Create BDD for 'true' or 'false'.
        Assert.check(valueObj instanceof Boolean);
        boolean value = (boolean)valueObj;
        return value ? cifBddSpec.factory.one() : cifBddSpec.factory.zero();
    }

    /**
     * Converts a CIF comparison predicate to a BDD predicate.
     *
     * @param lhs The left hand side of the comparison predicate.
     * @param rhs The right hand side of the comparison predicate.
     * @param op The comparison operator ({@code =}, {@code !=}, {@code <}, {@code <=}, {@code >}, or {@code >=}).
     * @param initial Whether the predicate applies only to the initial state ({@code true}) or any state
     *     ({@code false}, includes the initial state).
     * @param cifBddSpec The CIF/BDD specification.
     * @param funcValuation The valuation of function parameters and local variables to their values (each as a
     *     {@link BDD} or {@link BddBitVector}). Must be {@code null} if the predicate is not part of a function body.
     * @return The BDD predicate.
     */
    public BDD convertCmpPred(Expression lhs, Expression rhs, BinaryOperator op, boolean initial,
            CifBddSpec cifBddSpec, FunctionValuation funcValuation)
    {
        // Check lhs and rhs types.
        CifType ltype = normalizeType(lhs.getType());
        CifType rtype = normalizeType(rhs.getType());
        Assert.check((ltype instanceof BoolType && rtype instanceof BoolType)
                || (ltype instanceof EnumType && rtype instanceof EnumType)
                || (ltype instanceof IntType && rtype instanceof IntType));

        // Special handling of boolean values.
        if (ltype instanceof BoolType && rtype instanceof BoolType) {
            BDD lbdd = convertPred(lhs, initial, cifBddSpec, funcValuation);
            BDD rbdd = convertPred(rhs, initial, cifBddSpec, funcValuation);
            switch (op) {
                case EQUAL:
                    return lbdd.biimpWith(rbdd);

                case UNEQUAL: {
                    BDD eq = lbdd.biimpWith(rbdd);
                    BDD rslt = eq.not();
                    eq.free();
                    return rslt;
                }

                default:
                    throw new RuntimeException("Unexpected op: " + op);
            }
        }

        // Get lhs and rhs vectors, in compatible representations.
        BddBitVector<?, ?> leftVector = convertExpr(lhs, initial, cifBddSpec, funcValuation);
        BddBitVector<?, ?> rightVector = convertExpr(rhs, initial, cifBddSpec, funcValuation);
        Pair<BddBitVector<?, ?>, BddBitVector<?, ?>> vectors = BddBitVector.ensureCompatible(leftVector, rightVector);
        leftVector = vectors.left;
        rightVector = vectors.right;

        // Resize the lhs and rhs vectors to have the same length, such that the comparison can be performed.
        BddBitVector.ensureSameLength(leftVector, rightVector);

        // Apply comparison.
        BDD result = switch (op) {
            case LESS_THAN -> leftVector.lessThanAny(rightVector);
            case LESS_EQUAL -> leftVector.lessOrEqualAny(rightVector);
            case GREATER_THAN -> leftVector.greaterThanAny(rightVector);
            case GREATER_EQUAL -> leftVector.greaterOrEqualAny(rightVector);
            case EQUAL -> leftVector.equalToAny(rightVector);
            case UNEQUAL -> leftVector.unequalToAny(rightVector);
            default -> throw new RuntimeException("Unexpected operator: " + op);
        };

        // Cleanup.
        leftVector.free();
        rightVector.free();

        // Return the result.
        return result;
    }

    /**
     * Converts a CIF expression to a BDD bit vector.
     *
     * @param expr The CIF expression. Has an integer or enumeration type.
     * @param initial Whether the expression applies only to the initial state ({@code true}) or any state
     *     ({@code false}, includes the initial state).
     * @param cifBddSpec The CIF/BDD specification.
     * @param funcValuation The valuation of function parameters and local variables to their values (each as a
     *     {@link BDD} or {@link BddBitVector}). Must be {@code null} if the predicate is not part of a function body.
     * @return The BDD bit vector resulting from the conversion.
     */
    public BddBitVector<?, ?> convertExpr(Expression expr, boolean initial, CifBddSpec cifBddSpec,
            FunctionValuation funcValuation)
    {
        // Check expression type.
        CifType type = normalizeType(expr.getType());
        Assert.check(type instanceof IntType || type instanceof EnumType);

        // Variable references.
        if (expr instanceof DiscVariableExpression discExpr) {
            // Check function valuation, if present.
            DiscVariable cifVar = discExpr.getVariable();
            if (funcValuation != null) {
                Object result = funcValuation.get(cifVar);
                if (result != null) {
                    Assert.check(result instanceof BddBitVector);
                    return ((BddBitVector<?, ?>)result).copy(); // Use a copy to keep the valuation intact.
                }
            }

            // Get variable.
            int varIdx = getDiscVarIdx(cifBddSpec.variables, cifVar);
            Assert.check(varIdx >= 0);
            CifBddVariable var = cifBddSpec.variables[varIdx];

            // Create bit vector for the domain of the variable.
            return var.createBitVector(false);
        } else if (expr instanceof InputVariableExpression inputExpr) {
            // Get variable.
            InputVariable cifVar = inputExpr.getVariable();
            int varIdx = getInputVarIdx(cifBddSpec.variables, cifVar);
            Assert.check(varIdx >= 0);
            CifBddVariable var = cifBddSpec.variables[varIdx];

            // Create bit vector for the domain of the variable.
            return var.createBitVector(false);
        } else if (expr instanceof AlgVariableExpression algExpr) {
            // Algebraic variable reference. If there is no cached result, convert the algebraic variable to a bit
            // vector and store it in the cache.
            AlgVariable var = algExpr.getVariable();
            BddBitVector<?, ?> result = conversionBitVectorCache.get(var);
            if (result == null) {
                // Get the single defining value expression, representing the value of the variable. It is in an 'if'
                // expression if an equation is provided per location of an automaton with more than one location.
                Expression value = CifEquationUtils.getSingleValueForAlgVar(var);

                // Convert the defining value expression and store the result in the cache.
                result = convertExpr(value, initial, cifBddSpec, funcValuation);
                conversionBitVectorCache.put(algExpr.getVariable(), result);
            }

            // Return a copy of the cached bit vector, to ensure the cache remains valid.
            return result.copy();
        }

        // Unary operators.
        if (expr instanceof UnaryExpression unaryExpr) {
            switch (unaryExpr.getOperator()) {
                case PLUS:
                    return convertExpr(unaryExpr.getChild(), initial, cifBddSpec, funcValuation);

                case NEGATE: {
                    // Convert child expression to a bit vector.
                    BddBitVector<?, ?> childVector = convertExpr(unaryExpr.getChild(), initial, cifBddSpec,
                            funcValuation);

                    // Ensure the bit vector is in signed representation.
                    SignedBddBitVector signedChildVector;
                    if (childVector instanceof UnsignedBddBitVector uVector) {
                        signedChildVector = SignedBddBitVector.createFromUnsignedBitVector(uVector);
                        uVector.free();
                    } else if (childVector instanceof SignedBddBitVector sVector) {
                        signedChildVector = sVector;
                    } else {
                        throw new AssertionError("Unknown bit vector representation: " + childVector.getClass());
                    }

                    // Resize bit vector to prevent overflow for most negative value of the representation.
                    signedChildVector.resize(signedChildVector.length() + 1);

                    // Compute negation.
                    SignedBddBitVectorAndCarry negatedVectorAndCarry = signedChildVector.negate();
                    Assert.check(negatedVectorAndCarry.carry.isZero());

                    // Cleanup.
                    signedChildVector.free();

                    // Shrink the result, preventing a cascade of length increases for repeated operations that increase
                    // the length, as well as in general preventing unnecessary computations later on.
                    negatedVectorAndCarry.vector.shrink();

                    // Return the result.
                    return negatedVectorAndCarry.vector;
                }

                default:
                    break; // Try static evaluation.
            }
        }

        // Binary operators.
        if (expr instanceof BinaryExpression binExpr) {
            Expression lhs = binExpr.getLeft();
            Expression rhs = binExpr.getRight();

            switch (binExpr.getOperator()) {
                case ADDITION: {
                    // Get lhs and rhs vectors, in compatible representations.
                    BddBitVector<?, ?> leftVector = convertExpr(lhs, initial, cifBddSpec, funcValuation);
                    BddBitVector<?, ?> rightVector = convertExpr(rhs, initial, cifBddSpec, funcValuation);
                    Pair<BddBitVector<?, ?>, BddBitVector<?, ?>> vectors = BddBitVector.ensureCompatible(
                            leftVector, rightVector);
                    leftVector = vectors.left;
                    rightVector = vectors.right;

                    // Resize the lhs and rhs vectors to have the same length, such that the addition can be performed.
                    // Also ensure that the addition result doesn't overflow.
                    BddBitVector.ensureSameLength(leftVector, rightVector, 1);

                    // Apply addition.
                    BddBitVectorAndCarry<?, ?> result = leftVector.addAny(rightVector);
                    Assert.check(result.carry.isZero());

                    // Cleanup.
                    leftVector.free();
                    rightVector.free();

                    // Shrink the result, preventing a cascade of length increases for repeated operations that increase
                    // the length, as well as in general preventing unnecessary computations later on.
                    result.vector.shrink();

                    // Return the result.
                    return result.vector;
                }

                case SUBTRACTION: {
                    // Get lhs and rhs vectors.
                    BddBitVector<?, ?> leftVector = convertExpr(lhs, initial, cifBddSpec, funcValuation);
                    BddBitVector<?, ?> rightVector = convertExpr(rhs, initial, cifBddSpec, funcValuation);

                    // Make sure the lhs and rhs bit vectors are in signed representation. Even if the CIF type
                    // indicates that the result is non-negative, the bit vectors may represent larger ranges than the
                    // CIF ranges, and the subtraction may then result in negative values. For instance, for 'x - 1',
                    // with 'x' of type 'int[1..3]', the CIF result is always non-negative. But, for 'x', a bit vector
                    // is created that can represent 'int[0..3]', and thus 'x - 1' may still result in negative values.
                    // In fact, all bit vectors can always represent value zero, and the subtraction of two bit vectors
                    // not resulting in any potential negative values is exceedingly rare.
                    if (leftVector instanceof UnsignedBddBitVector uLeftVector) {
                        leftVector = SignedBddBitVector.createFromUnsignedBitVector(uLeftVector);
                        uLeftVector.free();
                    }
                    if (rightVector instanceof UnsignedBddBitVector uRightVector) {
                        rightVector = SignedBddBitVector.createFromUnsignedBitVector(uRightVector);
                        uRightVector.free();
                    }

                    // Resize the lhs and rhs vectors to have the same length, such that the subtraction can be
                    // performed. Also ensure that the subtraction result doesn't overflow.
                    BddBitVector.ensureSameLength(leftVector, rightVector, 1);

                    // Apply subtraction.
                    BddBitVectorAndCarry<?, ?> result = leftVector.subtractAny(rightVector);
                    Assert.check(result.carry.isZero());

                    // Cleanup.
                    leftVector.free();
                    rightVector.free();

                    // Shrink the result, preventing a cascade of length increases for repeated operations that increase
                    // the length, as well as in general preventing unnecessary computations later on.
                    result.vector.shrink();

                    // Return the result.
                    return result.vector;
                }

                case MULTIPLICATION: {
                    // Get lhs and rhs vectors, in compatible representations.
                    BddBitVector<?, ?> leftVector = convertExpr(lhs, initial, cifBddSpec, funcValuation);
                    BddBitVector<?, ?> rightVector = convertExpr(rhs, initial, cifBddSpec, funcValuation);
                    Pair<BddBitVector<?, ?>, BddBitVector<?, ?>> vectors = BddBitVector.ensureCompatible(
                            leftVector, rightVector);
                    leftVector = vectors.left;
                    rightVector = vectors.right;

                    // Resize the lhs and rhs vectors to have the same length, such that the multiplication can be
                    // performed. Also ensure that the multiplication result doesn't overflow.
                    int length = Math.max(leftVector.length(), rightVector.length()) * 2;
                    leftVector.resize(length);
                    rightVector.resize(length);

                    // Apply multiplication.
                    BddBitVectorAndCarry<?, ?> result = leftVector.multiplyAny(rightVector);
                    Assert.check(result.carry.isZero());

                    // Cleanup.
                    leftVector.free();
                    rightVector.free();

                    // Shrink the result, preventing a cascade of length increases for repeated operations that increase
                    // the length, as well as in general preventing unnecessary computations later on.
                    result.vector.shrink();

                    // Return the result.
                    return result.vector;
                }

                case INTEGER_DIVISION:
                case MODULUS: {
                    // Get lhs vector.
                    BddBitVector<?, ?> leftVector = convertExpr(lhs, initial, cifBddSpec, funcValuation);

                    // Evaluate rhs.
                    Assert.check(CifValueUtils.hasSingleValue(rhs, initial, true));
                    Object rhsValueObj;
                    try {
                        rhsValueObj = CifEvalUtils.eval(rhs, initial);
                    } catch (CifEvalException ex) {
                        throw new AssertionError("Precondition violation.", ex);
                    }

                    // Get divisor (rhs value).
                    int divisor = (int)rhsValueObj;
                    Assert.check(divisor > 0); // No division by zero, no negative divisor.

                    // Resize lhs vector if needed. The divisor needs to fit. Depending on the representation and the
                    // operation that is performed, some highest bits may need to be 'false' as well.
                    int lhsLen = leftVector.length();
                    int rhsLen = switch (leftVector) {
                        case UnsignedBddBitVector u -> UnsignedBddBitVector.getMinimumLength(divisor);
                        case SignedBddBitVector s -> SignedBddBitVector.getMinimumLength(divisor);
                        default -> throw new AssertionError(
                                "Unknown bit vector representation: " + leftVector.getClass());
                    };
                    int length = Math.max(lhsLen, rhsLen);

                    boolean isDiv = binExpr.getOperator() == INTEGER_DIVISION;
                    if (leftVector instanceof SignedBddBitVector || !isDiv) {
                        length++;
                    }

                    leftVector.resize(length);

                    // Apply div/mod.
                    BddBitVector<?, ?> result = isDiv ? leftVector.div(divisor) : leftVector.mod(divisor);

                    // Cleanup.
                    leftVector.free();

                    // Shrink the result, preventing a cascade of length increases for repeated operations that increase
                    // the length, as well as in general preventing unnecessary computations later on.
                    result.shrink();

                    // Return the result.
                    return result;
                }

                default:
                    break; // Try static evaluation.
            }
        }

        // Standard library function calls.
        if (expr instanceof FunctionCallExpression fcExpr
                && fcExpr.getFunction() instanceof StdLibFunctionExpression stdlibRef)
        {
            switch (stdlibRef.getFunction()) {
                case ABS: {
                    // Convert argument to a bit vector.
                    Expression arg = Lists.single(fcExpr.getArguments());
                    BddBitVector<?, ?> argVector = convertExpr(arg, initial, cifBddSpec, funcValuation);

                    // Resize the argument bit vector to ensure computing the absolute value doesn't overflow.
                    argVector.resize(argVector.length() + 1);

                    // Compute the absolute value.
                    BddBitVectorAndCarry<?, ?> result = argVector.abs();
                    Assert.check(result.carry.isZero());

                    // Cleanup.
                    argVector.free();

                    // Shrink the result, preventing a cascade of length increases for repeated operations that increase
                    // the length, as well as in general preventing unnecessary computations later on.
                    result.vector.shrink();

                    // Return the result.
                    return result.vector;
                }

                case SIGN: {
                    // Convert argument to a bit vector.
                    Expression arg = Lists.single(fcExpr.getArguments());
                    BddBitVector<?, ?> argVector = convertExpr(arg, initial, cifBddSpec, funcValuation);

                    // Compute the sign.
                    BddBitVector<?, ?> result = argVector.sign();

                    // Cleanup.
                    argVector.free();

                    // Return the result.
                    return result;
                }

                case MAXIMUM: {
                    // Get argument vectors, in compatible representations.
                    BddBitVector<?, ?> arg0Vector = convertExpr(fcExpr.getArguments().get(0), initial, cifBddSpec,
                            funcValuation);
                    BddBitVector<?, ?> arg1Vector = convertExpr(fcExpr.getArguments().get(1), initial, cifBddSpec,
                            funcValuation);
                    Pair<BddBitVector<?, ?>, BddBitVector<?, ?>> vectors = BddBitVector.ensureCompatible(
                            arg0Vector, arg1Vector);
                    arg0Vector = vectors.left;
                    arg1Vector = vectors.right;

                    // Resize the argument vectors to have the same length, such that the operation can be performed.
                    BddBitVector.ensureSameLength(arg0Vector, arg1Vector);

                    // Apply operation.
                    BddBitVector<?, ?> result = arg0Vector.maxAny(arg1Vector);

                    // Cleanup.
                    arg0Vector.free();
                    arg1Vector.free();

                    // Return the result.
                    return result;
                }

                case MINIMUM: {
                    // Get argument vectors, in compatible representations.
                    BddBitVector<?, ?> arg0Vector = convertExpr(fcExpr.getArguments().get(0), initial, cifBddSpec,
                            funcValuation);
                    BddBitVector<?, ?> arg1Vector = convertExpr(fcExpr.getArguments().get(1), initial, cifBddSpec,
                            funcValuation);
                    Pair<BddBitVector<?, ?>, BddBitVector<?, ?>> vectors = BddBitVector.ensureCompatible(
                            arg0Vector, arg1Vector);
                    arg0Vector = vectors.left;
                    arg1Vector = vectors.right;

                    // Resize the argument vectors to have the same length, such that the operation can be performed.
                    BddBitVector.ensureSameLength(arg0Vector, arg1Vector);

                    // Apply operation.
                    BddBitVector<?, ?> result = arg0Vector.minAny(arg1Vector);

                    // Cleanup.
                    arg0Vector.free();
                    arg1Vector.free();

                    // Return the result.
                    return result;
                }

                default:
                    break; // Try static evaluation.
            }
        }

        // Direct calls to internal user-defined functions, so not on function-typed variables, functions returned by
        // calls to other functions, etc.
        if (expr instanceof FunctionCallExpression fcExpr
                && fcExpr.getFunction() instanceof FunctionExpression funcRef
                && funcRef.getFunction() instanceof InternalFunction internalFunc)
        {
            Object returnValue = convertFuncCall(internalFunc, fcExpr.getArguments(), initial, cifBddSpec,
                    funcValuation);
            Assert.check(returnValue instanceof BddBitVector);
            return (BddBitVector<?, ?>)returnValue;
        }

        // Conditional expression.
        if (expr instanceof IfExpression ifExpr) {
            // Convert else.
            BddBitVector<?, ?> result = convertExpr(ifExpr.getElse(), initial, cifBddSpec, funcValuation);

            // Convert elifs/thens.
            for (int i = ifExpr.getElifs().size() - 1; i >= 0; i--) {
                ElifExpression elifExpr = ifExpr.getElifs().get(i);

                // Convert the elif's guards and 'then'.
                BDD elifGuards = convertPreds(elifExpr.getGuards(), initial, cifBddSpec, funcValuation);
                BddBitVector<?, ?> elifThenVector = convertExpr(elifExpr.getThen(), initial, cifBddSpec, funcValuation);

                // Get 'result' and elif's 'then' in compatible representations.
                Pair<BddBitVector<?, ?>, BddBitVector<?, ?>> vectors = BddBitVector.ensureCompatible(
                        result, elifThenVector);
                result = vectors.left;
                elifThenVector = vectors.right;

                // Resize the 'result' and the elif's 'then' to have the same length, such that the if-then-else can be
                // performed.
                BddBitVector.ensureSameLength(result, elifThenVector);

                // Perform if-then-else.
                BddBitVector<?, ?> elifResult = BddBitVector.ifThenElseAny(elifGuards, elifThenVector, result);

                // Cleanup.
                elifGuards.free();
                elifThenVector.free();

                // Update the result.
                result.free();
                result = elifResult;
            }

            // Convert if/then.
            BDD ifGuards = convertPreds(ifExpr.getGuards(), initial, cifBddSpec, funcValuation);
            BddBitVector<?, ?> ifThenVector = convertExpr(ifExpr.getThen(), initial, cifBddSpec, funcValuation);

            // Get 'result' and if's 'then' in compatible representations.
            Pair<BddBitVector<?, ?>, BddBitVector<?, ?>> vectors = BddBitVector.ensureCompatible(
                    result, ifThenVector);
            result = vectors.left;
            ifThenVector = vectors.right;

            // Resize the 'result' and the if's 'then' to have the same length, such that the if-then-else can be
            // performed.
            BddBitVector.ensureSameLength(result, ifThenVector);

            // Perform if-then-else.
            BddBitVector<?, ?> ifResult = BddBitVector.ifThenElseAny(ifGuards, ifThenVector, result);

            // Cleanup.
            ifGuards.free();
            ifThenVector.free();

            // Update the result.
            result.free();
            result = ifResult;

            // Return the result.
            return result;
        }

        // Switch expression.
        if (expr instanceof SwitchExpression switchExpr) {
            Expression value = switchExpr.getValue();
            List<SwitchCase> cases = switchExpr.getCases();

            // Convert else.
            BddBitVector<?, ?> result = convertExpr(last(cases).getValue(), initial, cifBddSpec, funcValuation);

            // Convert cases.
            for (int i = cases.size() - 2; i >= 0; i--) {
                SwitchCase cse = cases.get(i);

                // Convert the case's guard and value.
                Expression caseGuardExpr = CifTypeUtils.isAutRefExpr(value) ? cse.getKey() : newBinaryExpression(
                        deepclone(value), BinaryOperator.EQUAL, null, deepclone(cse.getKey()), newBoolType());
                BDD caseGuard = convertPred(caseGuardExpr, initial, cifBddSpec, funcValuation);
                BddBitVector<?, ?> caseValueVector = convertExpr(cse.getValue(), initial, cifBddSpec, funcValuation);

                // Get 'result' and case's value in compatible representations.
                Pair<BddBitVector<?, ?>, BddBitVector<?, ?>> vectors = BddBitVector.ensureCompatible(
                        result, caseValueVector);
                result = vectors.left;
                caseValueVector = vectors.right;

                // Resize the 'result' and the case's value to have the same length, such that an if-then-else can be
                // performed.
                BddBitVector.ensureSameLength(result, caseValueVector);

                // Perform if-then-else.
                BddBitVector<?, ?> caseResult = BddBitVector.ifThenElseAny(caseGuard, caseValueVector, result);

                // Cleanup.
                caseGuard.free();
                caseValueVector.free();

                // Update the result.
                result.free();
                result = caseResult;
            }

            // Return the result.
            return result;
        }

        // Evaluate statically-evaluable expression.
        Object valueObj;
        try {
            valueObj = CifEvalUtils.eval(expr, initial);
        } catch (CifEvalException ex) {
            throw new AssertionError("Precondition violation.", ex);
        }

        // Create bit vector.
        if (valueObj instanceof Integer value) {
            // Create BDD bit vector for constant value.
            if (value < 0) {
                return SignedBddBitVector.createFromInt(cifBddSpec.factory, value);
            } else {
                return UnsignedBddBitVector.createFromInt(cifBddSpec.factory, value);
            }
        } else {
            // Get enumeration declaration and literal.
            Assert.check(valueObj instanceof CifEnumLiteral);
            EnumLiteral lit = ((CifEnumLiteral)valueObj).literal;
            EnumDecl enumDecl = (EnumDecl)lit.eContainer();

            // Create bit vector.
            int litIdx = enumDecl.getLiterals().indexOf(lit);
            return UnsignedBddBitVector.createFromInt(cifBddSpec.factory, litIdx);
        }
    }

    /**
     * Converts a call to an internal user-defined function to a BDD (in case of a function returning a boolean value)
     * or a BDD bit vector (in case of a function returning an integer or enumeration).
     *
     * @param func The internal user-defined function. Must be a function with a boolean, integer or enumeration as
     *     return value.
     * @param arguments The arguments to the function.
     * @param initial Whether the arguments apply only to the initial state ({@code true}) or any state ({@code false},
     *     includes the initial state).
     * @param cifBddSpec The CIF/BDD specification.
     * @param callerValuation The valuation of function parameters and local variables to their values (each as a
     *     {@link BDD} or {@link BddBitVector}), in the context of the function call. Must be {@code null} if the
     *     function call is not part of a function body.
     * @return The {@link BDD} or {@link BddBitVector} resulting from the conversion, indicating the value returned from
     *     the function.
     */
    private Object convertFuncCall(InternalFunction func, List<Expression> arguments,
            boolean initial, CifBddSpec cifBddSpec, FunctionValuation callerValuation)
    {
        // Sanity check. Function must return a boolean, integer or enumeration value.
        CifType returnType = CifTypeUtils.normalizeType(Lists.single(func.getReturnTypes()));
        Assert.check(returnType instanceof BoolType || returnType instanceof IntType || returnType instanceof EnumType);

        // Create the valuation that will hold the current values of the parameters and local variables. Each value is
        // either a BDD or a BDD bit vector.
        FunctionValuation bodyValuation = new FunctionValuation(arguments.size() + func.getVariables().size());

        // Add parameters with their arguments to the valuation.
        Assert.areEqual(func.getParameters().size(), arguments.size());
        for (int i = 0; i < arguments.size(); i++) {
            DiscVariable param = func.getParameters().get(i).getParameter();
            CifType paramType = normalizeType(param.getType());
            Expression argExpr = arguments.get(i);
            if (paramType instanceof BoolType) {
                bodyValuation.put(param, convertPred(argExpr, initial, cifBddSpec, callerValuation));
            } else {
                bodyValuation.put(param, convertExpr(argExpr, initial, cifBddSpec, callerValuation));
            }
        }

        // Add local variables with their initial values to the valuation. Local variables are ordered based on their
        // dependencies, as one may be initialized using another, regardless of the order in which the local variables
        // are declared.
        List<DiscVariable> localVars = func.getVariables();
        localVars = new FuncLocalVarOrderer().computeOrder(localVars);
        Assert.notNull(localVars); // Sanity check: no dependency cycles.
        for (DiscVariable localVar: localVars) {
            CifType varType = normalizeType(localVar.getType());
            Expression varValue = (localVar.getValue() == null) ? CifValueUtils.getDefaultValue(varType, null)
                    : Lists.single(localVar.getValue().getValues());
            if (varType instanceof BoolType) {
                bodyValuation.put(localVar, convertPred(varValue, false, cifBddSpec, bodyValuation));
            } else {
                bodyValuation.put(localVar, convertExpr(varValue, false, cifBddSpec, bodyValuation));
            }
        }

        // Return the BDD or bit vector representing the return value of the function.
        return convertFuncStatements(func.getStatements(), bodyValuation, cifBddSpec);
    }

    /**
     * Converts function statements of an internal user-defined function to a BDD (in case of a function returning a
     * boolean value) or a BDD bit vector (in case of a function returning an integer or enumeration).
     *
     * @param statements The function statements to consider.
     * @param bodyValuation The valuation of function parameters and local variables to their values (each as a
     *     {@link BDD} or {@link BddBitVector}), in the body of the function for which the statements are to be
     *     executed. Must never be {@code null}. It is {@link FunctionValuation#free freed} by this method.
     * @param cifBddSpec The CIF/BDD specification.
     * @return The {@link BDD} or {@link BddBitVector} resulting from the function statements, indicating the value
     *     returned from the function.
     */
    private Object convertFuncStatements(List<FunctionStatement> statements,
            FunctionValuation bodyValuation, CifBddSpec cifBddSpec)
    {
        // Separate first statement and remaining statements.
        FunctionStatement statement = Lists.first(statements);
        List<FunctionStatement> remainingStatements = Lists.slice(statements, 1, null);

        // Handle statements differently depending on the first statement.
        if (statement instanceof AssignmentFuncStatement assign) {
            // Note that we have a precondition that the type of the value must be contained in the type of the
            // addressable. Hence, the assignment itself won't result in a runtime out-of-range error.

            // Get assigned variable.
            Assert.check(assign.getAddressable() instanceof DiscVariableExpression);
            DiscVariable assignedVar = ((DiscVariableExpression)assign.getAddressable()).getVariable();

            // Get value expression.
            Expression valueExpr = assign.getValue();

            // Update the valuation.
            CifType varType = normalizeType(assignedVar.getType());
            if (varType instanceof BoolType) {
                BDD newValue = convertPred(valueExpr, false, cifBddSpec, bodyValuation);
                ((BDD)bodyValuation.get(assignedVar)).free();
                bodyValuation.put(assignedVar, newValue);
            } else {
                BddBitVector<?, ?> newValue = convertExpr(valueExpr, false, cifBddSpec, bodyValuation);
                ((BddBitVector<?, ?>)bodyValuation.get(assignedVar)).free();
                bodyValuation.put(assignedVar, newValue);
            }

            // Convert the remaining statements.
            return convertFuncStatements(remainingStatements, bodyValuation, cifBddSpec);
        } else if (statement instanceof IfFuncStatement ifStatement) {
            // Consider:
            //   if g1, g2: <stats1> elif g3, g4: <stats2> ... else <stats3> end ; <stats4>
            //
            // We generate:
            //   if g1 and g2 -> <stats1> ; <stats4>
            //   else if g3 and g4 -> <stats2> ; <stats4>
            //   ...
            //   else <stats3> ; <stats4>
            //   end
            //
            // Note that:
            // - The 'then', 'elif/then' and 'else' statements can be empty, as can the statements after the 'if', in
            //   case each 'if', 'elif/then' and 'else' ends with a 'return' statement. But, each path must end with a
            //   'return', so in all cases there is at least one statement for the path.
            // - No 'else' statements means either no 'else' at all, or an 'else' without any statements. They are
            //   treated in the same way.
            // - Since we have to handle multiple code paths, we create a unique valuation for each of them to update
            //   in-place.

            // Handle 'if'/'then' branch.
            BDD ifGuard = convertPreds(ifStatement.getGuards(), false, cifBddSpec, bodyValuation);
            FunctionValuation ifValuation = bodyValuation.copy();
            Object ifReturn = convertFuncStatements(concat(ifStatement.getThens(), remainingStatements), ifValuation,
                    cifBddSpec);

            // Handle 'elif'/'then' branches.
            List<ElifFuncStatement> elifStatements = ifStatement.getElifs();
            List<BDD> elifGuards = listc(elifStatements.size());
            List<Object> elifReturns = listc(elifStatements.size());
            for (ElifFuncStatement elifStatement: elifStatements) {
                elifGuards.add(convertPreds(elifStatement.getGuards(), false, cifBddSpec, bodyValuation));
                FunctionValuation elifValuation = bodyValuation.copy();
                elifReturns.add(convertFuncStatements(concat(elifStatement.getThens(), remainingStatements),
                        elifValuation, cifBddSpec));
            }

            // Handle 'else' branch (including no 'else' present).
            FunctionValuation elseValuation = bodyValuation.copy();
            Object elseReturn = convertFuncStatements(concat(ifStatement.getElses(), remainingStatements),
                    elseValuation, cifBddSpec);

            // Cleanup.
            bodyValuation.free();

            // Form the resulting 'if' predicate or 'if' BDD bit vector.
            if (elseReturn instanceof BDD elseReturnBdd) {
                // 'else'.
                BDD rslt = elseReturnBdd;

                // 'elif'/'then'.
                for (int i = elifStatements.size() - 1; i >= 0; i--) {
                    BDD elifGuard = elifGuards.get(i);
                    BDD elifThen = (BDD)elifReturns.get(i);
                    BDD elifRslt = elifGuard.ite(elifThen, rslt);
                    elifGuard.free();
                    elifThen.free();
                    rslt.free();
                    rslt = elifRslt;
                }

                // 'if'/'then'.
                BDD ifRslt = ifGuard.ite((BDD)ifReturn, rslt);
                ifGuard.free();
                ((BDD)ifReturn).free();
                rslt.free();
                rslt = ifRslt;

                // Return return value of the entire 'if'.
                return rslt;
            } else if (elseReturn instanceof BddBitVector elseReturnVector) {
                // 'else'.
                BddBitVector<?, ?> rslt = elseReturnVector;

                // 'elif'/'then'.
                for (int i = elifStatements.size() - 1; i >= 0; i--) {
                    // Get guard.
                    BDD elifGuard = elifGuards.get(i);

                    // Get 'elifThen' and 'rslt', in compatible representation and same lengths.
                    BddBitVector<?, ?> elifThen = (BddBitVector<?, ?>)elifReturns.get(i);
                    Pair<BddBitVector<?, ?>, BddBitVector<?, ?>> vectors = BddBitVector.ensureCompatible(elifThen,
                            rslt);
                    elifThen = vectors.left;
                    rslt = vectors.right;
                    BddBitVector.ensureSameLength(elifThen, rslt);

                    // Compute 'if <elifGuard>: <elifThen> else <rslt> end'.
                    BddBitVector<?, ?> elifRslt = BddBitVector.ifThenElseAny(elifGuard, elifThen, rslt);

                    // Clean.
                    elifGuard.free();
                    elifThen.free();
                    rslt.free();

                    // Update the 'rslt'.
                    rslt = elifRslt;
                }

                // 'if'/'then'. First, ensure compatible representations and same lengths for 'ifReturn' and 'rslt'.
                BddBitVector<?, ?> ifReturnVector = (BddBitVector<?, ?>)ifReturn;
                Pair<BddBitVector<?, ?>, BddBitVector<?, ?>> vectors = BddBitVector.ensureCompatible(ifReturnVector,
                        rslt);
                ifReturnVector = vectors.left;
                rslt = vectors.right;
                BddBitVector.ensureSameLength(ifReturnVector, rslt);

                // Compute 'if <ifGuard>: <ifReturn> else <rslt> end'.
                BddBitVector<?, ?> ifRslt = BddBitVector.ifThenElseAny(ifGuard, ifReturnVector, rslt);

                // Clean up.
                ifGuard.free();
                ((BddBitVector<?, ?>)ifReturn).free();
                rslt.free();

                // Get final 'rslt'.
                rslt = ifRslt;

                // Return return value of the entire 'if'.
                return rslt;
            } else {
                throw new AssertionError("Unexpected 'if' return: " + ifReturn);
            }
        } else if (statement instanceof ReturnFuncStatement returnStatement) {
            // Convert the return expression. Ignore any remaining statements. Note that the CIF type checker already
            // ensures that the returned value's type is contained in the return type of the function, preventing
            // out-of-range errors.
            Expression returnExpr = Lists.single(returnStatement.getValues());
            CifType returnType = normalizeType(returnExpr.getType());
            Object returnValue = switch (returnType) {
                case BoolType boolType -> convertPred(returnExpr, false, cifBddSpec, bodyValuation);
                default -> convertExpr(returnExpr, false, cifBddSpec, bodyValuation);
            };
            bodyValuation.free();
            return returnValue;
        } else {
            throw new AssertionError("Unsupported function statement: " + statement);
        }
    }

    /**
     * A valuation of internal user-defined function parameters and local variables to their values (each as a
     * {@link BDD} or {@link BddBitVector}).
     */
    private static class FunctionValuation extends HashMap<DiscVariable, Object> {
        /**
         * Constructor for the {@link FunctionValuation} class.
         *
         * @param size The initial capacity of the valuation.
         */
        FunctionValuation(int size) {
            super(size);
        }

        /**
         * Creates a copy of this valuation.
         *
         * @return The copy.
         */
        FunctionValuation copy() {
            FunctionValuation copy = new FunctionValuation(this.size());
            for (Entry<DiscVariable, Object> entry: this.entrySet()) {
                Object value = entry.getValue();
                if (value instanceof BDD bdd) {
                    value = bdd.id();
                } else if (value instanceof BddBitVector vector) {
                    value = vector.copy();
                } else {
                    throw new AssertionError("Unexpected value: " + value);
                }
                copy.put(entry.getKey(), value);
            }
            return copy;
        }

        /** Frees this valuation. */
        void free() {
            for (Object value: values()) {
                if (value instanceof BDD bdd) {
                    bdd.free();
                } else if (value instanceof BddBitVector vector) {
                    vector.free();
                } else {
                    throw new AssertionError("Unexpected value: " + value);
                }
            }
        }
    }

    /** Frees this converter, releasing its BDD-related resources and clearing its caches. */
    public void free() {
        // Free and clear conversion BDD cache.
        for (BDD bdd: conversionBddCache.values()) {
            bdd.free();
        }
        conversionBddCache.clear();

        // Free and clear conversion bit vector cache.
        for (BddBitVector<?, ?> vector: conversionBitVectorCache.values()) {
            vector.free();
        }
        conversionBitVectorCache.clear();
    }

    /**
     * Collect the events declared in the given component (recursively).
     *
     * @param comp The component.
     * @param events The events collected so far. Is modified in-place.
     */
    public static void collectEvents(ComplexComponent comp, List<Event> events) {
        // Collect locally.
        for (Declaration decl: comp.getDeclarations()) {
            if (decl instanceof Event) {
                events.add((Event)decl);
            }
        }

        // Collect recursively.
        if (comp instanceof Group) {
            for (Component child: ((Group)comp).getComponents()) {
                collectEvents((ComplexComponent)child, events);
            }
        }
    }

    /**
     * Collect the automata of the given component (recursively).
     *
     * @param comp The component.
     * @param automata The automata collected so far. Is modified in-place.
     */
    private static void collectAutomata(ComplexComponent comp, List<Automaton> automata) {
        if (comp instanceof Automaton) {
            // Add automaton.
            automata.add((Automaton)comp);
        } else {
            // Collect recursively.
            for (Component child: ((Group)comp).getComponents()) {
                collectAutomata((ComplexComponent)child, automata);
            }
        }
    }

    /**
     * Collect CIF objects for which CIF/BDD variables need to be constructed.
     *
     * @param comp The component in which to collect (recursively).
     * @param objs The objects collected so far. Is modified in-place.
     */
    private static void collectVariableObjects(ComplexComponent comp, List<PositionObject> objs) {
        // Collect automaton locally.
        if (comp instanceof Automaton) {
            Automaton aut = (Automaton)comp;
            if (aut.getLocations().size() > 1) {
                objs.add(aut);
            }
        }

        // Collect variables locally.
        for (Declaration decl: comp.getDeclarations()) {
            if (decl instanceof DiscVariable) {
                objs.add(decl);
            }
            if (decl instanceof InputVariable) {
                objs.add(decl);
            }
        }

        // Collect recursively.
        if (comp instanceof Group) {
            for (Component child: ((Group)comp).getComponents()) {
                collectVariableObjects((ComplexComponent)child, objs);
            }
        }
    }

    /**
     * Returns the CIF/BDD variable index of the given CIF discrete variable.
     *
     * @param vars The CIF/BDD variables, in which to look for the given CIF discrete variable. May not be a
     *     dummy/internal location pointer variable created for an automaton with two or more locations.
     * @param var The CIF discrete variable to look for, and for which the index is to be returned.
     * @return The 0-based index of the CIF/BDD variable.
     */
    public static int getDiscVarIdx(CifBddVariable[] vars, DiscVariable var) {
        // Make sure the given discrete variable is an actual discrete variable, and not a dummy one created for a
        // location pointer of an automaton.
        Assert.check(var.getType() != null);

        // Look up the discrete variable.
        return getTypedVarIdx(vars, var);
    }

    /**
     * Returns the CIF/BDD variable index of the given CIF input variable.
     *
     * @param vars The CIF/BDD variables, in which to look for the given CIF input variable.
     * @param var The CIF input variable to look for, and for which the index is to be returned.
     * @return The 0-based index of the CIF/BDD variable.
     */
    public static int getInputVarIdx(CifBddVariable[] vars, InputVariable var) {
        return getTypedVarIdx(vars, var);
    }

    /**
     * Returns the CIF/BDD variable index of the given CIF typed object, i.e. a discrete variable or an input variable.
     *
     * @param vars The CIF/BDD variables, in which to look for the given CIF typed object. May not be a dummy/internal
     *     location pointer variable created for an automaton with two or more locations.
     * @param var The CIF variable to look for, and for which the index is to be returned.
     * @return The 0-based index of the CIF/BDD variable.
     */
    public static int getTypedVarIdx(CifBddVariable[] vars, Declaration var) {
        for (int i = 0; i < vars.length; i++) {
            CifBddVariable cifBddVar = vars[i];
            if (!(cifBddVar instanceof CifBddTypedVariable)) {
                continue;
            }
            CifBddTypedVariable cifBddVarTypedVar = (CifBddTypedVariable)cifBddVar;
            if (cifBddVarTypedVar.obj == var) {
                return i;
            }
        }
        throw new AssertionError("Unexpected variable: " + var);
    }

    /**
     * Returns the CIF/BDD variable index of the given CIF automaton, or rather the location pointer variable for that
     * automaton.
     *
     * @param vars The CIF/BDD variables, in which to look for the given CIF automaton.
     * @param aut The CIF automaton to look for, and for which the index is to be returned.
     * @return The 0-based index of the CIF/BDD variable, or {@code -1} if it is not found. If not found, no location
     *     pointer was created for the automaton because it only has one location.
     */
    public static int getLpVarIdx(CifBddVariable[] vars, Automaton aut) {
        for (int i = 0; i < vars.length; i++) {
            CifBddVariable cifBddVar = vars[i];
            if (!(cifBddVar instanceof CifBddLocPtrVariable)) {
                continue;
            }
            CifBddLocPtrVariable cifBddLpVar = (CifBddLocPtrVariable)cifBddVar;
            if (cifBddLpVar.aut == aut) {
                return i;
            }
        }
        Assert.areEqual(aut.getLocations().size(), 1);
        return -1;
    }
}
