/*******************************************************************************
 * Copyright (c) 2011 protos software gmbh (http://www.protos.de).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * CONTRIBUTORS:
 * 		Henrik Rentz-Reichert (initial contribution)
 * 		Thomas Schuetz (changed for C code generator)
 *
 *******************************************************************************/

package org.eclipse.etrice.generator.c.gen

import com.google.inject.Inject
import com.google.inject.Singleton
import org.eclipse.etrice.core.fsm.fSM.ComponentCommunicationType
import org.eclipse.etrice.core.genmodel.etricegen.ExpandedActorClass
import org.eclipse.etrice.core.genmodel.etricegen.Root
import org.eclipse.etrice.core.genmodel.fsm.FsmGenExtensions
import org.eclipse.etrice.core.room.CommunicationType
import org.eclipse.etrice.core.room.InterfaceItem
import org.eclipse.etrice.core.room.Message
import org.eclipse.etrice.core.room.Operation
import org.eclipse.etrice.core.room.RoomModel
import org.eclipse.etrice.generator.base.io.IGeneratorFileIO
import org.eclipse.etrice.generator.base.logging.ILogger
import org.eclipse.etrice.generator.c.setup.GeneratorOptionsHelper
import org.eclipse.etrice.generator.generic.GenericActorClassGenerator
import org.eclipse.etrice.generator.generic.ILanguageExtension
import org.eclipse.etrice.generator.generic.ProcedureHelpers

@Singleton
class ActorClassGen extends GenericActorClassGenerator {

	@Inject protected extension CExtensions
	@Inject protected extension ProcedureHelpers
	@Inject protected extension StateMachineGen
	@Inject protected extension GeneratorOptionsHelper

	@Inject protected ILanguageExtension langExt
	@Inject protected IGeneratorFileIO fileIO
	@Inject protected ILogger logger

	def doGenerate(Root root) {
		root.actorClasses.filter[!isDeprecatedGeneration].map[root.getExpandedActorClass(it)].forEach[xpac |
			val path = xpac.actorClass.getPath
			var file = xpac.actorClass.getCHeaderFileName

			// header file
			fileIO.generateFile("generating ActorClass header", path + file, root.generateHeaderFile(xpac))

			// utils file
			file = xpac.actorClass.getCUtilsFileName
			fileIO.generateFile("generating ActorClass utils", path + file, root.generateUtilsFile(xpac))

			// source file
			if (xpac.actorClass.isBehaviorAnnotationPresent("BehaviorManual")) {
				logger.logInfo("omitting ActorClass source for '" + xpac.actorClass.name + "' since @BehaviorManual is specified")
			}
			else {
				file = xpac.actorClass.getCSourceFileName
				fileIO.generateFile("generating ActorClass source", path + file, root.generateSourceFile(xpac))
			}
		]
	}

	def protected generateHeaderFile(Root root, ExpandedActorClass xpac) {
		val ac = xpac.actorClass
		val eventPorts = ac.allEndPorts.filter[protocol.commType==CommunicationType::EVENT_DRIVEN]
		val sendPorts = ac.allEndPorts.filter[protocol.commType==CommunicationType::DATA_DRIVEN && conjugated]
		val recvPorts = ac.allEndPorts.filter[protocol.commType==CommunicationType::DATA_DRIVEN && !conjugated]
		val dataDriven = ac.commType==ComponentCommunicationType::DATA_DRIVEN
		val async = ac.commType==ComponentCommunicationType::ASYNCHRONOUS
		val hasConstData = !(eventPorts.empty && recvPorts.empty && ac.allSAPs.empty && ac.allServiceImplementations.empty)
		val isEmptyStateGraph = FsmGenExtensions.isEmpty(xpac.graphContainer.graph)

	'''
		/**
		 * @author generated by eTrice
		 *
		 * Header File of ActorClass ac.name
		 *
		 */

		generateIncludeGuardBegin(ac)

		#include "modelbase/etActor.h"

		/* include all referenced room classes */
		FOR path : (root.getReferencedDataClasses(ac) + root.getReferencedEnumClasses(ac) + root.getReferencedProtocolClasses(ac)).map[includePath].sort
			#include path
		ENDFOR

		ac.userCode(1, true)

		typedef struct ac.name ac.name;

		/* const part of ActorClass (ROM) */
		typedef struct ac.name_const {
			#ifdef ET_MSC_LOGGER_ACTIVATE
				const char* instName;
			#endif
			IF xpac.isTracingEnabled
				#if defined ET_MSC_TRACER_ACTIVATE || defined ET_MSC_LOGGER_ACTIVATE
					etInt16 objId;
				#endif
			ENDIF
			IF !hasConstData
				#if !defined(ET_MSC_LOGGER_ACTIVATE) IF xpac.isTracingEnabled&& !defined(ET_MSC_TRACER_ACTIVATE)ENDIF
					/* This actor class has no const data. To keep this case simple we introduce this dummy variable. */
					int dummy;
				#endif
			ENDIF			
			
			/* simple ports */
			FOR ep : eventPorts
				IF ep.multiplicity==1
					const ep.getPortClassName() ep.name;
				ENDIF
			ENDFOR

			/* data receive ports */
			FOR ep : recvPorts
				IF ep.multiplicity==1
					const ep.getPortClassName() ep.name;
				ENDIF
			ENDFOR

			/* saps */
			FOR sap: ac.allSAPs
				const sap.getPortClassName() sap.name;
			ENDFOR

			/* replicated ports */
			FOR ep : ac.allEndPorts
				IF ep.multiplicity!=1
					const etReplPort ep.name;
				ENDIF
			ENDFOR

			/* services */
			FOR svc : ac.allServiceImplementations
				const etReplPort svc.spp.name;
			ENDFOR
		} ac.name_const;

		IF !isEmptyStateGraph

			xpac.genHeaderConstants
		ENDIF

		/* variable part of ActorClass (RAM) */
		struct ac.name {
			const ac.name_const* ET_CONST_MEMBER constData;

			/* data send ports */
			FOR ep : sendPorts
				IF ep.multiplicity==1
					ep.getPortClassName() ep.name;
				ENDIF
			ENDFOR

			ac.allAttributes.attributes

			IF !isEmptyStateGraph

				xpac.genDataMembers
			ENDIF
		};

		void ac.name_init(ac.name* self);

		void ac.name_receiveMessage(void* self, const void* ifitem, const etMessage* msg);

		IF dataDriven || async
			void ac.name_execute(ac.name* self);
		ENDIF

		ac.userStructorsDeclaration

		ac.latestOperations.operationsDeclaration(ac.name)

		ac.userCode(2, true)

		generateIncludeGuardEnd(ac)

	'''
	}

	def protected generateUtilsFile(Root root, ExpandedActorClass xpac) {
		val ac = xpac.actorClass
		val eventPorts = ac.allEndPorts.filter[protocol.commType==CommunicationType::EVENT_DRIVEN]
		val replEventPorts = eventPorts.filter[multiplicity!=1]
		val sendPorts = ac.allEndPorts.filter[protocol.commType==CommunicationType::DATA_DRIVEN &&  conjugated && multiplicity==1]
		val recvPorts = ac.allEndPorts.filter[protocol.commType==CommunicationType::DATA_DRIVEN && !conjugated && multiplicity==1]
		val portsWithOperations = ac.allInterfaceItems.filter(p|p.portClass!==null && p.portClass.operations.size>0)
		val filename = (ac.eContainer as RoomModel).name.replaceAll("\\.","_")+"_"+ac.name+"_Utils"

	'''
		/**
		 * @author generated by eTrice
		 *
		 * Utils File of ActorClass ac.name
		 *
		 */

		generateIncludeGuardBegin(filename)

		#include ac.includePath

		/*
		 * access macros for ports, operations and attributes
		*/

		/* simple event ports */
		FOR ep : eventPorts.filter[multiplicity==1]
			FOR msg : ep.outgoing
				val data1 = if (msg.data!==null) "data"
				val data2 = if (msg.data!==null) ", data"
				generateDefine('''ep.name_msg.name(data1)''', #[
					'''ep.portClassName_msg.name(&self->constData->ep.namedata2)''',
					xpac.tracingCall(ep, false, msg, false)
				])
			ENDFOR
		ENDFOR

		/* data receive ports */
		FOR ep : recvPorts
			FOR msg : ep.incoming
				#define ep.name_msg.name ep.portClassName_msg.name_get(&self->constData->ep.name)
			ENDFOR
		ENDFOR

		/* data send ports */
		FOR ep : sendPorts
			FOR msg : ep.outgoing
				val data1 = if (msg.data!==null) "data"
				val data2 = if (msg.data!==null) ", data"
				#define ep.name_msg.name(data1) ep.portClassName_msg.name_set(&self->ep.namedata2)
			ENDFOR
		ENDFOR

		/* saps */
		FOR sap: ac.allSAPs
			FOR msg : sap.outgoing
				val data1 = if (msg.data!==null) "data"
				val data2 = if (msg.data!==null) ", data"
				generateDefine('''sap.name_msg.name(data1)''', #[
					'''sap.portClassName_msg.name(&self->constData->sap.namedata2)''',
					xpac.tracingCall(sap, false, msg, false)
				])
			ENDFOR
		ENDFOR

		/* replicated event ports */
		IF !replEventPorts.empty
			#define ifitem_index (((etReplSubPort*)ifitem)->index)
		ENDIF
		FOR ep : replEventPorts
			FOR msg : ep.outgoing
				val data1 = if (msg.data!==null) "data"
				val data2 = if (msg.data!==null) ", data"
				generateDefine('''ep.name_msg.name_broadcast(data1)''', #[
					'''ep.portClassName_msg.name_broadcast(&self->constData->ep.namedata2)''',
					xpac.tracingCall(ep, false, msg, true)
				])
				generateDefine('''ep.name_msg.name(idxdata2)''', #[
					'''ep.portClassName_msg.name(&self->constData->ep.name, idxdata2)''',
					xpac.tracingCall(ep, true, msg, false)
				])
			ENDFOR
		ENDFOR

		/* services */
		FOR svc : ac.allServiceImplementations
			FOR msg : svc.spp.outgoing
				val data1 = if (msg.data!==null) "data"
				val data2 = if (msg.data!==null) ", data"
				generateDefine('''svc.spp.name_msg.name_broadcast(data1)''', #[
					'''svc.spp.portClassName_msg.name_broadcast(&self->constData->svc.spp.namedata2)''',
					xpac.tracingCall(svc.spp, false, msg, true)
				])
				generateDefine('''svc.spp.name_msg.name(idxdata2)''', #[
					'''svc.spp.portClassName_msg.name(&self->constData->svc.spp.name, idxdata2)''',
					xpac.tracingCall(svc.spp, true, msg, false)
				])
			ENDFOR
		ENDFOR

		/* operations */
		FOR op : ac.latestOperations
			val params = op.operationParams
			val args = op.operationArgs
			#define op.name(params) ac.name_op.name(selfIF !op.arguments.empty, argsENDIF)
		ENDFOR

		/* attributes */
		FOR a : ac.allAttributes
			#define a.name (self->a.name)
		ENDFOR

		/* port operations */
		FOR p : portsWithOperations
			FOR op : p.portClass.operations
				val params = op.operationParams
				val args = op.operationArgs
				#define p.name_op.name(params) p.portClassName_op.name(&self->constData->p.nameIF !op.arguments.empty, argsENDIF)
			ENDFOR
		ENDFOR

		generateIncludeGuardEnd(filename)

	'''
	}
	
	private def generateDefine(String sig, Iterable<String> lines) {
		val filteredLines = lines.filterNull.toList
		switch filteredLines {
			case filteredLines.size > 1: '''
				#define sig { filteredLines.headIF filteredLines.size > 1; \ENDIF
								filteredLines.tail.join('; \\\n') }
			'''
			default: '''#define sig filteredLines.head'''
		}
	}
	
	private def tracingCall(ExpandedActorClass xpac, InterfaceItem item, boolean idx, Message msg, boolean broadcast) {
		if(xpac.isTracingEnabled) {
			val dir = if (item.conjugated) "IN_" else "OUT_"
			val peerAddress =  switch item {
				case idx: '''((self->constData->item.name.size>(etInt16)idx) ? self->constData->item.name.ports[idx].port.peerAddress : 1)'''
				case broadcast: '''((self->constData->item.name.size > 0) ? self->constData->item.name.ports[0].port.peerAddress : 1)'''
				default: '''self->constData->item.name.peerAddress'''
			}
			val traceMacro = if(broadcast) 'ET_MSC_TRACER_ASYNC_BCAST' else 'ET_MSC_TRACER_ASYNC_OUT'
			return '''traceMacro(peerAddress, memberInUse(item.protocol.name, dir+msg.name));'''
		}
	}

	private def operationParams(Operation op) {
		op.arguments.map[
			switch it {
				case isVarargs: '...'
				default: name
			}
		].join(', ')
	}
	
	private def operationArgs(Operation op) {
		op.arguments.map[
			switch it {
				case isVarargs: '__VA_ARGS__'
				default: name
			}
		].join(', ')
	}

	def protected generateSourceFile(Root root, ExpandedActorClass xpac) {
		val ac = xpac.actorClass
		val async = ac.commType==ComponentCommunicationType::ASYNCHRONOUS
		val eventDriven = ac.commType==ComponentCommunicationType::EVENT_DRIVEN
		val dataDriven = ac.commType==ComponentCommunicationType::DATA_DRIVEN
		val handleEvents = async || eventDriven
		val isEmptyStateGraph = FsmGenExtensions.isEmpty(xpac.graphContainer.graph)

	'''
		/**
		 * @author generated by eTrice
		 *
		 * Source File of ActorClass ac.name
		 *
		 */

		#include "ac.getCHeaderFileName"

		#include "debugging/etLogger.h"
		#include "debugging/etMSCLogger.h"
		#include "etUnit/etUnit.h"
		#include "base/etMemory.h"
		#ifdef ET_MSC_TRACER_ACTIVATE
			#include "debugging/etMSCTracer.h"
		#endif

		FOR path : root.getReferencedProtocolClasses(ac).map[includePath].sort
			#include path
		ENDFOR

		#include "ac.getCUtilsFileName"

		ac.userCode(3, true)

		/* interface item IDs */
		xpac.genInterfaceItemConstants

		IF !isEmptyStateGraph
			xpac.graphContainer.genStateMachine
		ENDIF

		void ac.name_init(ac.name* self){
			ET_MSC_LOGGER_SYNC_ENTRY("ac.name", "init")
			IF !isEmptyStateGraph
				xpac.genInitialization
			ENDIF
			ET_MSC_LOGGER_SYNC_EXIT
		}


		void ac.name_receiveMessage(void* self, const void* ifitem, const etMessage* msg){
			ET_MSC_LOGGER_SYNC_ENTRY("ac.name", "_receiveMessage")
			IF !isEmptyStateGraph
				IF handleEvents
					IF xpac.isTracingEnabledET_MSC_TRACER_ASYNC_IN(((etPort*)ifitem)->peerAddress, msg->evtID);ENDIF
					langExt.operationScope(ac.name, false)receiveEvent((ac.name*) self, (etPort*)ifitem, msg->evtID, (void*)(((char*)msg)+MEM_CEIL(sizeof(etMessage))));
				ELSE
					langExt.operationScope(ac.name, false)receiveEventInternal((ac.name*) self);
				ENDIF
			ENDIF

			ET_MSC_LOGGER_SYNC_EXIT
		}

		IF dataDriven || async
			void ac.name_execute(ac.name* self) {
				ET_MSC_LOGGER_SYNC_ENTRY("ac.name", "_execute")
				IF !isEmptyStateGraph

					IF handleEvents
						langExt.operationScope(ac.name, false)receiveEvent(self, NULL, 0, NULL);
					ELSE
						langExt.operationScope(ac.name, false)receiveEventInternal(self);
					ENDIF
				ENDIF

				ET_MSC_LOGGER_SYNC_EXIT
			}
		ENDIF

		ac.userStructorsImplementation

		operationsImplementation(ac.latestOperations, ac.name)

		'''
	}
}
