/*******************************************************************************
 * Copyright (c) 2017 Pivotal Inc. and others.
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *  Martin Lippert (Pivotal Inc.) - initial implementation
 *******************************************************************************/
package org.eclipse.lsp4e.test.edit;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;

import java.io.File;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import org.eclipse.core.filesystem.EFS;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.lsp4e.LSPEclipseUtils;
import org.eclipse.lsp4e.LanguageServiceAccessor;
import org.eclipse.lsp4e.test.AllCleanRule;
import org.eclipse.lsp4e.test.TestUtils;
import org.eclipse.lsp4e.tests.mock.MockLanguageServer;
import org.eclipse.lsp4j.DidOpenTextDocumentParams;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.texteditor.AbstractTextEditor;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;

public class DocumentDidOpenTest {

	@Rule public AllCleanRule clear = new AllCleanRule();
	private IProject project;

	@Before
	public void setUp() throws CoreException {
		project =  TestUtils.createProject("DocumentDidChangeTest"+System.currentTimeMillis());
	}

	@Test
	public void testOpen() throws Exception {
		IFile testFile = TestUtils.createUniqueTestFile(project, "");
		
		CompletableFuture<DidOpenTextDocumentParams> didOpenExpectation = new CompletableFuture<DidOpenTextDocumentParams>();
		MockLanguageServer.INSTANCE.setDidOpenCallback(didOpenExpectation);

		TestUtils.openEditor(testFile);

		// Force LS to initialize and open file
		IDocument document = LSPEclipseUtils.getDocument(testFile);
		assertNotNull(document);
		LanguageServiceAccessor.getLanguageServers(document, capabilites -> Boolean.TRUE);

		DidOpenTextDocumentParams lastOpen = didOpenExpectation.get(1000, TimeUnit.MILLISECONDS);
		assertNotNull(lastOpen.getTextDocument());
		assertEquals("lspt", lastOpen.getTextDocument().getLanguageId());
	}

	@Test
	public void testOpenExternalFile() throws Exception {
		File file = TestUtils.createTempFile("testOpenExternalFile", ".lspt");
		CompletableFuture<DidOpenTextDocumentParams> didOpenExpectation = new CompletableFuture<DidOpenTextDocumentParams>();
		MockLanguageServer.INSTANCE.setDidOpenCallback(didOpenExpectation);
		IEditorPart editor = IDE.openEditorOnFileStore(PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage(), EFS.getStore(file.toURI()));
		// Force LS to initialize and open file
		LanguageServiceAccessor.getLanguageServers(LSPEclipseUtils.getDocument(editor.getEditorInput()), capabilites -> Boolean.TRUE);

		DidOpenTextDocumentParams lastOpen = didOpenExpectation.get(1000, TimeUnit.MILLISECONDS);
		assertNotNull(lastOpen.getTextDocument());
		assertEquals("lspt", lastOpen.getTextDocument().getLanguageId());
	}

	@Test
	public void testOpenWithSpecificLanguageId() throws Exception {
		IFile testFile = TestUtils.createUniqueTestFile(project, "lspt-different", "");
		
		CompletableFuture<DidOpenTextDocumentParams> didOpenExpectation = new CompletableFuture<DidOpenTextDocumentParams>();
		MockLanguageServer.INSTANCE.setDidOpenCallback(didOpenExpectation);

		IEditorPart editor = TestUtils.openEditor(testFile);

		// Force LS to initialize and open file
		IDocument document = LSPEclipseUtils.getDocument(testFile);
		assertNotNull(document);
		LanguageServiceAccessor.getLanguageServers(document, capabilites -> Boolean.TRUE);

		DidOpenTextDocumentParams lastOpen = didOpenExpectation.get(1000, TimeUnit.MILLISECONDS);
		assertNotNull(lastOpen.getTextDocument());
		assertEquals("differentLanguageId", lastOpen.getTextDocument().getLanguageId());

		((AbstractTextEditor)editor).close(false);
	}

}
