"use strict";
/*********************************************************************
 * Copyright (c) 2023 Ericsson and others
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *********************************************************************/
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const chai_1 = require("chai");
const utils_1 = require("./utils");
describe('Stepping', function () {
    return __awaiter(this, void 0, void 0, function* () {
        let dc;
        const steppingProgram = path.join(utils_1.testProgramsDir, 'stepping');
        const steppingSource = path.join(utils_1.testProgramsDir, 'stepping.c');
        const lineTags = {
            'main for': 0,
            'main getFromElsewhere call': 0,
            'main printf call': 0,
            'getFromElsewhere entry': 0,
            'getFromElsewhere for': 0,
        };
        before(function () {
            (0, utils_1.resolveLineTagLocations)(steppingSource, lineTags);
        });
        beforeEach(function () {
            return __awaiter(this, void 0, void 0, function* () {
                dc = yield (0, utils_1.standardBeforeEach)();
                yield dc.hitBreakpoint((0, utils_1.fillDefaults)(this.currentTest, { program: steppingProgram }), {
                    path: steppingSource,
                    line: lineTags['main getFromElsewhere call'],
                });
            });
        });
        afterEach(() => __awaiter(this, void 0, void 0, function* () {
            yield dc.stop();
        }));
        function getFrameState(threadId) {
            return __awaiter(this, void 0, void 0, function* () {
                const stack = yield dc.stackTraceRequest({ threadId });
                const main = stack.body.stackFrames.find((frame) => frame.name === 'main');
                const elsewhere = stack.body.stackFrames.find((frame) => frame.name === 'getFromElsewhere');
                return { main, elsewhere };
            });
        }
        function expectStackState(state, check) {
            var _a;
            if (check.elsewhereDefined) {
                (0, chai_1.expect)(state.elsewhere).not.to.be.undefined;
            }
            else {
                (0, chai_1.expect)(state.elsewhere).to.be.undefined;
            }
            const target = check.elsewhereDefined ? 'elsewhere' : 'main';
            (0, chai_1.expect)(state[target]).not.to.be.undefined;
            (0, chai_1.expect)((_a = state[target]) === null || _a === void 0 ? void 0 : _a.line).equal(check.line, `It should have stopped at line ${check.line}`);
        }
        it('steps in by line', () => __awaiter(this, void 0, void 0, function* () {
            const threads = yield dc.threadsRequest();
            const threadId = threads.body.threads[0].id;
            expectStackState(yield getFrameState(threadId), {
                elsewhereDefined: false,
                line: lineTags['main getFromElsewhere call'],
            });
            yield Promise.all([
                dc.stepInRequest({ threadId, granularity: 'statement' }),
                dc.waitForEvent('stopped'),
            ]);
            expectStackState(yield getFrameState(threadId), {
                elsewhereDefined: true,
                line: lineTags['getFromElsewhere entry'],
            });
            yield Promise.all([
                dc.stepInRequest({ threadId, granularity: 'statement' }),
                dc.waitForEvent('stopped'),
            ]);
            expectStackState(yield getFrameState(threadId), {
                elsewhereDefined: true,
                line: lineTags['getFromElsewhere for'],
            });
        }));
        it('steps in by instruction', () => __awaiter(this, void 0, void 0, function* () {
            var _a;
            const threads = yield dc.threadsRequest();
            const threadId = threads.body.threads[0].id;
            let state = yield getFrameState(threadId);
            expectStackState(state, {
                elsewhereDefined: false,
                line: lineTags['main getFromElsewhere call'],
            });
            yield Promise.all([
                dc.stepInRequest({ threadId, granularity: 'instruction' }),
                dc.waitForEvent('stopped'),
            ]);
            // First step should not take us straight to the function.
            expectStackState((state = yield getFrameState(threadId)), {
                elsewhereDefined: false,
                line: lineTags['main getFromElsewhere call'],
            });
            // Step until we leave that line.
            while (((_a = state.main) === null || _a === void 0 ? void 0 : _a.line) === lineTags['main getFromElsewhere call'] &&
                !state.elsewhere) {
                yield Promise.all([
                    dc.stepInRequest({ threadId, granularity: 'instruction' }),
                    dc.waitForEvent('stopped'),
                ]);
                state = yield getFrameState(threadId);
            }
            // First line we see should be inside `getFromElsewhere`
            expectStackState(state, {
                elsewhereDefined: true,
                line: lineTags['getFromElsewhere entry'],
            });
        }));
        it('steps next by line and skips a function', () => __awaiter(this, void 0, void 0, function* () {
            const threads = yield dc.threadsRequest();
            const threadId = threads.body.threads[0].id;
            expectStackState(yield getFrameState(threadId), {
                elsewhereDefined: false,
                line: lineTags['main getFromElsewhere call'],
            });
            yield Promise.all([
                dc.nextRequest({ threadId, granularity: 'statement' }),
                dc.waitForEvent('stopped'),
            ]);
            expectStackState(yield getFrameState(threadId), {
                elsewhereDefined: false,
                line: lineTags['main printf call'],
            });
            yield Promise.all([
                dc.nextRequest({ threadId, granularity: 'statement' }),
                dc.waitForEvent('stopped'),
            ]);
            expectStackState(yield getFrameState(threadId), {
                elsewhereDefined: false,
                line: lineTags['main for'],
            });
        }));
        it('steps next by instruction and skips a function', () => __awaiter(this, void 0, void 0, function* () {
            var _b;
            const threads = yield dc.threadsRequest();
            const threadId = threads.body.threads[0].id;
            let state = yield getFrameState(threadId);
            expectStackState(state, {
                elsewhereDefined: false,
                line: lineTags['main getFromElsewhere call'],
            });
            // Step until we get off line 'main getFromElsewhere call'.
            while (((_b = state.main) === null || _b === void 0 ? void 0 : _b.line) === lineTags['main getFromElsewhere call'] &&
                !state.elsewhere) {
                yield Promise.all([
                    dc.nextRequest({ threadId, granularity: 'instruction' }),
                    dc.waitForEvent('stopped'),
                ]);
                state = yield getFrameState(threadId);
            }
            // The first line we should see after 'main getFromElsewhere call'
            // is 'main printf call', not something in `getFromElsewhere`.
            expectStackState(state, {
                elsewhereDefined: false,
                line: lineTags['main printf call'],
            });
        }));
    });
});
//# sourceMappingURL=stepping-granularity.spec.js.map