/*********************************************************************
 * Copyright (c) 2018 QNX Software Systems and others
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *********************************************************************/
import { IGDBBackend } from '../types/gdb';
import { MIBreakpointInfo, MIResponse } from './base';
export interface MIBreakInsertResponse extends MIResponse {
    bkpt: MIBreakpointInfo;
    /**
     * In cases where GDB inserts multiple breakpoints, the "children"
     * breakpoints will be stored in multiple field.
     */
    multiple?: MIBreakpointInfo[];
}
export interface MIBreakDeleteRequest {
}
export interface MIBreakDeleteResponse extends MIResponse {
}
export interface MIBreakListResponse extends MIResponse {
    BreakpointTable: {
        nr_rows: string;
        nr_cols: string;
        hrd: Array<{
            width: string;
            alignment: string;
            col_name: string;
            colhdr: string;
        }>;
        body: MIBreakpointInfo[];
    };
}
export type MIBreakpointMode = 'hardware' | 'software';
export interface MIBreakpointInsertOptions {
    temporary?: boolean;
    /**
     * The `mode` property is prioritised over the `hardware` property.
     * If `mode` is defined, then the information in the `hardware` flag
     * is ignored during the insert breakpoint operation.
     *
     * The value of the mode wil be:
     *
     * - `'hardware'`: If user explicitly selects the breakpoint mode as
     *   'Hardware Breakpoint' at the user interface.
     * - `'software'`: If user explicitly selects the breakpoint mode as
     *    'Software Breakpoint' at the user interface.
     * - `undefined`: If user didn't make an explicitly breakpoint mode
     *   selection, in this case the `hardware` flag will be used.
     */
    mode?: MIBreakpointMode;
    /**
     * @deprecated The `hardware` property will be removed soon. Please
     * use the `mode` property instead of the `hardware`.
     */
    hardware?: boolean;
    pending?: boolean;
    disabled?: boolean;
    tracepoint?: boolean;
    condition?: string;
    ignoreCount?: number;
    threadId?: string;
}
export interface MIBreakpointLocation {
    locationType?: 'source' | 'function';
    source?: string;
    line?: string;
    fn?: string;
}
export declare function sourceBreakpointLocation(gdb: IGDBBackend, source: string, line?: string, forInsert?: boolean): string;
export declare function functionBreakpointLocation(gdb: IGDBBackend, fn: string, forInsert?: boolean): string;
export declare function sendBreakpointInsert(gdb: IGDBBackend, location: string, options?: MIBreakpointInsertOptions): Promise<MIBreakInsertResponse>;
export declare function sendBreakDelete(gdb: IGDBBackend, request: {
    breakpoints: string[];
}): Promise<MIBreakDeleteResponse>;
export declare function sendBreakList(gdb: IGDBBackend): Promise<MIBreakListResponse>;
export declare function sendFunctionBreakpointInsert(gdb: IGDBBackend, fn: string, options?: MIBreakpointInsertOptions): Promise<MIBreakInsertResponse>;
export declare function sendSourceBreakpointInsert(gdb: IGDBBackend, source: string, line?: string, options?: MIBreakpointInsertOptions): Promise<MIBreakInsertResponse>;
