// BEGIN FLOCK GPL
//
// Copyright Flock Inc. 2005-2009
// http://flock.com
//
// This file may be used under the terms of the
// GNU General Public License Version 2 or later (the "GPL"),
// http://www.gnu.org/licenses/gpl.html
//
// Software distributed under the License is distributed on an "AS IS" basis,
// WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
// for the specific language governing rights and limitations under the
// License.
//
// END FLOCK GPL
//

// the default object definitions for the Favorites datastore
// passed into this environment are Coop (the type) and coop (the instance)

const FLOCK_NS = 'http://flock.com/rdf#';
const NSCP_NS = 'http://home.netscape.com/NC-rdf#';
const RDF_NS = 'http://www.w3.org/1999/02/22-rdf-syntax-ns#';
const WEB_NS = 'http://home.netscape.com/WEB-rdf#';

const RDFS = Components.classes["@mozilla.org/rdf/rdf-service;1"]
             .getService(Components.interfaces.nsIRDFService);
//////////////////////////////////////////////////////////
// Common Stuff 
//////////////////////////////////////////////////////////

coop.Object = coop.add_type (FLOCK_NS+'Object', {
  fields: {
    name: NSCP_NS+'Name',
    flockType: FLOCK_NS+'flockType',
    rdftype: RDF_NS + 'type',
    isTransient: FLOCK_NS+'isTransient'
  },
  field_types: {
    rdftype: Coop.resource, 
    isTransient: Coop.boolean
  },    
  field_defaults: {
    rdftype: FLOCK_NS+'Object',
    name: ' '
  }
});

// A virtual class for common attributes - don't instantiate it!
coop.FlockObject = coop.add_type (FLOCK_NS+'FlockObject', {
  inherits_from: FLOCK_NS+'Object',
  fields: {
    favicon: FLOCK_NS+'favicon',
    isIndexable: FLOCK_NS+'isIndexable'
  },
  field_types: {
    isIndexable: Coop.boolean,
    rdftype: Coop.resource
  }
});

coop.CountedObject = coop.add_type(FLOCK_NS+'CountedObject', {
  inherits_from: FLOCK_NS+'FlockObject',
  fields: {
    unseenItems: FLOCK_NS+'unseenItems',
    count: FLOCK_NS+'count',
    maxItems: FLOCK_NS+'maxItems',
    capItems: FLOCK_NS+'capItems',
    hasUnseenItems: FLOCK_NS+'hasUnseenItems'
  },
  field_types: {
    unseenItems: Coop.stringinteger,
    count: Coop.stringinteger,
    maxItems: Coop.stringinteger,
    capItems: Coop.boolean,
    hasUnseenItems: Coop.boolean
  },
  field_defaults: {
    unseenItems: 0,
    count: 0,
    flockType: 'CountedObject',
    rdftype: FLOCK_NS+'CountedObject',
    hasUnseenItems: false
  }
});

coop.Folder = coop.add_type (FLOCK_NS+'Folder', {
  inherits_from: FLOCK_NS+'CountedObject',
  fields: {
    description: NSCP_NS+'Description',
    BookmarkAddDate: NSCP_NS+'BookmarkAddDate'
  },
  field_types: {
    BookmarkAddDate: Coop.date
  },
  field_defaults: {
    flockType: 'folder',
    rdftype: FLOCK_NS + "Folder"
  },
  container: 'Seq'
});

coop.UncountedFolder = coop.add_type (FLOCK_NS+'UncountedFolder', {
  inherits_from: FLOCK_NS+'FlockObject',
  field_defaults: {
    flockType: 'folder',
    rdftype: FLOCK_NS+'Folder'
  },
  container: 'Seq'
});

coop.TopFavesFolder = coop.add_type (FLOCK_NS+'TopFavesFolder', {
  inherits_from: FLOCK_NS+'Folder',
  field_defaults: {
    flockType: 'topfavesfolder',
    rdftype: FLOCK_NS+'TopFavesFolder'
  },
  container: 'Seq'
});

coop.Pollable = coop.add_type (FLOCK_NS+'Pollable', {
  inherits_from: FLOCK_NS+'CountedObject',
  fields: {
    isPollable: FLOCK_NS+'isPollable',
    isRefreshing: FLOCK_NS+'isRefreshing',
    nextRefresh: FLOCK_NS+'nextRefresh',
    refreshInterval: FLOCK_NS+'refreshInterval'
  },
  field_types: {
    isPollable: Coop.boolean,
    isRefreshing: Coop.boolean,
    nextRefresh: Coop.date,
    refreshInterval: Coop.integer
  },
  field_defaults: {
    rdftype: FLOCK_NS+'Pollable'
  }
});

coop.WebObject = coop.add_type (FLOCK_NS+'WebObject', {
  inherits_from: FLOCK_NS+'FlockObject',
  fields: {
      URL: NSCP_NS+'URL'
  },
  field_defaults: {
      URL: 'about:blank'
  }
});

coop.Account = coop.add_type (FLOCK_NS+'Account', {
  inherits_from: FLOCK_NS+'Pollable',
  fields: {
    URL: NSCP_NS+'URL',
    service: FLOCK_NS + "service",
    serviceId: FLOCK_NS+'serviceId',
    accountId: FLOCK_NS+'accountId',
    favicon: FLOCK_NS+'favicon',
    avatar: FLOCK_NS+'avatar',
    status: FLOCK_NS+'status',
    statusMessage: FLOCK_NS+'statusMessage',
    hasNew: FLOCK_NS+'hasNew',
    isAuthenticated: FLOCK_NS+'isAuthenticated',
    authToken: FLOCK_NS+'authToken',
    accountMessages: FLOCK_NS + "accountMessages",
    lastUpdateDate: FLOCK_NS + "lastUpdateDate",
    lastStatusMessageUpdateDate: FLOCK_NS + "lastStatusMessageUpdateDate",
    screenName: FLOCK_NS + "screenName",
    friendsList: FLOCK_NS + "friendsList",
    accountParamList: FLOCK_NS + "accountParamList",
    friendActivityLog: FLOCK_NS + "friendActivityLog",
    otherActivityLog: FLOCK_NS + "otherActivityLog"
  },
  field_types: {
    service: Coop.object,
    hasNew: Coop.boolean,
    isAuthenticated: Coop.boolean,
    lastUpdateDate: Coop.date,
    lastStatusMessageUpdateDate: Coop.integer,
    friendsList: Coop.object,
    accountParamList: Coop.object,
    friendActivityLog: Coop.object,
    otherActivityLog: Coop.object
  },
  field_defaults: {
    flockType: "Account",
    statusMessage: "",
    rdftype: FLOCK_NS +"Account",
    isAuthenticated: false,
    accountMessages: "0"
  },
  container: 'Seq'
});


coop.Defaults = coop.add_type(FLOCK_NS+'Defaults', {
  fields: {
  },
  container: 'Seq'
});

coop.AccountParam = coop.add_type(FLOCK_NS + "AccountParam", {
  fields: {
    key: FLOCK_NS + "key",
    value: FLOCK_NS + "value",
    type: FLOCK_NS + "type"
  }
});
coop.AccountParamList = coop.add_type(FLOCK_NS + "AccountParamList", {
  inherits_from: FLOCK_NS + "Object",
  fields: {
    account: FLOCK_NS + "account"
  },
  field_types: {
    account: Coop.object
  },
  field_defaults: {
    flockType: "AccountParamList",
    rdftype: FLOCK_NS + "AccountParamList"
  },
  container: "Bag"
});

coop.AccountParamList.prototype.getCustomParam =
function getCustomParam(aParamName) {
  var paramList = Components.classes["@mozilla.org/rdf/container;1"]
                 .createInstance(Components.interfaces.flockIRDFContainer);
  paramList.Init(coop.datasource, this.__Coop_resource);
  var keyRes = RDFS.GetResource(FLOCK_NS + "key")
  var keyLit = RDFS.GetLiteral(aParamName)
  var paramEnum = paramList.GetElements();
  while (paramEnum.hasMoreElements()) {
    var param = paramEnum.getNext();
    if (coop.datasource.HasAssertion(param, keyRes, keyLit, true)) {
      return coop.get_from_resource(param.QueryInterface(CI.nsIRDFResource));
    }
  }
  return null;
};

coop.AccountParamList.prototype.setCustomParam =
function setCustomParam(aParamName, aValue)
{
  var param = this.getCustomParam(aParamName);
  if (param) {
    param.value = aValue;
  } else {
    var param = new coop.AccountParam(
      {
        key: aParamName,
        value: aValue,
        type: typeof(aValue)
      });
    this.children.add(param);
  }
};
//////////////////////////////////////////////////////////
// Favorites
//////////////////////////////////////////////////////////

coop.Favorite = coop.add_type (FLOCK_NS+'Favorite', {
  inherits_from: FLOCK_NS+'WebObject',
  fields: {
    unseen: FLOCK_NS+'unseen',
    service: FLOCK_NS+'service',
    thumbnail: FLOCK_NS+'thumbnail',
    datevalue: FLOCK_NS+'datevalue',
    description: NSCP_NS+'Description',
    LastCharset: WEB_NS+'LastCharset'
  },
  field_types: {
    unseen: Coop.boolean,
    service: Coop.object,
    datevalue: Coop.date
  },
  field_defaults: {
    flockType: "bookmark",
    rdftype: FLOCK_NS+'Favorite',
    isIndexable: true
  },
  field_multiple: {
    service: true
  }
});

// ERW: tag (multiple) is duplicated from tags. I'm just trying this model with online bookmarks to make sure it works well, and if it appears to be good I will extend it to local bookmarks too. The goal is to have fast tag->bookmark queries.
coop.Bookmark = coop.add_type (FLOCK_NS+'Bookmark', {
  inherits_from: FLOCK_NS+'Favorite',
  fields: {
    private: FLOCK_NS+'private',
    tags: FLOCK_NS+'tags',
    tag:  FLOCK_NS+'tag', 
    ShortcutURL:  NSCP_NS+'ShortcutURL',
    BookmarkAddDate: NSCP_NS+'BookmarkAddDate',
    LastModifiedDate: WEB_NS +'LastModifiedDate',
    LastVisitDate: WEB_NS +'LastVisitDate'
  },
  field_types: {
    private: Coop.boolean,
    BookmarkAddDate: Coop.date,
    LastModifiedDate: Coop.date,
    LastVisitDate: Coop.date
  },
  field_multiple: {
    tag: true
  },
  field_defaults: {
    name: '',
    description: '',
    tags: ''
  }
});

coop.Bookmark.get_id = function Bookmark_get_id(obj) {
  return null;
}



coop.Bookmark.get_from_url = function get_from_url (aURL) {
  var ios = Components.classes["@mozilla.org/network/io-service;1"]
                      .getService(Components.interfaces.nsIIOService);
  var literal = RDFS.GetLiteral(ios.newURI(aURL, null, null).spec);
  var urlResource = RDFS.GetResource(NSCP_NS+"URL");
  var sources = this.ds.GetSources(urlResource, literal, true);
  urlResource = RDFS.GetResource(FLOCK_NS+"CoopType");
  literal = RDFS.GetLiteral(FLOCK_NS+"Bookmark");
  var obj = null;
  while (sources && sources.hasMoreElements()) {
    var source = sources.getNext()
                        .QueryInterface(Components.interfaces.nsIRDFResource);

    var ts = this.ds.GetTarget(source, urlResource, true );

    // if the id of source is NOT an anonymous node we have a
    // onlinebookmark, we need to keep looking
    if (ts.Value == FLOCK_NS+"Bookmark" &&
        source.Value.indexOf("rdf:") > -1)
    {
      obj = coop.get_from_resource(source);
      break;
    }
  }

  return obj
}
  
coop.Separator = coop.add_type (NSCP_NS+'BookmarkSeparator', {
  inherits_from: FLOCK_NS+'Object',
  field_defaults: {
    rdftype: NSCP_NS+'BookmarkSeparator'
  }
});


coop.Stream = coop.add_type (FLOCK_NS+'Stream', {
  inherits_from: FLOCK_NS+'Pollable',
  fields: {
    URL: NSCP_NS+'URL',
    apiType: FLOCK_NS+'apiType',
    notify: FLOCK_NS+'notify',
    userid: FLOCK_NS+'userid',
    serviceId: FLOCK_NS+'serviceId',
    lastModification: FLOCK_NS+'lastModification',
    state: FLOCK_NS+'state'
  },
  field_types: {
    notify: Coop.boolean
  },
  field_defaults: {
    URL: 'about:blank',
    notify: false,
    isIndexable: true,
    isPollable: true,
    flockType: 'stream',
    rdftype: NSCP_NS+'Livemark',
    state: 'ok'
  },
  container: 'Seq'
});

// Not used in Ibis and later, but we still need it for migration
// (to delete existing bookmark streams)
coop.OnlineBookmarksStream = coop.add_type (FLOCK_NS+'OnlineBookmarksStream', {
  inherits_from: FLOCK_NS+'Stream',
  fields: {
    last_update_time: FLOCK_NS+'LastUpdateTime', // last value from posts/update
    tag: FLOCK_NS+'tag'
  },
  field_types: {
    last_update_time: Coop.date
  },
  field_multiple: {
    tag: true
  },
  field_defaults: {
    last_update_time: new Date(0),
    rdftype: FLOCK_NS+'OnlineFolder'
  }
});

//////////////////////////////////////////////////////////
// Media
//////////////////////////////////////////////////////////

coop.Photo = coop.add_type (FLOCK_NS+'Photo', {
  inherits_from: FLOCK_NS+'Object',
  fields: {
    datevalue: FLOCK_NS+'datevalue',
    unseen: FLOCK_NS+'unseen',
    URL: NSCP_NS+'URL',
    midSizePhoto: FLOCK_NS+'midSizePhoto'
  },
  field_types: {
    datevalue: Coop.date,
    unseen: Coop.boolean
  },
  field_defaults: {
    datevalue: new Date(0),
    unseen: true
  }
});

coop.RichPhoto = coop.add_type (NSCP_NS+'RichPhoto', {
  inherits_from: FLOCK_NS+'Favorite',
  fields: {
    midSizePhoto: FLOCK_NS+'midSizePhoto',
    largeSizePhoto: FLOCK_NS+'largeSizePhoto',
    username: FLOCK_NS+'username',
    userid: FLOCK_NS+'userid',
    photoid: FLOCK_NS+'photoid',
    is_public: FLOCK_NS+'is_public',
    lastUpdate: FLOCK_NS+'lastUpdate',
    is_video: FLOCK_NS + "is_video",
    rating_count: FLOCK_NS + "rating_count"
  },
  field_types: {
    is_public: Coop.boolean,
    is_video: Coop.boolean,
    rating_count: Coop.integer,
    lastUpdate: Coop.date
  },
  field_defaults: {
    rdftype: NSCP_NS+'Favorite',
    is_video: false,
    isIndexable: false
  }
});


coop.MediaQuery = coop.add_type (NSCP_NS+'MediaQuery', {
  inherits_from: FLOCK_NS+'Pollable',
  fields: {
    service: FLOCK_NS+'svc',
    query: FLOCK_NS+'query',
    favicon: FLOCK_NS+'favicon',
    serviceId: FLOCK_NS+'serviceId',
    latestSeq: FLOCK_NS+'latestSeq',
    latestDate: FLOCK_NS+'latestDate',
    firstRefresh: FLOCK_NS+'firstRefresh'
  },
  field_types: {
    latestDate: Coop.date,
    firstRefresh: Coop.boolean
  },
  field_defaults: {
    flockType: 'media',
    rdftype: NSCP_NS+'MediaQuery',
    serviceId: '@flock.com/photo-api-manager;1?',
    firstRefresh: true
  },
  container: 'Seq'
});

coop.MediaQuery.prototype.markAllSeen = function ()
{
  // XXX - Put a batch around this somewhere
  var childEnum = this.children.enumerate();
  while (childEnum.hasMoreElements()) 
  {
    var child = childEnum.getNext();
    if (child) child.unseen = false;
  }
}

coop.MediaUploadItem = coop.add_type(FLOCK_NS + "MediaUploadItem", {
  inherits_from: FLOCK_NS + "Object",
  fields: {
    originalFileSpec: FLOCK_NS + "originalFileSpec",
    originalFilePath: FLOCK_NS + "originalFilePath",
    workingFilePath: FLOCK_NS + "workingFilePath",
    thumbFilePath: FLOCK_NS + "thumbFilePath",
    thumbFileSpec: FLOCK_NS + "thumbFileSpec",
    previewFilePath: FLOCK_NS + "previewFilePath",
    previewFileSpec: FLOCK_NS + "previewFileSpec",
    mimetype: FLOCK_NS + "mimetype",
    fileName: FLOCK_NS + "fileName",
    privacy_use_batch: FLOCK_NS + "privacy_use_batch",
    is_public: FLOCK_NS + "is_public",
    is_friend: FLOCK_NS + "is_friend",
    is_family: FLOCK_NS + "is_family",
    tags: FLOCK_NS + "tags",
    title: FLOCK_NS + "title",
    notes: FLOCK_NS + "notes",
    state: FLOCK_NS + "state",
    description: FLOCK_NS + "description",
    rotation: FLOCK_NS + "rotation",
    album: FLOCK_NS + "album",
    crop: FLOCK_NS + "crop"
  },
  field_defaults: {
    rdftype: FLOCK_NS + "MediaUploadItem"
  }
});

//////////////////////////////////////////////////////////
// Feeds
//////////////////////////////////////////////////////////

coop.Feed = coop.add_type (FLOCK_NS+'Feed', {
  inherits_from: FLOCK_NS+'Stream',
  fields: {
    link: FLOCK_NS+'link',
    format: FLOCK_NS+'format',
    type: FLOCK_NS+'type',
    subtitle: FLOCK_NS+'subtitle',
    image: FLOCK_NS+'image',
    author: FLOCK_NS+'author',
    datevalue: FLOCK_NS+'datevalue',
    finalURL: FLOCK_NS+'finalURL',
    realTitle: FLOCK_NS + "realTitle",
    lastFetch: FLOCK_NS+'lastFetch',
    lastFaviconFetch: FLOCK_NS+'lastFaviconFetch',
    metadataOnly: FLOCK_NS+'metadataOnly'
  },
  field_types: {
    datevalue: Coop.date,
    lastFetch: Coop.date,
    lastFaviconFetch: Coop.date,
    metadataOnly: Coop.boolean
  },
  field_defaults: {
    isIndexable: false,
    isPollable: false,
    flockType: 'feed',
    rdftype: NSCP_NS+'Livemark',
    capItems: true
  },
  container: 'Seq'
});

coop.Feed.get_id = function Feed_get_id(obj) {
  if (obj.URL)
    return 'urn:flock:feed:' + obj.URL;
  return null;
}

coop.FeedItem = coop.add_type (FLOCK_NS+'FeedItem', {
  inherits_from: FLOCK_NS+'Favorite',
  fields: {
    itemid: FLOCK_NS+'itemid',
    subtitle: FLOCK_NS+'subtitle',
    author: FLOCK_NS+'author',
    pubdate: FLOCK_NS+'pubdate',
    flagged: FLOCK_NS+'flagged'
  },
  field_types: {
    pubdate: Coop.date,
    flagged: Coop.boolean
  },
  field_defaults: {
    isIndexable: false,
    flockType: 'feeditem',
    rdftype: FLOCK_NS+'Favorite'
  }
});

coop.FeedFolder = coop.add_type (FLOCK_NS+'FeedFolder', {
  inherits_from: FLOCK_NS+'Folder',
  field_defaults: {
    flockType: 'folder',
    rdftype: FLOCK_NS+'Folder'
  },
  container: 'Seq'
});

coop.FeedContext = coop.add_type (FLOCK_NS+'FeedContext', {
  inherits_from: FLOCK_NS+'FeedFolder',
  fields: {
    refreshInterval: FLOCK_NS+'refreshInterval',
    flaggedItems: FLOCK_NS+'flaggedItems'
  },
  field_types: {
    refreshInterval: Coop.integer,
    flaggedItems: Coop.object
  },
  field_defaults: {
    flockType: 'folder',
    rdftype: FLOCK_NS+'Folder',
    refreshInterval: 3600
  },
  container: 'Seq'
});

coop.FeedContext.get_id = function FeedContext_get_id(obj) {
  if (obj.name)
    return 'urn:flock:feedcontext:' + obj.name;
  return null;
}

coop.FeedFlaggedStream = coop.add_type (FLOCK_NS+'FeedFlaggedStream', {
  inherits_from: FLOCK_NS+'Stream',
  field_defaults: {
    isIndexable: false,
    isPollable: false,
    rdftype: NSCP_NS+'Livemark'
  },
  container: 'Seq'
});

coop.FeedFlaggedStream.get_id = function FeedFlaggedStream_get_id(obj) {
  if (obj.context) {
    obj.name = obj.context.name;
    return obj.context.id() + ':flagged';
  }
  return null;
}

//////////////////////////////////////////////////////////
// Blog
//////////////////////////////////////////////////////////

// A blog that belongs to the current user, e.g. for
// publication, NOT for consumption. See "Feed" for consumption.
coop.Blog = coop.add_type (FLOCK_NS+'Blog', {
  inherits_from: FLOCK_NS+'WebObject',
  fields: {
    favicon: FLOCK_NS+'favicon',
    title: FLOCK_NS+'title',
    api: FLOCK_NS+'api',
    apiLink: FLOCK_NS+'apiLink',
    blogid: FLOCK_NS+'blogid',
    username: FLOCK_NS+'username',
    authtoken: FLOCK_NS+'authtoken',
    pwhost: FLOCK_NS+'pwhost',
    publishStories: FLOCK_NS + "publishStories",
    categories: FLOCK_NS+'categories'
  },
  field_types: {
    publishStories: Coop.boolean,
    categories: Coop.object
  },
  field_multiple: {
    categories: true
  },
  field_defaults: {
    rdftype: FLOCK_NS + "Blog"
  }
});

coop.BlogCategory = coop.add_type (FLOCK_NS+'BlogCategory', {
  inherits_from: FLOCK_NS+'Object',
  fields: {
    categoryId: FLOCK_NS+'categoryId'
  },
  field_defaults: {
    rdftype: FLOCK_NS+'BlogCategory'
  }
});

coop.Service = coop.add_type (FLOCK_NS+'Service', {
  inherits_from: FLOCK_NS+'Object',
  fields: {
    serviceId: FLOCK_NS+'serviceId',
    desc: FLOCK_NS+'Description',
    loginURL: FLOCK_NS+'LoginURL',
    logoutOption: FLOCK_NS+'LogoutOption',
    domains: FLOCK_NS+'Domains'
  },
  field_types: {
    logoutOption: Coop.boolean
  },
  field_defaults: {
    rdftype: FLOCK_NS+'Service',
    logoutOption: true
  },
  container: 'Seq'
});

coop.Toolbar = coop.add_type (FLOCK_NS+'Toolbar', {
  fields: { 
    folder: FLOCK_NS + "folder",
    rdftype: RDF_NS + 'type'
  },
  field_types: {
    folder: Coop.object,
    rdftype: Coop.resource
  },
  field_defaults: {
    rdftype: FLOCK_NS+'Toolbar'
  }
});

//////////////////////////////////////////////////////////
// Notes
//////////////////////////////////////////////////////////

coop.WebSnippet = coop.add_type (FLOCK_NS+'Snippet', {
  inherits_from: FLOCK_NS+'WebObject',
  fields: {
    content: FLOCK_NS+'Content',
    creationDate: FLOCK_NS+'date',
    tags: FLOCK_NS+'tags',
    snippetType: FLOCK_NS+'snippetType'
  },
  field_types: {
    creationDate: Coop.date
  },
  field_defaults: {
    snippetType: 'document',
    flockType: 'websnippet',
    rdftype: FLOCK_NS+'Snippet'
  }
});

//////////////////////////////////////////////////////////
// Twitter search
//////////////////////////////////////////////////////////

coop.TwitterSearch = coop.add_type (FLOCK_NS+"TwitterSearch", {
  inherits_from: FLOCK_NS+"Pollable",
  fields: {
    serviceId: FLOCK_NS+"serviceId",
    searchTerm: FLOCK_NS+"searchTerm"
  },
  field_defaults: {
    flockType: "TwitterSearch",
    rdftype: FLOCK_NS +"TwitterSearch",
    searchTerm: ""
  },
  container: "Seq"
});

coop.TwitterSearchItem = coop.add_type(FLOCK_NS+"TwitterSearchItem", {
  inherits_from: FLOCK_NS+"Object",
  fields: {
    itemId: FLOCK_NS+"itemId",
    itemText: FLOCK_NS+"itemText",
    fromUserId: FLOCK_NS+"fromUserId",
    fromUsername: FLOCK_NS+"fromUsername",
    toUserId: FLOCK_NS+"toUserId",
    toUsername: FLOCK_NS+"toUsername",
    avatar: FLOCK_NS+"avatar",
    createDate: FLOCK_NS+"createDate"
  },  
  field_types: {
    createDate: Coop.date
  },  
  field_defaults: {
    flockType: "TwitterSearchItem",
    rdftype: FLOCK_NS+"TwitterSearchItem"
  }
});

//////////////////////////////////////////////////////////
// Business logic
//////////////////////////////////////////////////////////


coop.favorites_root = coop.get(FLOCK_NS+'FavoritesRoot');
if (!coop.favorites_root) coop.favorites_root = new coop.Folder (FLOCK_NS+'FavoritesRoot', {name: "master root"});
coop.me_root = new coop.Folder(FLOCK_NS+'MeRoot');
coop.accounts_root = coop.get(FLOCK_NS+'AccountsRoot');
if (!coop.accounts_root) coop.accounts_root = new coop.Folder (FLOCK_NS+'AccountsRoot');
coop.accounts_root.name = 'Me';

coop.my_flock = new coop.Folder(FLOCK_NS+'MyFlock');
coop.my_flock.name = 'My Flock';

// coop.accounts_root.children.addOnce(coop.current_items_folder);

coop.favorites_root.children.addOnce(coop.accounts_root);
coop.favorites_root.children.addOnce(coop.my_flock);

coop.me_root.children.addOnce(coop.accounts_root);

// the bookmarks have to be contained with a folder for the UI to work 
// (it will show all of the bookmarks siblings so we are
//  creating a new folder instead of using coop.favorites_root)

coop.bookmarks_root = new coop.Folder (FLOCK_NS+'BookmarksRoot');
coop.bookmarks_root.name = 'Favorites';
coop.toolbar = new coop.Toolbar ('urn:flock:toolbar'); // ja - is this an orphan???
coop.folders_root = new coop.UncountedFolder(FLOCK_NS+"FoldersRoot", {name: "Bookmarks"});
coop.folders_root.children.addOnce(coop.bookmarks_root);
coop.favorites_root.children.addOnce(coop.folders_root);
var mediaFaves = new coop.Folder("urn:media:favorites");
coop.media_favorites = mediaFaves;
coop.favorites_root.children.addOnce(coop.media_favorites);

// Twitter search
coop.twitter_search_root = new coop.Folder(FLOCK_NS+"TwitterSearchRoot");
coop.twitter_search_root.name = "Twitter Search Root";
coop.favorites_root.children.addOnce(coop.twitter_search_root);
